package org.lsst.ccs.subsystem.power.ui;

import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DecimalFormat;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFormattedTextField;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.text.NumberFormatter;
import org.lsst.ccs.subsystem.power.config.Power;

/**
 ******************************************************************************
 **
 **  Power configuration editor.
 **
 **  @author  Owen Saxton
 **
 ******************************************************************************
 */
public class PowerConfigPanel extends JPanel {

    private final Power[] power;
    private final NumberFormatter fmt7_3 = new NumberFormatter(new DecimalFormat("##0.0##"));
    private final NumberFormatter fmt5 = new NumberFormatter(new DecimalFormat("####0"));
    private final int TF_WIDTH = 60;
    private final JLabel lbHostname = new JLabel("Hostname:");
    private final JTextField tfHostname = new JTextField();
    private final JLabel lbPort = new JLabel("Port:");
    private final JFormattedTextField tfPort = new JFormattedTextField(fmt5);
    private final JLabel[] lbChannel;
    private final JLabel lbVoltage = new JLabel("Voltage");
    private final JFormattedTextField[] tfVoltage;
    private final JLabel lbCurrent = new JLabel("Current");
    private final JFormattedTextField[] tfCurrent;
    private final JLabel lbOnDelay = new JLabel("On delay");
    private final JFormattedTextField[] tfOnDelay;
    private final JLabel lbOffDelay = new JLabel("Off delay");
    private final JFormattedTextField[] tfOffDelay;
    private static final long serialVersionUID = 1L;


   /**
    ***************************************************************************
    **
    **  Constructor.
    **
    **  @param  pwr  Array of Power objects to be manipulated
    **
    ***************************************************************************
    */
    public PowerConfigPanel(Power[] pwr) {
        power = pwr;
        fmt7_3.setValueClass(Double.class);
        fmt5.setValueClass(Integer.class);
        int nChan = 0;
        for (Power p : power) {
            nChan += p.getChannels().length;
        }
        setBorder(BorderFactory.createTitledBorder(""));

        /*
        **  Create text fields, etc, and set their properties
        */
        Dimension dh = tfHostname.getPreferredSize();
        dh.width = 120;
        tfHostname.setPreferredSize(dh);
        tfHostname.setMinimumSize(dh);
        Dimension dp = tfPort.getPreferredSize();
        dp.width = 50;
        tfPort.setPreferredSize(dp);
        tfPort.setMinimumSize(dp);

        lbChannel = new JLabel[nChan];
        tfVoltage = new JFormattedTextField[nChan];
        tfCurrent = new JFormattedTextField[nChan];
        tfOnDelay = new JFormattedTextField[nChan];
        tfOffDelay = new JFormattedTextField[nChan];

        Dimension d = (new JFormattedTextField()).getPreferredSize();
        d.width = 60;
        for (int j = 0; j < nChan; j++) {
            lbChannel[j] = new JLabel();

            tfVoltage[j] = new JFormattedTextField(fmt7_3);
            tfVoltage[j].setPreferredSize(d);
            tfVoltage[j].setMinimumSize(d);

            tfCurrent[j] = new JFormattedTextField(fmt7_3);
            tfCurrent[j].setPreferredSize(d);
            tfCurrent[j].setMinimumSize(d);

            tfOnDelay[j] = new JFormattedTextField(fmt7_3);
            tfOnDelay[j].setPreferredSize(d);
            tfOnDelay[j].setMinimumSize(d);

            tfOffDelay[j] = new JFormattedTextField(fmt7_3);
            tfOffDelay[j].setPreferredSize(d);
            tfOffDelay[j].setMinimumSize(d);
        }

        /*
        **  Add graphical elements to the panel
        */
        setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.insets = new Insets(6, 6, 6, 6);

        // Connection sub-panel
        JPanel conPanel = new JPanel();
        conPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbcC = new GridBagConstraints();
        gbcC.anchor = GridBagConstraints.WEST;
        gbcC.insets = new Insets(6, 4, 6, 4);
        gbcC.gridx = 0;
        gbcC.gridy = 0;
        conPanel.add(lbHostname, gbcC);
        gbcC.gridx++;
        conPanel.add(tfHostname, gbcC);
        gbcC.gridx++;
        conPanel.add(lbPort, gbcC);
        gbcC.gridx++;
        conPanel.add(tfPort, gbcC);
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.gridwidth = 5;
        add(conPanel, gbc);

        // Headings
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.gridwidth = 1;
        gbc.gridx = 1;
        gbc.gridy++;
        add(lbVoltage, gbc);
        gbc.gridx++;
        add(lbCurrent, gbc);
        gbc.gridx++;
        add(lbOnDelay, gbc);
        gbc.gridx++;
        add(lbOffDelay, gbc);

        // Channel data
        for (int j = 0; j < nChan; j++) {
            gbc.anchor = GridBagConstraints.EAST;
            gbc.gridx = 0;
            gbc.gridy++;
            add(lbChannel[j], gbc);
            gbc.anchor = GridBagConstraints.CENTER;
            gbc.gridx++;
            add(tfVoltage[j], gbc);
            gbc.gridx++;
            add(tfCurrent[j], gbc);
            gbc.gridx++;
            add(tfOnDelay[j], gbc);
            gbc.gridx++;
            add(tfOffDelay[j], gbc);
        }

        read();
    }


   /**
    ***********************************************************************
    **
    **  Saves new values to the power data.
    **
    ***********************************************************************
    */
    private void save() {

        int jc = 0;
        for (Power p : power ) {
            //power.setDevcId(tfHostname.getText());
            //power.setDevcParm((Integer)tfPort.getValue());
            for (Power.Channel chan : p.getChannels()) {
                chan.setVoltage((Double)tfVoltage[jc].getValue());
                chan.setCurrent((Double)tfCurrent[jc].getValue());
                chan.setOnDelay((Double)tfOnDelay[jc].getValue());
                chan.setOffDelay((Double)tfOffDelay[jc].getValue());
                jc++;
            }
        }
    }


   /**
    ***********************************************************************
    **
    **  Reads power data and populates graphical elements.
    **
    ***********************************************************************
    */
    private void read() {

        int jc = 0;
        for (Power p : power) {
            //tfHostname.setText(power.getDevcId());
            //tfPort.setValue(power.getDevcParm());
            for (Power.Channel chan : p.getChannels()) {
                lbChannel[jc].setText(chan.getName() + ":");
                tfVoltage[jc].setValue(chan.getVoltage());
                tfCurrent[jc].setValue(chan.getCurrent());
                tfOnDelay[jc].setValue(chan.getOnDelay());
                tfOffDelay[jc].setValue(chan.getOffDelay());
                jc++;
            }
        }
    }


   /**
    ***************************************************************************
    **
    **  Edits the power configuration.
    **
    **  @param  parent  The parent window
    **
    **  @param  power   The array of power objects to be manipulated
    **
    **  @return  Whether any values were changed
    **
    ***************************************************************************
    */
    public static boolean edit(Window parent, Power[] power)
    {
        final boolean[] changed = {false};
        final JDialog dlg = new JDialog(parent, "Power Configuration",
                                        Dialog.ModalityType.APPLICATION_MODAL);
        final PowerConfigPanel p = new PowerConfigPanel(power);
        dlg.setLayout(new GridBagLayout());
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = new Insets(10, 10, 10, 10);
        // dlg.add(p, BorderLayout.CENTER);
        dlg.add(p, gbc);

        JPanel btPanel = new JPanel();
        JButton btCancel = new JButton("Cancel");
        btCancel.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                dlg.setVisible(false);
                dlg.dispose();
            }
        });

        JButton btOK = new JButton("OK");
        btOK.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                changed[0] = true;
                p.save();
                dlg.setVisible(false);
                dlg.dispose();
            }
        });

        btPanel.setLayout(new GridBagLayout());
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.insets = new Insets(5, 5, 5, 5);
        gbc.gridx = 0;
        gbc.gridy = 0;
        btPanel.add(btCancel, gbc);
        gbc.gridx = 1;
        btPanel.add(btOK, gbc);

        // dlg.add(btPanel, BorderLayout.SOUTH);
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.anchor = GridBagConstraints.NORTH;
        dlg.add(btPanel, gbc);

        dlg.pack();
        dlg.setLocationRelativeTo(parent);
        dlg.setVisible(true);

        return changed[0];
    }

}
