package org.lsst.ccs.subsystem.power.constants;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.drivers.auxelex.Bfr;
import org.lsst.ccs.drivers.auxelex.Pdu24VC;
import org.lsst.ccs.drivers.auxelex.Pdu24VD;
import org.lsst.ccs.drivers.auxelex.Pdu48V;
import org.lsst.ccs.drivers.auxelex.Pdu5V;
import org.lsst.ccs.drivers.auxelex.RebBulkPS;

/**
 * Quad box switch definitions.
 *
 * @author The CCS tesm
 */
public class QuadBoxSwitches {

    public static final int
        DEVC_BFR      = 0,
        DEVC_PDU_5V   = 1,
        DEVC_PDU_24VC = 2,
        DEVC_PDU_24VD = 3,
        DEVC_PDU_48V  = 4,
        DEVC_REB_BULK = 5,
        DEVC_MAQ20    = 6,
        NUM_DEVICES   = 7;

    public static final int
        SW_MAQ20_REB_PS_HCU   = 0,
        SW_MAQ20_CRYOSTAT_HCU = 1,
        SW_MAQ20_FES_HCU      = 2,
        SW_MAQ20_SHUTTER_HCU  = 3,
        SW_MAQ20_FES_PLC      = 4,
        NUM_MAQ20_SWITCHES    = 5;

    public static final int
        SW_RELAY_HTR_1  = Bfr.RELAY_HTR_1 + 256 * DEVC_BFR,
        SW_RELAY_HTR_2  = Bfr.RELAY_HTR_2 + 256 * DEVC_BFR,
        SW_RELAY_PWS_2  = Bfr.RELAY_PWS_2 + 256 * DEVC_BFR,
        SW_RELAY_PWS_3  = Bfr.RELAY_PWS_3 + 256 * DEVC_BFR,
        SW_RELAY_PWS_4  = Bfr.RELAY_PWS_4 + 256 * DEVC_BFR,
        SW_RELAY_PWS_16 = Bfr.RELAY_PWS_16 + 256 * DEVC_BFR,
        SW_RELAY_PWS_17 = Bfr.RELAY_PWS_17 + 256 * DEVC_BFR,
        SW_RELAY_PWS_18 = Bfr.RELAY_PWS_18 + 256 * DEVC_BFR,
        SW_RELAY_PWS_19 = Bfr.RELAY_PWS_19 + 256 * DEVC_BFR,
        SW_RELAY_PWS_25 = Bfr.RELAY_PWS_25 + 256 * DEVC_BFR,
        SW_RELAY_PWS_20 = Bfr.RELAY_PWS_20 + 256 * DEVC_BFR,
        SW_RELAY_PWS_21 = Bfr.RELAY_PWS_21 + 256 * DEVC_BFR,
        SW_PDU_5V        = SW_RELAY_PWS_21,
        SW_PDU_24VC      = SW_RELAY_PWS_20,
        SW_PDU_24VD      = SW_RELAY_PWS_3,
        SW_PDU_48V       = SW_RELAY_PWS_2,
        SW_REB_BULK_PS_0 = SW_RELAY_PWS_16,
        SW_REB_BULK_PS_1 = SW_RELAY_PWS_17,
        SW_REB_BULK_PS_2 = SW_RELAY_PWS_18,
        SW_REB_BULK_PS_3 = SW_RELAY_PWS_19,
        SW_REB_BULK_PS_4 = SW_RELAY_PWS_25,
        SW_TRIM_HEATERS  = SW_RELAY_PWS_4,
        SW_HEATER_1      = SW_RELAY_HTR_1,  // Used by utility trunk system
        SW_HEATER_2      = SW_RELAY_HTR_2;  // Used by utility trunk system

    public static final int
        SW_OTM_0_A = Pdu5V.CHAN_OTM_0_A + 256 * DEVC_PDU_5V,
        SW_OTM_0_B = Pdu5V.CHAN_OTM_0_B + 256 * DEVC_PDU_5V,
        SW_OTM_1_A = Pdu5V.CHAN_OTM_1_A + 256 * DEVC_PDU_5V,
        SW_OTM_1_B = Pdu5V.CHAN_OTM_1_B + 256 * DEVC_PDU_5V,
        SW_OTM_2_A = Pdu5V.CHAN_OTM_2_A + 256 * DEVC_PDU_5V,
        SW_OTM_2_B = Pdu5V.CHAN_OTM_2_B + 256 * DEVC_PDU_5V,
        SW_OTM_3_A = Pdu5V.CHAN_OTM_3_A + 256 * DEVC_PDU_5V,
        SW_OTM_3_B = Pdu5V.CHAN_OTM_3_B + 256 * DEVC_PDU_5V,
        SW_OTM_4_A = Pdu5V.CHAN_OTM_4_A + 256 * DEVC_PDU_5V,
        SW_OTM_4_B = Pdu5V.CHAN_OTM_4_B + 256 * DEVC_PDU_5V,
        SW_OTM_5_A = Pdu5V.CHAN_OTM_5_A + 256 * DEVC_PDU_5V,
        SW_OTM_5_B = Pdu5V.CHAN_OTM_5_B + 256 * DEVC_PDU_5V;

    public static final int
        SW_MAIN_24C      = 0xff + 256 * DEVC_PDU_24VC,
        SW_ION_PUMPS     = Pdu24VC.CHAN_ION_PUMPS + 256 * DEVC_PDU_24VC,
        SW_UNUSED_24C    = Pdu24VC.CHAN_UNUSED + 256 * DEVC_PDU_24VC,
        SW_FES_CHANGER_C = Pdu24VC.CHAN_FES_CHANGER + 256 * DEVC_PDU_24VC,
        SW_CRYO_MAQ20    = Pdu24VC.CHAN_UT_MAQ20S + 256 * DEVC_PDU_24VC,
        SW_SHTR_PLC_1    = Pdu24VC.CHAN_SHTR_PLC_1 + 256 * DEVC_PDU_24VC,
        SW_SHTR_PLC_2    = Pdu24VC.CHAN_SHTR_PLC_2 + 256 * DEVC_PDU_24VC,
        SW_FES_CAROUSEL  = Pdu24VC.CHAN_FES_CAROUSEL + 256 * DEVC_PDU_24VC,
        SW_FES_LOADER_C  = Pdu24VC.CHAN_FES_LOADER + 256 * DEVC_PDU_24VC,
        SW_BODY_MAQ20    = Pdu24VC.CHAN_BODY_MAQ20 + 256 * DEVC_PDU_24VC,
        SW_GAUGES        = Pdu24VC.CHAN_GAUGES + 256 * DEVC_PDU_24VC;

    public static final int
        SW_MAIN_24D      = 0xff + 256 * DEVC_PDU_24VD,
        SW_FES_LOADER_D  = Pdu24VD.CHAN_FES_LOADER + 256 * DEVC_PDU_24VD,
        SW_FES_CHANGER_D = Pdu24VD.CHAN_FES_CHANGER + 256 * DEVC_PDU_24VD,
        SW_CRYO_TURBO    = Pdu24VD.CHAN_CRYO_TURBO + 256 * DEVC_PDU_24VD,
        SW_HEX_TURBO     = Pdu24VD.CHAN_HEX_TURBO + 256 * DEVC_PDU_24VD,
        SW_FES_CLAMPS    = Pdu24VD.CHAN_FES_CLAMPS + 256 * DEVC_PDU_24VD,
        SW_FES_BRAKES    = Pdu24VD.CHAN_FES_BRAKES + 256 * DEVC_PDU_24VD,
        SW_SHTR_BRAKES   = Pdu24VD.CHAN_SHTR_BRAKES + 256 * DEVC_PDU_24VD,
        SW_CRYO_SCROLL   = Pdu24VD.CHAN_CRYO_SCROLL + 256 * DEVC_PDU_24VD,  // Used by vacuum system
        SW_HEX_SCROLL    = Pdu24VD.CHAN_HEX_SCROLL + 256 * DEVC_PDU_24VD,   // Used by vacuum system
        SW_INST_SCROLL   = Pdu24VD.CHAN_INST_SCROLL + 256 * DEVC_PDU_24VD;  // Used by vacuum system

    public static final int
        SW_MAIN_48        = 0xff + 256 * DEVC_PDU_48V,
        SW_FES_CAROUSEL_D = Pdu48V.CHAN_FES_CAROUSEL + 256 * DEVC_PDU_48V,
        SW_SHTR_MOTOR_1   = Pdu48V.CHAN_SHTR_MOTOR_1 + 256 * DEVC_PDU_48V,
        SW_SHTR_MOTOR_2   = Pdu48V.CHAN_SHTR_MOTOR_2 + 256 * DEVC_PDU_48V,
        SW_PURGE_FAN      = Pdu48V.CHAN_PURGE_FAN + 256 * DEVC_PDU_48V,
        SW_FES_HEATER     = Pdu48V.CHAN_FES_HEATER  + 256 * DEVC_PDU_48V;

    public static final int
        SW_REB_BULK_0 = 0 + 256 * DEVC_REB_BULK,
        SW_REB_BULK_1 = 1 + 256 * DEVC_REB_BULK,
        SW_REB_BULK_2 = 2 + 256 * DEVC_REB_BULK,
        SW_REB_BULK_3 = 3 + 256 * DEVC_REB_BULK,
        SW_REB_BULK_4 = 4 + 256 * DEVC_REB_BULK;

    public static final int
        SW_REB_PS_HCU    = SW_MAQ20_REB_PS_HCU + 256 * DEVC_MAQ20,
        SW_CRYOSTAT_HCU  = SW_MAQ20_CRYOSTAT_HCU + 256 * DEVC_MAQ20,
        SW_FES_HCU       = SW_MAQ20_FES_HCU + 256 * DEVC_MAQ20,
        SW_SHUTTER_HCU   = SW_MAQ20_SHUTTER_HCU + 256 * DEVC_MAQ20,
        SW_FES_PLC       = SW_MAQ20_FES_PLC + 256 * DEVC_MAQ20;

    public static final int[] NUM_SWITCHES = new int[NUM_DEVICES];
    static {
        NUM_SWITCHES[DEVC_BFR] = Bfr.NUM_RELAYS;
        NUM_SWITCHES[DEVC_PDU_5V] = Pdu5V.NUM_CHANS;
        NUM_SWITCHES[DEVC_PDU_24VC] = Pdu24VC.NUM_CHANS;
        NUM_SWITCHES[DEVC_PDU_24VD] = Pdu24VD.NUM_CHANS;
        NUM_SWITCHES[DEVC_PDU_48V] = Pdu48V.NUM_CHANS;
        NUM_SWITCHES[DEVC_REB_BULK] = RebBulkPS.NUM_BULKS;
        NUM_SWITCHES[DEVC_MAQ20] = NUM_MAQ20_SWITCHES;
    }
    
    private static final Map<String, Integer> switchNameToId = new LinkedHashMap<>();
    static {
        switchNameToId.put("Clean5V", SW_RELAY_PWS_21);
        switchNameToId.put("Clean24V", SW_RELAY_PWS_20);
        switchNameToId.put("RebPs0-2", SW_RELAY_PWS_16);
        switchNameToId.put("RebPs3-5", SW_RELAY_PWS_17);
        switchNameToId.put("RebPs6-8", SW_RELAY_PWS_18);
        switchNameToId.put("RebPs9-12", SW_RELAY_PWS_19);
        switchNameToId.put("RebPsSpare", SW_RELAY_PWS_25);
        switchNameToId.put("Dirty24V", SW_RELAY_PWS_3);
        switchNameToId.put("Dirty28V", SW_RELAY_PWS_4);
        switchNameToId.put("Dirty48V", SW_RELAY_PWS_2);
        switchNameToId.put("Heater1", SW_RELAY_HTR_1);
        switchNameToId.put("Heater2", SW_RELAY_HTR_2);
        switchNameToId.put("Otm0-A", SW_OTM_0_A);
        switchNameToId.put("Otm0-B", SW_OTM_0_B);
        switchNameToId.put("Otm1-A", SW_OTM_1_A);
        switchNameToId.put("Otm1-B", SW_OTM_1_B);
        switchNameToId.put("Otm2-A", SW_OTM_2_A);
        switchNameToId.put("Otm2-B", SW_OTM_2_B);
        switchNameToId.put("Otm3-A", SW_OTM_3_A);
        switchNameToId.put("Otm3-B", SW_OTM_3_B);
        switchNameToId.put("Otm4-A", SW_OTM_4_A);
        switchNameToId.put("Otm4-B", SW_OTM_4_B);
        switchNameToId.put("Otm5-A", SW_OTM_5_A);
        switchNameToId.put("Otm5-B", SW_OTM_5_B);
        switchNameToId.put("Main24C", SW_MAIN_24C);
        switchNameToId.put("IonPumps", SW_ION_PUMPS);
        switchNameToId.put("BodyMaq20", SW_BODY_MAQ20);
        switchNameToId.put("CryoMaq20", SW_CRYO_MAQ20);
        switchNameToId.put("Gauges", SW_GAUGES);
        switchNameToId.put("FesCarousel", SW_FES_CAROUSEL);
        switchNameToId.put("FesChangerC", SW_FES_CHANGER_C);
        switchNameToId.put("FesLoaderC", SW_FES_LOADER_C);
        switchNameToId.put("ShtrPlc1", SW_SHTR_PLC_1);
        switchNameToId.put("ShtrPlc2", SW_SHTR_PLC_2);
        switchNameToId.put("Main24D", SW_MAIN_24D);
        switchNameToId.put("CryoTurbo", SW_CRYO_TURBO);
        switchNameToId.put("HexTurbo", SW_HEX_TURBO);
        switchNameToId.put("FesClamps", SW_FES_CLAMPS);
        switchNameToId.put("FesBrake", SW_FES_BRAKES);
        switchNameToId.put("FesChangerD", SW_FES_CHANGER_D);
        switchNameToId.put("FesLoaderD", SW_FES_LOADER_D);
        switchNameToId.put("ShtrBrakes", SW_SHTR_BRAKES);
        switchNameToId.put("CryoScroll", SW_CRYO_SCROLL);
        switchNameToId.put("HexScroll", SW_HEX_SCROLL);
        switchNameToId.put("InstScroll", SW_INST_SCROLL);
        switchNameToId.put("Main48", SW_MAIN_48);
        switchNameToId.put("PurgeFan", SW_PURGE_FAN);
        switchNameToId.put("FesCarouselD", SW_FES_CAROUSEL_D);
        switchNameToId.put("FesHeater", SW_FES_HEATER); 
        switchNameToId.put("ShtrMotor1", SW_SHTR_MOTOR_1);
        switchNameToId.put("ShtrMotor2", SW_SHTR_MOTOR_2);
        switchNameToId.put("RebBulk0-2", SW_REB_BULK_0);
        switchNameToId.put("RebBulk3-5", SW_REB_BULK_1);
        switchNameToId.put("RebBulk6-8", SW_REB_BULK_2);
        switchNameToId.put("RebBulk9-12", SW_REB_BULK_3);
        switchNameToId.put("RebBulkSpare", SW_REB_BULK_4);
        switchNameToId.put("RebPsHcu", SW_REB_PS_HCU);
        switchNameToId.put("CryostatHcu", SW_CRYOSTAT_HCU);
        switchNameToId.put("FesHcu", SW_FES_HCU);
        switchNameToId.put("ShutterHcu", SW_SHUTTER_HCU);
        switchNameToId.put("FesPlc", SW_FES_PLC);
    }

    private static final Map<Integer, String> switchIdToName = new LinkedHashMap<>();
    static {
        for (String name : switchNameToId.keySet()) {
            switchIdToName.put(switchNameToId.get(name), name);
        }
    }

    public static int getSwitchId(String name)
    {
        return switchNameToId.get(name);
    }

    public static String getSwitchName(int id)
    {
        return switchIdToName.get(id);
    }

    public static List<String> getSwitchNames()
    {
        return new ArrayList<>(QuadBoxSwitches.switchNameToId.keySet());
    }

}
