package org.lsst.ccs.subsystem.power.data;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.data.DataProviderInfo;

public enum PowerDataGroup {
    PS_REB_BOARD_POWER("Board Power", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/(Dig[VI]|Ana[VI]|Clk[HL][VI]|Power|OD[VI])")),
    PS_REB_DIGITAL_VOLTAGE("PS Reb Digital Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/digital/V(.*)")),
    PS_REB_DIGITAL_CURRENT("PS Reb Digital Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/digital/I(.*)")),
    PS_REB_OD_VOLTAGE("PS Reb OD Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/OD/V(.*)")),
    PS_REB_OD_CURRENT("PS Reb OD Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/OD/I(.*)")),
    PS_REB_CLOCK_HI_VOLTAGE("PS Reb Clock High Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/clockhi/V(.*)")),
    PS_REB_CLOCK_HI_CURRENT("PS Reb Clock High Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/clockhi/I(.*)")),
    PS_REB_CLOCK_LO_VOLTAGE("PS Reb Clock Low Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/clocklo/V(.*)")),
    PS_REB_CLOCK_LO_CURRENT("PS Reb Clock Low Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/clocklo/I(.*)")),
    PS_REB_HEATER_VOLTAGE("PS Reb Heater Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/heater/V(.*)")),
    PS_REB_HEATER_CURRENT("PS Reb Heater Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/heater/I(.*)")),
    PS_REB_HVBIAS_VOLTAGE("PS Reb HV Bias Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/hvbias/V(.*)")),
    PS_REB_HVBIAS_CURRENT("PS Reb HV Bias Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/hvbias/I(.*)")),
    PS_REB_ANALOG_VOLTAGE("PS Reb Analog Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/analog/V(.*)")),
    PS_REB_ANALOG_CURRENT("PS Reb Analog Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/analog/I(.*)")),
    PS_REB_DPHI_VOLTAGE("PS Reb DPhi Voltage", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/dphi/V(.*)")),
    PS_REB_DPHI_CURRENT("PS Reb DPhi Current", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>(Reb\\d|Reb[WG]))/dphi/I(.*)"));


    
    private final String name;
    private final Pattern pattern;
    
    static public PowerDataGroup forName(String name) {
        for (PowerDataGroup g : values()) {
            if (g.name.equals(name)) {
                return g;
            }
        }
        throw new IllegalArgumentException("No PowerDataGroup constant with "+ name +" name.");
    }

    private PowerDataGroup(String name, Pattern pattern) {
        this.name = name;
        this.pattern = pattern;
    }

    public String getName() {
        return name;
    }

    @Override
    public String toString() {
        return getName();
    }

    public boolean matchesData(DataProviderInfo dataInfo) {
        return pattern.matcher(dataInfo.getPath()).matches();
    }

    public static PowerDataGroup findPowerDataGroup(DataProviderInfo dataInfo) {
        PowerDataGroup fpdg = null;
        for (PowerDataGroup dataGroup : PowerDataGroup.values()) {
            if (dataGroup.matchesData(dataInfo)) {
                if (fpdg != null ) {
                    throw new RuntimeException("DataInfo "+dataInfo.getPath()+" matches more than one group: "+fpdg+" "+dataGroup);
                }
                fpdg = dataGroup;
            }
        }
        return fpdg;
    }

    public void addAttributesToDataInfo(DataProviderInfo dataInfo) {
        Matcher m = pattern.matcher(dataInfo.getPath());
        if ( m.matches() ) {
            dataInfo.addAttribute(DataProviderInfo.Attribute.DATA_GROUP, getName());
            try {
                String raftId = m.group("raftId");
                if (raftId != null) {
                    dataInfo.addAttribute(DataProviderInfo.Attribute.RAFT_ID, raftId);
                }
            } catch (Exception e) {
            }
        }        
    }    
}
