package org.lsst.ccs.subsystem.pathfinder;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.pathfinder.constants.CmprPosnState;
import org.lsst.ccs.subsystem.pathfinder.constants.CmprSetState;
import org.lsst.ccs.subsystem.pathfinder.constants.HexCmprState;
import org.lsst.ccs.subsystem.pathfinder.constants.HexValves;
import org.lsst.ccs.subsystem.pathfinder.data.PathfinderAgentProperties;
import org.lsst.ccs.subsystem.pathfinder.constants.ValvePosnState;
import org.lsst.ccs.subsystem.pathfinder.constants.ValveSetState;
import org.lsst.ccs.subsystem.pathfinder.data.HexValveState;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;

/**
 *  Implements the heat exchanger valve control subsystem.
 *
 *  @author Owen Saxton
 */
public class HexValveMain extends Subsystem implements HasLifecycle, ClearAlertHandler, AlertListener {

    /**
     *  Constants.
     */

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;

    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    ValveMaq20Device maqDevc;
    @LookupField(strategy=LookupField.Strategy.DESCENDANTS)
    private final Map<String, HexValve> valveMap = new HashMap<>();

    private static final Logger LOG = Logger.getLogger(HexValveMain.class.getName());
    private static final HexValve[] valveList = new HexValve[HexValves.NUM_VALVES];
    private final HexValveState valveState = new HexValveState();


    /**
     *  Constructor.
     */
    public HexValveMain() {
        super("hexvalve", AgentInfo.AgentType.WORKER);
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Create and schedule an AgentPeriodicTask to publish the refrigeration state when updated
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("valve-state", () -> publishUpdatedValveState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the heat exchanger valve subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a hex valve subsystem.
        propertiesService.setAgentProperty(PathfinderAgentProperties.HEX_VALVE_TYPE, HexValveMain.class.getCanonicalName());

        // Add alert listener
        alertService.addListener(this);

        // Initialize component references
        if (maqDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 device", "not defined");
        }
        for (String vName : HexValves.valveNameMap.keySet()) {
            if (!valveMap.containsKey(vName)) {
                ErrorUtils.reportConfigError(LOG, name, "valve " + vName, "not defined");
            }
            valveList[HexValves.valveNameMap.get(vName)] = valveMap.get(vName);
        }
        for (String vName : valveMap.keySet()) {
            if (!HexValves.valveNameMap.containsKey(vName)) {
                LOG.log(Level.WARNING, "Unrecognized valve name defined: {0} - will be removed", vName);
                valveMap.remove(vName);
            }
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.log(Level.INFO, "Heat exchanger valve subsystem started");
    }


    /**
     *  Gets the state of the hex valve system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The hex valve state
     */
    @Command(type=CommandType.QUERY, description="Get the hex valve state", level=0)
    public HexValveState getSystemState()
    {
        synchronized (valveState) {
            return valveState;
        }
    }    


    /**
     *  Gets a list of the valid heat exchanger names.
     *
     *  @return  The list of names
     */
    @Command(type=CommandType.QUERY, description="Get the valid heat exchanger names", level=0)
    public List<String> getHexNames()
    {
        return new ArrayList(HexValves.hxNameMap.keySet());
    }    


    /**
     *  Gets a list of the valid compressor names.
     *
     *  @return  The list of names
     */
    @Command(type=CommandType.QUERY, description="Get the valid compressor names", level=0)
    public List<String> getCmprNames()
    {
        return new ArrayList(HexValves.cmprNameMap.keySet());
    }    


    /**
     *  Connect a heat exchanger to a cryo compressor.
     *
     *  @param  hex  The heat exchanger name
     *  @param  comp  The compressor name
     *  @throws  VacuumException
     */
    @Command(type=CommandType.ACTION, description="Connect hex to compressor")
    public void connectHex(@Argument(description="Heat exchanger name") String hex,
                           @Argument(description="Compressor name") String comp) throws VacuumException
    {
        Integer hexNum = HexValves.hxNameMap.get(hex);
        if (hexNum == null) {
            throw new VacuumException("Invalid heat exchanger name: " + hex);
        }
        Integer cmprNum = HexValves.cmprNameMap.get(comp);
        if (cmprNum == null) {
            throw new VacuumException("Invalid compressor name: " + comp);
        }
        if (!HexValves.hxCmprSets[hexNum].contains(cmprNum)) {
            throw new VacuumException("Heat exchanger " + hex + " cannot be connected to compressor " + comp);
        }
        for (int cNum : HexValves.hxCmprSets[hexNum]) {
            for (int valve : HexValves.cmprValveSets[cNum]) {
	    //		if (!(valveList[valve].getName().contains("PVC6R") || valveList[valve].getName().contains("PVC1R"))) {
		    if (cNum == cmprNum) {
			valveList[valve].setOpen();
		    }
		    else {
			valveList[valve].setShut(); 
		    }
	    /*
		}
		else {
		    if (cNum == cmprNum) {
			valveList[valve].setShut();
		    }
		    else {
			valveList[valve].setOpen(); 
		    }
		}
	    */
            }
        }
        maqDevc.setValves();
    }


    /**
     *  set the state of a single valve
     *
     *  @param  valve number
     *  @param  true for close. false for open
     *  @throws  VacuumException
     */
    @Command(type=CommandType.ACTION,  level=10, description="set the state of a single valve")
    public void setValve(@Argument(description="valve number") int valve,
                    @Argument(description="false=open or true=close") boolean vstate) throws VacuumException
    {
	if (!vstate) {
	    valveList[valve].setOpen();
	}
	else {
	    valveList[valve].setShut(); 
	}
        maqDevc.setValves();
    }


    /**
     *  Sets the monitor update period.
     *
     *  @param  value  The period (milliseconds) to set.
     */
    @Command(type=CommandType.ACTION, description="Set the tick interval")
    public void setUpdatePeriod(@Argument(description = "Tick interval (ms)") int value)
    {
        synchronized (valveState) {
            setTickPeriod(value);
            publishState();
        }
    }


    /**
     * Callback to clear an {@code Alert} instance.
     * 
     * @param alert The Alert instance to clear.
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     * 
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        String id = alert.getAlertId();
        return ClearAlertCode.UNKNOWN_ALERT;
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
        }
    }


    /**
     *  Publishes the updated hex valve state.
     *
     *  This is called regularly as a timer task.
     */
    private void publishUpdatedValveState()
    {
        synchronized (valveState) {
            boolean changed = false;
            maqDevc.getSetValves();
            maqDevc.readPositions();
            for (int vNum = 0; vNum < HexValves.NUM_VALVES; vNum++) {
                HexValve valve = valveList[vNum];
                ValveSetState vsState = valve.getSetState();
                if (vsState != valveState.getSetState(vNum)) {
                    valveState.setSetState(vNum, vsState);
                    changed = true;
                }
                ValvePosnState vpState = valve.getPosnState();
                if (vpState != valveState.getPosnState(vNum)) {
                    valveState.setPosnState(vNum, vpState);
                    changed = true;
                }
            }
            if (changed) {
                for (int hNum = 0; hNum < HexValves.NUM_HEXS; hNum++) {
                    int csStates = 0, cpStates = 0;
                    for (int cNum : HexValves.hxCmprSets[hNum]) {
                        int vsStates = 0, vpStates = 0;
                        for (int vNum : HexValves.cmprValveSets[cNum]) {
                            vsStates |= 1 << valveState.getSetState(vNum).ordinal();
                            vpStates |= 1 << valveState.getPosnState(vNum).ordinal();
                        }
                        CmprSetState cSetState = (vsStates & (1 << ValveSetState.OFFLINE.ordinal())) != 0 ? CmprSetState.OFFLINE :
                                                 (vsStates & (1 << ValveSetState.OPEN.ordinal())) != 0 ? 
                                                   (vsStates & (1 << ValveSetState.SHUT.ordinal())) != 0 ?
                                                     CmprSetState.MIXED : CmprSetState.OPEN : CmprSetState.SHUT;
                        CmprPosnState cPosnState = (vpStates & (1 << ValvePosnState.OFFLINE.ordinal())) != 0 ? CmprPosnState.OFFLINE :
                                                   (vpStates & (1 << ValvePosnState.BOTH.ordinal())) != 0 ? CmprPosnState.BOTH :
                                                   (vpStates & (1 << ValvePosnState.NONE.ordinal())) != 0 ? CmprPosnState.NONE :
                                                   (vpStates & (1 << ValvePosnState.OPEN.ordinal())) != 0 ? 
                                                     (vpStates & (1 << ValvePosnState.SHUT.ordinal())) != 0 ?
                                                       CmprPosnState.MIXED : CmprPosnState.OPEN : CmprPosnState.SHUT;
                        valveState.setCmprState(cNum, cSetState);
                        csStates |= 1 << cSetState.ordinal();
                        cpStates |= 1 << cPosnState.ordinal();
                    }
                    HexCmprState hexState = (csStates & (1 << CmprSetState.OFFLINE.ordinal())) != 0 
                                              || (cpStates & (1 << CmprPosnState.OFFLINE.ordinal())) != 0 ? HexCmprState.OFFLINE :
                                            (csStates & (1 << CmprSetState.MIXED.ordinal())) != 0 ? HexCmprState.SETERROR :
                                            (cpStates & (1 << CmprPosnState.MIXED.ordinal())) != 0
                                              || (cpStates & (1 << CmprPosnState.BOTH.ordinal())) != 0
                                              || (cpStates & (1 << CmprPosnState.NONE.ordinal())) != 0 ?
                                              HexCmprState.POSNERROR : HexCmprState.NORMAL;
                    valveState.setHexState(hNum, hexState);
                }
                publishState();
            }
        }
    }


    /**
     *  Publishes the state of the hex valve module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        valveState.setTickMillis(getTickPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(HexValveState.KEY, valveState));
    }    


    /**
     *  Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period)
    {
        periodicTaskService.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }
    

    /**
     *  Gets the monitoring publishing period.
     */
    private int getTickPeriod()
    {
        return (int)periodicTaskService.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }



    /**
     *  Show the state of the compressor valves.
     *
     *  @return  compressor valve states
     *  @throws  VacuumException
     */
    @Command(type=CommandType.ACTION, description="report state of valves")
    public String showValveStates() throws VacuumException
    {
	String valveStateString = "";
	boolean changed = false;
	maqDevc.getSetValves();
	maqDevc.readPositions();
	for (int vNum = 0; vNum < HexValves.NUM_VALVES; vNum++) {
	    HexValve valve = valveList[vNum];
                ValveSetState vsState = valve.getSetState();
                ValvePosnState vpState = valve.getPosnState();

		String singleValveState = "Valve# " + vNum + " valve = " + valve.getName() + " vsState = " + vsState + " vpState = " + vpState ;
		LOG.log(Level.INFO, singleValveState);
		valveStateString += singleValveState + "\n";
	}

	return(valveStateString);
    }
}
