package org.lsst.ccs.subsystem.pathfinder;

import java.util.HashSet;
import java.util.Set;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20AnalogControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;

/**
 *  Controls Pathfinder Maq20 device (operates heaters)
 * 
 *  @author saxton/homer
 */
public class Maq20DevicePF extends Maq20Device {


    @LookupField (strategy=LookupField.Strategy.CHILDREN)
    private Maq20AnalogControl maqCtrl;

    private static final Logger LOG = Logger.getLogger(Maq20DevicePF.class.getName());
    private static final int NUM_HEATERS = 2;
    private static final int WATER_VALVE_CHANNEL = 1;
    private int ctrlModIndex, currModIndex = -1;
    private long[] setTime = new long[NUM_HEATERS];


    public static final int
        CHAN_COLD_HEATER_SET   = 0,
        CHAN_CRYO_HEATER_SET   = 2,
        CHAN_OTHER1_HEATER_SET   = 1,
        CHAN_OTHER2_HEATER_SET   = 3,
        CHAN_OTHER3_HEATER_SET   = 4,
        CHAN_OTHER4_HEATER_SET   = 5,
        CHAN_COLD_HEATER_READ  = 11,
        CHAN_CRYO_HEATER_READ  = 12;

    private static final int[] heaterSetChans = new int[NUM_HEATERS];
    static {
        heaterSetChans[0] = CHAN_COLD_HEATER_SET;
        heaterSetChans[1] = CHAN_CRYO_HEATER_SET;
    }
    private static final double[] heaterMaxLim = new double[NUM_HEATERS];
    static {
        heaterMaxLim[0] = 20.0;
        heaterMaxLim[1] = 4.0 + 16.0/3.0;   // cope with 1200W heater where a 400W heater is expected
    }

    private final int minChan=0, maxChan=1;
    private final Map<Integer, ChannelState> states = new HashMap<>();
    public static class ChannelState {

        private boolean
            enabled,       // Output has been enabled
            currError;     // Error setting the current
	//            noLoad;        // Unable to determine the load resistance

    }


    /**
     *  Basic initialization.
     */
    @Override
    public void initDevice()
    {
        super.initDevice();
        if (maqCtrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "analog control", "not defined");
        }
        ctrlModIndex = maqCtrl.getModIndex();
        for (int ix = 0; ix < getModuleCount(); ix++) {
            ModuleData modData = getModuleData(ix);
            if (modData.modDef.type == Maq20.ModuleType.IS) {
                currModIndex = ix;
                break;
            }
        }
	for (int chan = minChan; chan <= maxChan; chan++) {
            states.put(chan, new ChannelState());
        }
	
	// more intelligence TBA
	for (int chan : states.keySet()) {
	    ChannelState state = states.get(chan);
	    state.enabled = true;
	    state.currError = false;
	}
    }


    /**
     *  Enables the output.
     *
     *  @param  chan    The hardware channel
     *  @param  enable  The enabled state to set
     */
    public void enableOutput(int chan, boolean enable)
    {
        ChannelState state = getChannelState(chan);
	state.enabled = enable;
	if (!enable) {
	    setHeaterPower(chan, 0.0);
	    state.currError = false;
	}
    }
    
    
    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  Channel value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        int index = type & 0xff;

        return value;
    }


    /**
     *  Sets a heater level
     *
     *  @param  heaterId  The heater ID
     *  @param  fractionPower (0.0-1.0) 
     *  @throws  VacuumException
     */
    public void setHeaterPower(int heaterId, double fractionPower)
    {
	if (fractionPower<0.0 || fractionPower>1.0) {
	    LOG.log(Level.SEVERE,"Disallowed heater power fraction requested " + heaterId
				      + " (heater# " + heaterId + ") fractionPower: " + fractionPower);
	} else {
	    try {
		setHeaterLevel(heaterId, 4.0 + fractionPower*(heaterMaxLim[heaterId] - 4.0));
	    } catch (VacuumException e) {
		logError("Error setting power output for", heaterId, e);
	    }
	}
 
    }
 
    /**
     *  Get heater control current
     *
     *  @return  The heater power fraction
     *  @throws  VacuumException
     */
    public double getHeaterPower(int heaterId) throws VacuumException
    {
	return 1000.0*(Math.max(getHeaterLevel(heaterId),0.004)-0.004)/(heaterMaxLim[heaterId] - 4.0);

    }

   /**
     *  Sets a heater level
     *
     *  @param  heaterId  The heater ID
     *  @param  level (4-20) 
     *  @throws  VacuumException
     */
    public double setHeaterLevel(int heaterId, double level) throws VacuumException
    {
        int chan = heaterSetChans[heaterId];
	level /= 1000.; 
        double value = Math.min(Math.max(level, 0.000), heaterMaxLim[heaterId]/1000.0);
	LOG.info("For heaterId " + heaterId + "(chan = "+chan+ ") the value being sent is " + value + " input level = " + level);

        try {
            maqCtrl.writeValue(chan, value);
            setTime[heaterId] = System.currentTimeMillis();
        }
        catch (DriverException e) {
            throw new VacuumException("Error setting heater " + heaterId
				       + " (channel " + chan + ") level: " + e);
        }
	return(value);
    }

    /**
     *  open/close water valve
     *
     *  @param  openValve (true: open, false: close)
     *  @throws  VacuumException
     */
    public void setWaterValveOpen(boolean openValve) throws VacuumException
    {
        try {
	    double curr = openValve ? 0.004 : 0.020;
	    LOG.info("Setting maq20 chan " + WATER_VALVE_CHANNEL + " to I = " + curr);
            maqCtrl.writeValue(WATER_VALVE_CHANNEL, curr);
        }
        catch (DriverException e) {
            throw new VacuumException("Error actuating water valve: " + e);
	}
    }


    /**
     *  Get the control current of the valve
     *
     *  @return  The valve control current
     *  @throws  VacuumException
     */
    public double getWaterValveControlCurrent() throws VacuumException
    {
        try {
            double level = maqCtrl.readValue(WATER_VALVE_CHANNEL);
            return level;
        }
        catch (DriverException e) {
            throw new VacuumException(e);
        }
    }

    /**
     *  Get the heater control current
     *
     *  @return  The heater control current
     *  @throws  VacuumException
     */
    public double getHeaterLevel(int heaterId) throws VacuumException
    {
        int chan = heaterSetChans[heaterId];
        try {
            double level = maqCtrl.readValue(chan);
            return level;
        }
        catch (DriverException e) {
            throw new VacuumException("Error getting heater " + heaterId
				       + " (channel " + chan + ") level: " + e);
        }
    }


    /**
     *  Gets the set time for a heater.
     *
     *  Need to know this because it can take several seconds to reach its set point
     * 
     *  @param  heaterId  The heater ID
     *  @return  The set time (ms)
     */
    public long getHeaterSetTime(int heaterId)
    {
        return setTime[heaterId];
    }

     /**
     *  Gets the state object for a channel.
     *
     *  @param  chan   The hardware channel
     *  @return  The channel state
     */
    private ChannelState getChannelState(int chan)
    {
        return states.get(chan);
    }

    /**
     *  Log an error message for a channel.
     *
     *  @param  msg   The message text
     *  @param  chan  The hardware channel
     *  @param  excp  The exception, or null if none
     */
    private void logError(String msg, int chan, VacuumException e)
    {
        LOG.log(Level.SEVERE, "{0} {1} channel {2}{3}", new Object[]{msg, name, chan, e == null ? "" : ": " + e});
    }

    
}
