package org.lsst.ccs.subsystem.pathfinder;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.pathfinder.constants.UtilTrunkFans;
import org.lsst.ccs.subsystem.pathfinder.constants.FanState;
//import org.lsst.ccs.subsystem.pathfinder.data.UtilTrunkState;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
//import org.lsst.ccs.subsystem.common.actions.PathfinderSharedVacState;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
//import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.pathfinder.alerts.PathfinderAlerts;
import org.lsst.ccs.subsystem.pathfinder.constants.PLCState;
import org.lsst.ccs.subsystem.pathfinder.constants.ConditionState;
import org.lsst.ccs.subsystem.pathfinder.constants.LatchState;
import org.lsst.ccs.subsystem.pathfinder.constants.DeviceState;
//import org.lsst.ccs.subsystem.vacuum.constants.ImageChannel;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchEnable;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchState;
import org.lsst.ccs.subsystem.pathfinder.data.PathfinderAgentProperties;
import org.lsst.ccs.subsystem.pathfinder.constants.VacuumState;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;
import org.lsst.ccs.subsystem.pathfinder.data.VacSysState;
import org.lsst.ccs.subsystem.pathfinder.constants.TrimHeaterOpState;
import org.lsst.ccs.subsystem.pathfinder.constants.TrimHeaterState;

//import org.lsst.ccs.utilities.logging.Logger;
import java.util.logging.Logger;


//import org.lsst.ccs.common.devices.power;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorr84Device;
import org.lsst.ccs.subsystem.common.devices.refrigeration.AVCCryoDevice;

/**
 * The PathfinderSubsystem is designed specifically for running ComCam
 * thermal and vacuum control,
 */
public class PathfinderSubsystem extends Subsystem implements HasLifecycle, AlertListener {

    // Instances of the foreline pressure appearing below when there
    // is no foreline pressure gauge intended for ComCamare for placeholders
    // for a future similar check
    
    private static final double PRESS_ATMOS = 759.0,
            PRESS_TURBO_LOW = 5.0,
            PRESS_FORELINE_LOW = 5.0,
            PRESS_DIFF_LOW = 0.09,
            PRESS_DIFF_HIGH = 20.0,
	    PRESS_ION_OFF = 1.0e-5,
	    PRESS_ION_ENABLE = 4.0e-6,
            PRESS_VACUUM = 1.0e-7,
            COMCAM_PRESS_HIGH = 20.0,
            TURBO_MAX = 81000,
            TURBO_LOW = 0.1 * TURBO_MAX,
            TURBO_HIGH = 0.5 * TURBO_MAX;

    private static final int
	    MAX_PRESS_ERRORS = 2,
            MAX_SPEED_ERRORS = 0,
            DELAY_ION_OFF = 20000;


    /**
     *  Inner class for holding trim heater parameters.
     */
    static class TrimHeater {

        TrimHeaterControl tempCtrl;
        PathfinderPowerDevice powerDevc;
        int powerChan;
        boolean inited = false;

        TrimHeater(TrimHeaterControl tempCtrl, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerChan = powerChan;
        }
        TrimHeater(TrimHeaterControl tempCtrl, PathfinderPowerDevice powerDevc, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }


    
    private static final int[] switchChannels = new int[VacSysState.NUM_SWITCHES];
    private final FanPIControl[] fanControl = new FanPIControl[UtilTrunkFans.NUM_FANS];
    //    private final VacSysState utilTrunkState = new VacSysState();
    private MonitorTaskControl monitorControl;

    static {
        switchChannels[VacSysState.SW_HX_ION_PUMP1] = IonPumpDevice.CHAN_CIP2;
        switchChannels[VacSysState.SW_OR_ION_PUMP] = IonPumpDevice.CHAN_OIP;
        switchChannels[VacSysState.SW_HX_VALVE] = VacPlutoDevice.SW_OPEN_VHX00;

    }
    private static final Map<TwisTorr84.PumpStatus, DeviceState> turboStateMap = new HashMap<>();

    static {
        turboStateMap.put(TwisTorr84.PumpStatus.STOP, DeviceState.STOPPED);
        turboStateMap.put(TwisTorr84.PumpStatus.WAIT_INTLK, DeviceState.WAITING);
        turboStateMap.put(TwisTorr84.PumpStatus.STARTING, DeviceState.STARTNG);
        turboStateMap.put(TwisTorr84.PumpStatus.NORMAL, DeviceState.NORMAL);
        turboStateMap.put(TwisTorr84.PumpStatus.BRAKING, DeviceState.BRAKING);
        turboStateMap.put(TwisTorr84.PumpStatus.FAIL, DeviceState.FAILED);
        turboStateMap.put(TwisTorr84.PumpStatus.AUTO_TUNING, DeviceState.AUTOTUN);
    }

    private static final Map<Integer, PathfinderAlerts> alertMap = new HashMap<>();
    static {
        alertMap.put(VacSysState.LATCH_HX_VACUUM, PathfinderAlerts.HX_VACUUM_BAD);
        alertMap.put(VacSysState.LATCH_HX_GATE_NFC, PathfinderAlerts.HX_GATE_FORCED_SHUT);
        alertMap.put(VacSysState.LATCH_HX_GATE_AO, PathfinderAlerts.HX_GATE_CANNOT_OPEN);
        alertMap.put(VacSysState.LATCH_HX_PUMP, PathfinderAlerts.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revAlertMap = new HashMap<>();
    static {
        for (int cond : alertMap.keySet()) {
            revAlertMap.put(alertMap.get(cond).getId(), cond);
        }
    }


    @LookupName
    private String name;

//    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.pathfinder");
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService ass;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, TrimHeaterControl> tempControlMap = new HashMap<>();


    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private VacPlutoDevice plutoDevc = null;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private CryoTurboDevice cryoTurboDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private IonPumpDevice ionPumpDevc;
//    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
//    private AVCCryoDevice cryotelDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    Map<String, AVCCryoDevice> avcMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Maq20DevicePF pfControl;

    //    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    //    private PathfinderPowerDevice pfPowerDevc;


    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new LinkedHashMap<>();


    private FanPIControl utFan;
    private Channel utFanSpeed;
    private final Channel[] fanSpeedChans = new Channel[UtilTrunkFans.NUM_FANS];


    // From Groovy file
    //    private String pressChan1, pressChan2, turboSpeedChan, forelinePressChan, CryoTempChan,
    private String pressChan1, turboSpeedChan, forelinePressChan, turboPressChan, CryoTempChan,
            Cold1TempChan, Cold2TempChan, IonPumpVoltageChan;
    private List<Integer> switches;

    
    // General

    private static final Logger LOG = Logger.getLogger(PathfinderSubsystem.class.getName());

    private final VacSysState vacState = new VacSysState();
//    private final PathfinderSharedVacState pathfinderSharedVacState = new PathfinderSharedVacState();
    private Set<Integer> switchSet;
    private final Device[] switchDevices = new Device[VacSysState.NUM_SWITCHES];
    private Channel turboPressure, HXPressure, turboSpeed, forelinePressure, CryoTemp, Cold1Temp, Cold2Temp;
    private boolean running = true;
    private final Map<String, Integer> switchNameMap = new LinkedHashMap<>();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private long ionOverStartTime = 0;

    private boolean plcActive = true;

    private volatile boolean alertResponseEnabled = true;

    private boolean monAlertEnabledVATShut = false;

//    private final Map<String, ChanParams> monChans = new HashMap<>();
//    private final ClientFactory clientFactory;
//    private PathfinderCommands vacuumCommands;
    private String last_alertStr;

//    private final GlobalProc globalProc = new GlobalProc();
    private boolean useInfoAlerts = System.getProperty("org.lsst.ccs.subsystem.pathfinderPowerInfoAlerts", "false").contains("true");

    static class SensorData {
        Channel channel;
        int maxErrors;
        int numErrors = 0;
        double value = 0.0;
        double goodValue = Double.NaN;

        private SensorData(Channel channel, int maxErrors) {
            this.channel = channel;
            this.maxErrors = maxErrors;
        }
    }
    private SensorData pfMainPr, turboMainPr, forelineMainPr;

    private final TrimHeater[] trimHeaters = new TrimHeater[2];


    
    /**
     * Main constructor for the PathfinderSubsystem, normally invoked from the
     * corresponding groovy file.
     *
     */

    public PathfinderSubsystem() {
        super("pathfinder", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

    @Override
    public void init() {
        //Register the alerts raised by the subsystem
        alertService.registerAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE.getAlert());        
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            PathfinderAlerts alert = alertMap.get(cond);
            if ( alert != null ) {
                alertService.registerAlert(alert.getAlert());
            }
        }
    }

    @Override
    public void postInit() {

	// Add alert listener
        alertService.addListener(this);

//        vacuumCommands = 
        //Set a property to define that this Agent is a vacuum subsystem.
        agentPropertiesService.setAgentProperty(PathfinderAgentProperties.PATHFINDER_TYPE_AGENT_PROPERTY, PathfinderSubsystem.class.getCanonicalName());


	// Initialize trim heater control
        Set<Integer> heaters = new HashSet<>();
        for (TrimHeaterControl ctrl : tempControlMap.values()) {
	    int htr = ctrl.getPowerChannel();
	    //            trimHeaters[htr] = new TrimHeater(ctrl, ctrl.getPowerDevice(), htr);
            trimHeaters[htr] = new TrimHeater(ctrl,htr);
	}

        if (switches != null) {
            switchSet = new HashSet<>(switches);
        } else {
            ErrorUtils.reportConfigError(LOG, name, "Switch list", "not specified");
        }

        // Check fan controllers
        if (utFan == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFan", "has not been specified");
        }
        fanControl[UtilTrunkFans.FAN_UT_ID] = utFan;
        // Check fan speed readback channels
        if (utFanSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "utFanSpeed", "has not been specified");
        }
        fanSpeedChans[UtilTrunkFans.FAN_UT_ID] = utFanSpeed;


	//hh
        if (plutoDevc != null) {
            for (int cond : plutoDevc.getLatchIds()) {
                vacState.addLatch(cond);
                vacState.setLatch(cond, LatchState.CLEAR);
		LOG.info("Cleared latch cond  "+cond);
            }
             for (int cond : plutoDevc.getConditionIds()) {
                vacState.addCondition(cond);
                vacState.setCondition(cond, ConditionState.NO);
            }
        } else {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }

        if (cryoTurboDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "PF turbo pump device", "not specified");
        }
        /*
        if (hxTurboDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "HX turbo pump device", "not specified");
        }
         */
        if (ionPumpDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "Ion pump device", "not specified");
        }
        if (forelinePressChan != null) {
            forelinePressure = channelMap.get(forelinePressChan);
        }
        forelineMainPr = new SensorData(forelinePressure, MAX_PRESS_ERRORS);
	
        if (turboPressChan != null) {
            turboPressure = channelMap.get(turboPressChan);
        }
        if (turboPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "Turbo pressure channel", "not specified or not defined");
        }
        turboMainPr = new SensorData(turboPressure, MAX_PRESS_ERRORS);
	
        if (pressChan1 != null) {
            HXPressure = channelMap.get(pressChan1);
        }
        if (HXPressure == null) {
            ErrorUtils.reportConfigError(LOG, name, "PF pressure channel", "not specified or not defined");
        }
        pfMainPr = new SensorData(HXPressure, MAX_PRESS_ERRORS);

        if (turboSpeedChan != null) {
            turboSpeed = channelMap.get(turboSpeedChan);
        }
        if (turboSpeed == null) {
            ErrorUtils.reportConfigError(LOG, name, "Turbo pump speed channel", "not specified or not defined");
        }

        if (pfControl == null) {
            ErrorUtils.reportConfigError(LOG, name, "Maq20DevicePF component", "has not been defined");
        }

	/*
        if (pfPowerDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "PF power device component", "has not been defined");
        } else {
	    pfPowerDevc(pfControl);
	}
	*/

        List<Integer> ipChannels = ionPumpDevc.getChannelNumbers();
	LOG.info("ipChannel Names = " + ionPumpDevc.getChannelNames());
	LOG.info("ipChannel Numbers = " + ionPumpDevc.getChannelNumbers());
        Iterator swIter = switchSet.iterator();
        while (swIter.hasNext()) {
            int sw = (Integer) swIter.next();
            switch (sw) {
                case VacSysState.SW_HX_TURBO:
                    switchDevices[sw] = cryoTurboDevc;
                    break;
		case VacSysState.SW_HX_VALVE:
		    //                case VacSysState.SW_OR_FPP_VALVE:
		    //                case VacSysState.SW_OR_FH_VALVE:
		    //                case VacSysState.SW_OR_L3H_VALVE:
		    //                case VacSysState.SW_OR_L3_VALVE:
		    switchDevices[sw] = plutoDevc;
		    //                    switchDevices[sw] = null;
                    break;
		case VacSysState.SW_HX_ION_PUMP1:
		//                case VacSysState.SW_OR_ION_PUMP:
                    switchDevices[sw] = ionPumpDevc;
                    if (!ipChannels.contains(switchChannels[sw])) {
                        swIter.remove();
                    }
                    break;
		    //                case VacSysState.SW_KOOLANCE:
		    //                    switchDevices[sw] = pdu20Devc;
		    //                    break;
//                case VacSysState.SW_KOOLCOLD:
//                    switchDevices[sw] = pdu15Devc;
//                    break;
//                case VacSysState.SW_CRYOTEL_PWR:
//                    switchDevices[sw] = pdu20Devc;
//                    break;
            }
        }

        for (int sw : switchSet) {
            vacState.addSwitch(sw);
            vacState.setSwitchState(sw, SwitchState.OFFLINE);
            vacState.setSwitchEnable(sw, SwitchEnable.OFF);
        }

        for (String swName : SwitchNames.NAME_MAP.keySet()) {
            int sw = SwitchNames.NAME_MAP.get(swName);
            if (switchSet.contains(sw)) {
                switchNameMap.put(swName, sw);
            }
        }

	//        GPVacMon835Device vqm = new GPVacMon835Device();


//        subsys.addCommandsFromObject(vqm, "");
        LOG.info("Configured the status aggregator to listen to subsystems: " + System.getProperty("org.lsst.ccs.subsystem.teststand", "ts") + " " + System.getProperty("org.lsst.ccs.subsystem.rebps", "ccs-rebps") + " " + System.getProperty("org.lsst.ccs.subsystem.pathfinder", "pathfinder"));
    }

    @Override
    public void build() {
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state",
                () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        apts.scheduleAgentPeriodicTask(pt);

        ass.registerState(VacuumState.class, "ComCam Vacuum state", this);
        ass.updateAgentState(VacuumState.UNKNOWN);

        for (PathfinderAlerts alert : PathfinderAlerts.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }

    @Override
    public void postStart() {
        LOG.info("Pathfinder subsystem started");
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        if (secs <= 0) {
            return;
        }
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            LOG.log(Level.SEVERE,"Rude awakening!",ex);
        }
    }

    /**
     * Gets the state of the Vacuum system.
     *
     * @return The vacuum state
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the vacuum system state")
    public VacSysState getVacuumState() {
        vacState.setTickMillis(getTickPeriod());
        return vacState;
    }

    /**
     * Gets the shared state of the Vacuum system.
     *
     * @return The vacuum state
     */
/*
    @Command(type = Command.CommandType.QUERY, description = "Get the vacuum system state")
    public PathfinderSharedVacState getPathfinderSharedVacState() {
        pathfinderSharedVacState.setTickMillis(getTickPeriod());
        return pathfinderSharedVacState;
    }
*/
    /**
     * Gets the list of switch names.
     *
     * @return The switch names.
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.QUERY, description = "Get switch names")
    public List<String> getSwitchNames() throws VacuumException {
        return new ArrayList(switchNameMap.keySet());
    }

    /**
     * Turns a switch on or off.
     *
     * @param sw The switch number.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Turn on/off a switch")
    public void setSwitchOn(@Argument(description = "The switch number") int sw,
            @Argument(description = "Whether to turn on") boolean on) throws VacuumException {
        try {
            if (!switchSet.contains(sw)) {
                throw new VacuumException("Invalid switch number: " + sw);
            }
            setSwitch(sw, on);
        } finally {
            publishState();
        }

    }

    /**
     * Turns a named switch on or off.
     *
     * @param name The switch name.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description = "The switch name") String name,
            @Argument(description = "Whether to turn on") boolean on) throws VacuumException {
        Integer sw = switchNameMap.get(name);
        try {
            if (sw == null) {
                throw new VacuumException("Invalid switch name: " + name);
            }
            setSwitch(sw, on);
        } finally {
            publishState();
        }
    }

    /**
     * Turns a switch on or off.
     *
     * @param sw The switch number.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException {
        SwitchState state = vacState.getSwitchState(sw);
        LOG.info("setSwitch called for sw = " + sw);
        if (state == SwitchState.OFFLINE) {
            LOG.info("Switch sw = " + sw + " is offline");
//	    if (!(sw == vacState.SW_CRYO_VALVE))
//		return;
//	    else
//		LOG.info("proceeding exceptionally for the VAT valve switch which will directly set the state as a request"); // HN

        }
        
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        // setting switch on is rejected here if enable not on
        if (on && enable != SwitchEnable.ON && enable != SwitchEnable.WAS_ON) {
            LOG.info("enable = " + enable + " SwitchEnable.ON = " + SwitchEnable.ON + " SwitchEnable.WAS_ON = " + SwitchEnable.WAS_ON);
            return;
        }
        LOG.info("proceeding with setting switch sw = " + sw);
        Device swDevice = switchDevices[sw];
        try {
            if (swDevice instanceof TwisTorr84Device) {
                if (on) {
                    ((TwisTorr84Device) swDevice).startTurboPump();
                } else {
                    ((TwisTorr84Device) swDevice).stopTurboPump();
                }
		//            } else if (swDevice == plutoDevc) {
		//                plutoDevc.setSwitchOn(switchChannels[sw], on);
            } else if (swDevice == ionPumpDevc) {
                ionPumpDevc.setChannelOn(switchChannels[sw], on);
	    } else if (swDevice == plutoDevc) {
                LOG.info("sending request to PLC(plutoDevc) to set sw=" + sw + " to " + on);
		plutoDevc.setSwitchOn(switchChannels[sw], on);
            } else {
                LOG.info("swDevice not identified for sw=" + sw);
            }
        } catch (DriverException e) {
            throw new VacuumException(e);
        }
    }

    /**
     * Gets whether a switch is on.
     *
     * @param sw The switch number.
     * @return Whether the switch is on
     * @throws VacuumException
     */
    private Boolean isSwitchOn(int sw) {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice instanceof TwisTorr84Device) {
            try {
                DeviceState st = turboStateMap.get(((TwisTorr84Device) swDevice).readTurboStatus());
                value = st != DeviceState.STOPPED && st != DeviceState.BRAKING;
            } catch (DriverException e) {
                value = null;
            }

        } else if (swDevice == plutoDevc) {
	    if (plutoDevc!=null) 
		value = plutoDevc.isSwitchOn(switchChannels[sw]);

        } else if (swDevice == ionPumpDevc) {
            value = ionPumpDevc.isChannelOn(switchChannels[sw]);

        }
	/* else if (sw == VacSysState.SW_HX_VALVE) {

	    value = null;  // HN
	    if (vacState.getSwitchState(sw) != SwitchState.OFFLINE) // HN
		value = vacState.getSwitchState(sw) == SwitchState.ON;  // HN

		}*/
        return value;
    }
    /*
    /**
     * Clears a condition.
     *
     * @param cond The condition number.
     * @throws VacuumException
     * /
    @Command(type = Command.CommandType.ACTION, description = "Clear a condition")
    public void clearCondition(@Argument(description = "The condition number") int cond) throws VacuumException {
        try {
            plutoDevc.clearCondition(cond);
        } finally {
            publishState();
        }
    }
*/

    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            plutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }

    /**
     *  Reads a sensor, allowing for possible temporary errors.
     * 
     *  @param  data  The sensor data object
     */
    private void readSensor(SensorData data) {
        data.value = data.channel.getValue();
        if (!Double.isNaN(data.value)) {
            data.goodValue = data.value;
            data.numErrors = 0;
        }
        else if (data.numErrors++ < data.maxErrors) {
            data.value = data.goodValue;
        }
    }
    

    /**
     * Sets the update period.
     *
     * @param value The update period (milliseconds) to set.
     */
    @Command(type = Command.CommandType.ACTION, description = "Set the update interval")
    public void setUpdatePeriod(@Argument(description = "The tick period (ms)") int value) {
        setTickPeriod(value);
        vacState.setTickMillis(getTickPeriod());
        publishState();
    }


    /**
     *  Sets a heater level
     *
     *  @param  heaterId  The heater ID
     *  @param level The heater level in mA (4-20)
     *  @return message indicating actual heater level set
     *  @throws  VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Sets the heater level")
    public String setHeater(@Argument(description = "heater ID") int heaterId, @Argument(description = "heater level in mA (4-20)") double level) throws VacuumException
    {
        return("Heater level set to "+pfControl.setHeaterLevel(heaterId, level));
    }

    /**
     *  Sets a heater level using the fraction of max allowed power
     *
     *  @param  heaterId  The heater ID
     *  @param level The heater power fraction (0.0 to 1.0)
     *  @return message indicating actual heater level set
     *  @throws  VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Sets the heater power level fraction")
    public String setHeaterPowerFraction(@Argument(description = "heater ID") int heaterId, @Argument(description = "heater level fraction (0.0 to 1.0)") double powerFraction) throws VacuumException
    {
        pfControl.setHeaterPower(heaterId, powerFraction);
        return("Heater level control current set to "+getHeaterLevel(heaterId));
    }


    /**
     *  Sets the state of the water valve
     *
     *  @param  heaterId  The heater ID
     *  @param openValve (true/false)
     *  @throws  VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Sets the water valve open or close")
    public void setWaterValveOpen(@Argument(description = "true(open)/false(close)") boolean openValve) throws VacuumException
    {
        pfControl.setWaterValveOpen(openValve);
    }



    /**
     *  Gets a heater power fraction of max allowed
     *
     *  @param  heaterId  The heater ID
     *  @return level The heater power fraction of max allowed (0.0 to 1.0)
     *  @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Gets the heater level")
    public double getHeaterPowerFraction(@Argument(description = "heater ID") int heaterId) throws VacuumException
    {
        return pfControl.getHeaterPower(heaterId);
    }


    /**
     *  Gets a heater level
     *
     *  @param  heaterId  The heater ID
     *  @return level The heater level (4-20)
     *  @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Gets the heater level")
    public double getHeaterLevel(@Argument(description = "heater ID") int heaterId) throws VacuumException
    {
        return pfControl.getHeaterLevel(heaterId);
    }



    /**
     *  Sets a trim heater operation state.
     *
     *  @param  htr    The heater number
     *  @param  state  The heater operation state to set: OFF, POWER or TEMP.
     *  @return message indicating success of state setting
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a trim heater operation state")
    public String setTrimHeaterState(int htr, TrimHeaterOpState state) throws VacuumException
    {
	TrimHeater heater = getTrimHeater(htr);
	String msg = "";
	if (state == TrimHeaterOpState.OFF || ((state == TrimHeaterOpState.TEMP || state == TrimHeaterOpState.POWER) && heater.inited)) {
	    heater.tempCtrl.setOpState(state);
	    msg = "state set";
	} else {
	    msg = "heater needs to be inited to set this state";
	}
	return(msg);
    }


    /**
     *  Sets a target pathfinder temperature value.
     *
     *  @param  htr   The heater number
     *  @param  temp  The plate temperature set point.
     *  @throws  VacuumException
     */

    @Command(type=Command.CommandType.ACTION, description="Set a pathfinder temperature set point")
    public void setTemperature(int htr, double temp) throws VacuumException
    {
	TrimHeater heater = getTrimHeater(htr);

	heater.tempCtrl.setTemp(temp);
	heater.inited = true;

    }

    /**
     *  Sets a pathfinder heater power fraction value.
     *
     *  @param  htr   The heater number
     *  @param  current  The heater power fraction
     *  @throws  VacuumException
     */

    @Command(type=Command.CommandType.ACTION, description="Set a pathfinder heater power fraction")
    public void setPower(int htr, double power) throws VacuumException
    {
	TrimHeater heater = getTrimHeater(htr);

	heater.tempCtrl.setPower(power);
	heater.inited = true;

    }




    /**
     *  Get the water valve control current
     *
     *  @return VTC control current (4-20)
     *  @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Gets the VTC control current")
    public double getWaterValveControlCurrent() throws VacuumException
    {
        return pfControl.getWaterValveControlCurrent();
    }


    /**
     * Updates the vacuum system state periodically.
     *
     * The vacuum state consists mainly of the state of the switches (lines)
     * being controlled, along with whether they can be turned on.
     *
     * Pressures and pump speeds are read to determine which switches are
     * enabled and/or need to be turned off. If the state changes it is
     * published.
     */
    private void updateVacuumState() {
        LOG.info("updateVacuumState called, running = " + running);


	if (!running) return;


	for (int fan = 0; fan < UtilTrunkFans.NUM_FANS; fan++) {
	    FanState oldState = vacState.getFanState(fan);
	    FanState state = oldState;

	    //	    if (utControl.isOnline()) {
		if (oldState == FanState.OFFLINE) {
		    Double speed = fanControl[fan].getSpeed();
		    //		    LOG.info("fanControl getSpeed returned v="+speed);
		    vacState.setFanSpeed(fan, speed == null ? 0.0 : Math.rint(1000.0 * speed) / 1000.0);
		    state = speed == null ? FanState.OFFLINE : speed > 0.0 ? FanState.SPEED : FanState.OFF;
		    vacState.setFanBaseState(fan, state);
		}
		/*
	    }
	    else {
		state = FanState.OFFLINE;
	    }
		*/
	    if (state != oldState) {
		vacState.setFanState(fan, state);
		//		changed = true;
	    }
	}



        Boolean plcActive = plutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline");
                vacState.setPlcState(PLCState.OFFLINE);
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died");
                vacState.setPlcState(PLCState.DEAD);
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive");
                vacState.setPlcState(PLCState.ALIVE);
            }
        }


	/*
        double forelinePr = 0.0;
        if (this.forelinePressChan != null) {
            forelinePr = forelinePressure.readValue();
        }
	*/
	readSensor(forelineMainPr);
	double forelinePr = forelineMainPr.value;
	LOG.info("forelinePressure = " + forelinePr);

	//	double turboPr = turboPressure.readValue();
	readSensor(turboMainPr);
	double turboPr = turboMainPr.value;
	LOG.info("turboPr = " + turboPr);

	//	double pfPr = HXPressure.readValue();
	readSensor(pfMainPr);
	double pfPr = pfMainPr.value;
	LOG.info("current pfPr = " + pfPr);

        double turboSp = turboSpeed.readValue();
        boolean changed = false;
	boolean forcedVATclosing = false;

        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if (!vacState.hasSwitch(sw)) {
//                LOG.info("sw = "+sw+" is not a switch");
                continue;
            }
//            try {
//                LOG.info("Updating status for switch "+sw);
            //+" name = "+this.getSwitchNames().get(sw));
//            } catch (VacuumException ex) {
//                LOG.error(ex);
//            }

            Boolean enable = false;
            boolean turnOff = false;
            DeviceState devState = null;
            switch (sw) {

                case VacSysState.SW_HX_VALVE:
		    //		    LOG.info("SW_HX_VALVE");
		    /*                    
                    // When foreline pressure is available, insure that the pressure difference is
                    // low enogh to enable opening valve with difference limits depending on the turbo running or not.
                    // For ComCam, pfPr == pfPr and the following has no effect.
                    if (!Double.isNaN(cryoPr) && !Double.isNaN(pfPr) && !Double.isNaN(turboSp)) {
                        double prDiff = Math.abs(pfPr - forelinePr);
                        enable = (prDiff <= PRESS_DIFF_HIGH && turboSp < TURBO_LOW)
                                || (prDiff <= PRESS_DIFF_LOW && turboSp > TURBO_HIGH);
                    }
                    */
		    enable = true;
		    //		    enable = vacState.getSwitchEnable(sw) == SwitchEnable.ON;

                    // to be anded with conditions that either the foreline press be low or the turbo speed be low
		    //                    boolean enable2 = !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)
//                            && (forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW);
		    boolean enable2 = false;
		    boolean enable_set = false; // tells whether a condition was able to be set
                    if (!Double.isNaN(forelinePr)) {

			enable2 |= forelinePr < PRESS_FORELINE_LOW;
			enable_set = true;
		    }

                    if (!Double.isNaN(turboSp)) {
			enable2 |= turboSp < TURBO_LOW;
			enable_set = true;
		    }


		    enable &= enable2;
		    if (enable_set) {
			turnOff = !enable2;
		    }
                    
                    // since ComCam doesn't have a foreline pressure reading, impose the following condition as well
                    // do not close valve just based on a bad reading (added for ComCam)
                    if (!Double.isNaN(pfPr)) {
			boolean enable3 = pfPr > COMCAM_PRESS_HIGH || turboSp > TURBO_HIGH;
			enable &= enable3;
                    
                    }

		    forcedVATclosing = turnOff;

		    //		    LOG.info("VAT - enable = " + enable + " enable2 = " + enable2 + " enable3 = " + enable3); 

                    break;

                case VacSysState.SW_HX_TURBO:
		    //		    LOG.info("SW_HX_TURBO");
                    LOG.fine("pfPr = " + pfPr + " forelinePr = " + forelinePr + " turboSp = " + turboSp);
		    /*
                    if (!Double.isNaN(pfPr) && !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                        enable = pfPr < PRESS_TURBO_LOW && (forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW);
                        LOG.fine("turbo enable = " + enable + " pfPr = " + pfPr + " PRESS_TURBO_LOW =" + PRESS_TURBO_LOW);
			}*/
                    if (!Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                        enable = forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW;
                        LOG.fine("turbo enable = " + enable + " pfPr = " + pfPr + " PRESS_TURBO_LOW =" + PRESS_TURBO_LOW);
                    }
                    // do not turn off just based on a single bad reading
                    if (!Double.isNaN(forelinePr)) {
                        turnOff = !enable;
                    }
                    try {
                        devState = turboStateMap.get(((CryoTurboDevice) switchDevices[sw]).readTurboStatus());
                    } catch (DriverException e) {
                    }
                    break;

                case VacSysState.SW_HX_ION_PUMP1:
                    if (!Double.isNaN(pfPr)) {
                        enable = pfPr < PRESS_ION_ENABLE;
//                        turnOff = pfPr >= PRESS_ION_OFF;
                        if (pfPr >= PRESS_ION_OFF) {
                            if (ionOverStartTime == 0) {
                                ionOverStartTime = System.currentTimeMillis();
                            } else {
                                turnOff = System.currentTimeMillis() - ionOverStartTime >= DELAY_ION_OFF;
                            }
                        } else {
                            ionOverStartTime = 0;
                        }
                    } else {
			//                        enable = vacState.getSwitchEnable(sw) == SwitchEnable.ON;
			turnOff = true;
                    }

                    break;
	        default:
		    LOG.info("no handler for sw = "+sw);

            }

            SwitchState oldState = vacState.getSwitchState(sw);
            if (turnOff && oldState == SwitchState.ON) {
                try {
                    setSwitch(sw, false);
                } catch (VacuumException e) {
                    LOG.log(Level.SEVERE,"Error setting switch: ", e);
                }
            }
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
            if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
                LOG.info("Enabling switch sw=" + sw + " enabled = " + enabled);
                vacState.setSwitchState(sw, state);
                vacState.setSwitchEnable(sw, enabled);
                changed = true;
            }
            if (devState != vacState.getDeviceState(sw)) {
                vacState.setDeviceState(sw, devState);
                changed = true;
            }
        }
	/*
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if (!vacState.hasCondition(cond)) {
                continue;
            }
	    Boolean active = false;
	    Boolean latched = false;
	    //hh
	    if (plutoDevc != null) {
		active = plutoDevc.isConditionActive(cond);
		latched = plutoDevc.isConditionLatched(cond);
	    }

            ConditionState state = active == null || latched == null ? ConditionState.OFFLINE
                    : latched ? ConditionState.LATCHED
                            : active ? ConditionState.ACTIVE : ConditionState.CLEAR;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
	*/
	/* ---- added 2020/1/28 ---- */
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if (!vacState.hasLatch(cond)) continue;
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);

	    if (cond == VacSysState.LATCH_CR_GATE_NFC || cond == VacSysState.LATCH_CR_GATE_AO) {
		active = forcedVATclosing;
	    }

            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
		latched ? LatchState.LATCHED :
		active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond);
            if (state != oldState) {
                vacState.setLatch(cond, state);
                PathfinderAlerts alert = alertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Vacuum PLC error condition set");
		}
		else if (state != LatchState.OFFLINE && oldState == LatchState.ACTIVE) {
                    lowerAlert(alert, "Vacuum PLC error condition cleared");
                }
		changed = true;
            }
        }

	/*
	  conditions currently in VacSysState:
    public static final int
        COND_CR_FORELINE_VAC = 0,
        COND_CR_TRB_PUMP_OFF = 1,
        COND_CR_TRB_PRESS_10 = 2,
        COND_CR_VACUUM_01    = 3,
        COND_CR_VACUUM_001   = 4,
        COND_HX_FORELINE_VAC = 5,
        COND_HX_TURBO_OFF    = 6,
        COND_HX_TRB_PRESS_10 = 7,
        COND_HX_VACUUM_01    = 8,
        COND_HX_VACUUM_001   = 9,
        COND_OR_FPP_VALVE    = 10,
        COND_OR_FH_VALVE     = 11,
        COND_OR_L3H_VALVE    = 12,
        COND_OR_L3_VALVE     = 13,
        NUM_CONDITIONS       = 14;

	*/



        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if (!vacState.hasCondition(cond)) continue;

            Boolean active = plutoDevc.isConditionActive(cond);

	    /* The ComCam vacuum pluto PLC is not currently (nor foreseen) to be connected to the vacuum gauge
               the vacuum conditions are being set by CCS */
	    /*
	    if ( cond == VacSysState.COND_CR_TRB_PRESS_10)
		if (pfPr < 10.0 ) 
		    active = true ;
		else
		    active = false ;

	    if ( cond == VacSysState.COND_CR_VACUUM_01)
		if (pfPr < 0.1 ) 
		    active = true ;
		else
		    active = false ;

	    if ( cond == VacSysState.COND_CR_VACUUM_001)
		if (pfPr < 0.001 ) 
		    active = true ;
		else
		    active = false ;
	    */

            ConditionState state = active == null ? ConditionState.OFF :
		active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
		vacState.setCondition(cond, state);
		changed = true;
            }
        }

	/* ----- */

        VacuumState vState;
//	PathfinderSharedVacState.VacuumState cvState;
        if (Double.isNaN(pfPr)) {
            vState = VacuumState.UNKNOWN;
//            cvState = PathfinderSharedVacState.VacuumState.UNKNOWN;
        } else if (pfPr <= PRESS_VACUUM && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.VACUUM;
//            cvState = PathfinderSharedVacState.VacuumState.VACUUM;
        } else if (vacState.getSwitchState(VacSysState.SW_HX_ION_PUMP1) == SwitchState.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.ION_ON;
//            cvState = PathfinderSharedVacState.VacuumState.ION_ON;
        } else if (vacState.getSwitchEnable(VacSysState.SW_HX_ION_PUMP1) == SwitchEnable.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.ION_OFF;
//            cvState = PathfinderSharedVacState.VacuumState.ION_OFF;
        } else if (vacState.getSwitchState(VacSysState.SW_HX_TURBO) == SwitchState.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_ON;
//            cvState = PathfinderSharedVacState.VacuumState.TURBO_ON;
        } else if (vacState.getSwitchEnable(VacSysState.SW_HX_TURBO) == SwitchEnable.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_OFF;
//            cvState = PathfinderSharedVacState.VacuumState.TURBO_OFF;
        } else if (pfPr < PRESS_ATMOS) {
            vState = VacuumState.FORELINE;
//            cvState = PathfinderSharedVacState.VacuumState.FORELINE;
        } else {
            vState = VacuumState.OFF;
//            cvState = PathfinderSharedVacState.VacuumState.OFF;
        }
        if (vState != vacState.getVacuumState()) {
            vacState.setVacuumState(vState);
            ass.updateAgentState(vState);
            changed = true;
        }

        if (changed) {
            publishState();
        }

	// publish the shared state
/*
        if (cvState != pathfinderSharedVacState.getVacuumState()) {
            pathfinderSharedVacState.setVacuumState(cvState);
        }
	pathfinderSharedVacState.setPressure(cryoPr);

	pathfinderSharedVacState.setForceValveClosing(forcedVATclosing || isMonAlertEnabledVATShut());
*/	
	if (forcedVATclosing || isMonAlertEnabledVATShut())
	    vacState.setSwitchState(VacSysState.SW_HX_VALVE, SwitchState.OFF);

/*
        if (vacState.getSwitchState(VacSysState.SW_HX_VALVE) != SwitchState.OFFLINE) {
	    pathfinderSharedVacState.setValveOpenStateRequested(vacState.getSwitchState(VacSysState.SW_HX_VALVE) == SwitchState.ON);
	} else {
	    pathfinderSharedVacState.setValveOpenStateRequested(null);
	}
	pathfinderSharedVacState.set_COND_CR_TRB_PRESS_10(vacState.getCondition(VacSysState.COND_CR_TRB_PRESS_10) == ConditionState.YES);
	pathfinderSharedVacState.set_COND_CR_VACUUM_01(vacState.getCondition(VacSysState.COND_CR_VACUUM_01) == ConditionState.YES);
	pathfinderSharedVacState.set_COND_CR_VACUUM_001(vacState.getCondition(VacSysState.COND_CR_VACUUM_001) == ConditionState.YES);
	pathfinderSharedVacState.setValveEnable(vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON);

        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(PathfinderSharedVacState.KEY, getPathfinderSharedVacState()));
*/
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable alert response")
    public void enableAlertResponse(boolean enable) {
        this.alertResponseEnabled = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isAlertResponseEnabled() {
        return alertResponseEnabled;
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable vacuum monitoring VAT shutting action")
    public void setMonAlertEnabledVATShut(boolean enable) {
        this.monAlertEnabledVATShut = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isMonAlertEnabledVATShut() {
        return monAlertEnabledVATShut;
    }

    /**
     *  Raises an alert.
     *
     *  @param  alert  The vacuum alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(PathfinderAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The vacuum alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(PathfinderAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }


    /**
     *  Alert event handler. 
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        
        if ( event.getType() == AlertEvent.AlertEventType.ALERT_RAISED && isAlertResponseEnabled() ) {
            Alert raisedAlert = event.getAlert();
            String alertId = raisedAlert.getAlertId();
            
            if ( alertId.equals(PathfinderAlerts.PRESSURE_TOO_HIGH.getAlert().getAlertId()) ) {
                if ( event.getLevel() == AlertState.ALARM ) {
                    //Shut the valve, then fork off the rest of the response
		    setMonAlertEnabledVATShut(true);
		    LOG.info("Shutting the vacuum valve!");
                    
                }
            }
            
            if ( alertId.equals(PathfinderAlerts.LOAD_RTD_OVERTEMP.getAlert().getAlertId()) ||
                   alertId.equals(PathfinderAlerts.TEMP_MAX_TOO_HIGH.getAlert().getAlertId()) ) {
                if (event.getLevel() == AlertState.ALARM) {
                    //Alarm triggered. Get the heater turned down to their minimum setting
                    for(int itry=0;itry<2;itry++) {
                        try {
                            setHeater(0,4.0);
                        } catch (VacuumException ve) {
                            LOG.log(Level.SEVERE, "Failed to turn off heater1 -- manual intervention needed{0}try={1}", new Object[]{ve, itry});
                        }
                        try {
			setHeater(1,4.0);
                        } catch (VacuumException ve) {
                            LOG.log(Level.SEVERE, "Failed to turn off heater2 -- manual intervention needed{0}try={1}", new Object[]{ve, itry});
                        }
                    }
                    LOG.info("Set SCR heaters to their minimum setting!");
                }
            }

	}
        
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     * Publishes the state of the vacuum system.
     *
     * This is intended to be called whenever any element of the state is
     * changed.
     */
    private void publishState() {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
//        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(PathfinderSharedVacState.KEY, getPathfinderSharedVacState()));

    }

    /**
     * Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period) {
        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }

    /**
     * Gets the monitoring publishing period.
     */
    private int getTickPeriod() {
        return (int) apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }



    /**                                                                                                                                                                                                      
     *  Sets a fan state (using its name).                                                                                                                                                                   
     *                                                                                                                                                                                                       
     *  @param  name   The fan name                                                                                                                                                                          
     *  @param  state  The state to set                                                                                                                                                                      
     *  @throws  VacuumException                                                                                                                                                                            
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan control state")
    public void setFanState(@Argument(description="The fan name") String name,
                            @Argument(description="The control state") FanState state) throws VacuumException
    {
        try {
            int fanId = UtilTrunkFans.getId(name);
            if (state == FanState.OFFLINE || (state == FanState.TEMP && !UtilTrunkFans.hasTempState(fanId))) {
                throw new VacuumException("Invalid " + name + " state: " + state);
            }
            synchronized (vacState) {
                if (state == vacState.getFanBaseState(fanId)) return;
                vacState.setFanBaseState(fanId, state);
                if (vacState.getFanState(fanId) == FanState.OFFLINE) return;
                vacState.setFanState(fanId, state);
                FanPIControl ctrlr = fanControl[fanId];
                if (state == FanState.TEMP) {
                    ctrlr.setTemperature(vacState.getDeltaTemp(fanId));
                    ctrlr.startLoop(vacState.getFanSpeed(fanId));
                }
                else {
                    if (UtilTrunkFans.hasTempState(fanId)) {
                        ctrlr.stopLoop();
                    }
                    ctrlr.setSpeed(state == FanState.SPEED ? vacState.getFanSpeed(fanId) : 0.0);
                }
            }
        }
        finally {
            publishState();
        }
    }



    /**
     *  Sets a fan speed (using its name).
     *
     *  @param  name   The fan name
     *  @param  speed  The fan speed
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a fan speed")
    public void setFanSpeed(@Argument(description="The fan name") String name,
                            @Argument(description="The fan speed") double speed) throws VacuumException
    {
	LOG.info("SetFanSpeed called with speed = "+speed);

        try {
            int fanId = UtilTrunkFans.getId(name);
	    LOG.info("SetFanSpeed called with fanId = "+fanId);
            synchronized (vacState) {
                vacState.setFanSpeed(fanId, speed);
		LOG.info("SetFanSpeed checking fanState. State = "+ vacState.getFanState(fanId));
                if (vacState.getFanState(fanId) == FanState.SPEED) {
		    LOG.info("SetFanSpeed calling setspeed");
                    fanControl[fanId].setSpeed(speed);
                }
            }
        }
        finally {
            publishState();
        }
    }

    /**
     *  Gets the subsystem state.*
     *  @return  The full PDU system state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the full state")
    public VacSysState getFullState()
    {
        vacState.setTickMillis(monitorControl.getPublishPeriod());
        return vacState;
    }


    /**
     *  Gets the list of fan names.
     *
     *  @return  The list of names
     */
    @Command(type=Command.CommandType.QUERY, description="Get list of fan names")
    public List<String> getFanNames()
    {
        return UtilTrunkFans.getNames();
    }


    /**
     *  Checks a trim heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  VacuumException if the number is invalid or object is null
     */

    private TrimHeater getTrimHeater(int htr) throws VacuumException
    {
        return trimHeaters[htr];
    }


}
