package org.lsst.ccs.subsystem.pathfinder;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.monitor.Alarm;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.pathfinder.alerts.PathfinderAlerts;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;

/**
 * A class with information about Pathfinder alarms
 *
 * @author The LSST CCS Team
 */
public class PathfinderAlertHandler implements HasLifecycle, Monitor.AlarmHandler, ClearAlertHandler {

    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.pathfinder.alarms");

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    //The ComCam Vacuum subsystem
    @LookupField(strategy = LookupField.Strategy.TREE)
    private PathfinderSubsystem pathfinder;


    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {

	LOG.info("processing alarms .... alert response enable = "+pathfinder.isAlertResponseEnabled());

        if ( (parm == PathfinderAlerts.PRESSURE_TOO_HIGH.ordinal()) && pathfinder.isAlertResponseEnabled() ) {
            alertService.raiseAlert(PathfinderAlerts.PRESSURE_TOO_HIGH.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
	    switch (event) {
	    case Alarm.EVENT_TRIP:                    
		//Alarm triggered
		//		try { //Shut the valve, then fork off the rest of the response
		    pathfinder.setMonAlertEnabledVATShut(true);
		    LOG.info("Shutting the vacuum valve!");
		/*
		} catch (DriverException de) {
                    LOG.severe("Failed to shut VACUUMVALVE outlet -- manual intervention needed" + de);
		}
		*/
		Thread t = new Thread(new VacuumAlarmResponse());
		t.start();
		break;
	    case Alarm.EVENT_LIMBO_FROM_ERROR:
		//In Warning band from Alarm
		break;
	    case Alarm.EVENT_LIMBO_FROM_GOOD:
		//In Warning band from Nominal
		break;
	    case Alarm.EVENT_RESET:
		//Back to Nominal
		break;
            }
            return true;
	}
        if ( (parm == PathfinderAlerts.LOAD_RTD_OVERTEMP.ordinal()) && pathfinder.isAlertResponseEnabled() ) {
            alertService.raiseAlert(PathfinderAlerts.LOAD_RTD_OVERTEMP.getAlert(), getAlarmStateFromEvent(event), alarmName + ": " + cause);            
	    switch (event) {
	    case Alarm.EVENT_TRIP:                    
		//Alarm triggered. Get the heater turned down to their minimum setting
		for(int itry=0;itry<2;itry++) {
		    try {
			pathfinder.setHeater(0,4.0);
		    } catch (VacuumException ve) {
			LOG.severe("Failed to turn off heater1 -- manual intervention needed" + ve + "try=" + itry);
		    }
		    try {
			pathfinder.setHeater(1,4.0);
		    } catch (VacuumException ve) {
			LOG.severe("Failed to turn off heater2 -- manual intervention needed" + ve + "try=" + itry);
		    }
		}
		LOG.info("Set SCR heaters to their minimum setting!");
		break;
	    case Alarm.EVENT_LIMBO_FROM_ERROR:
		//In Warning band from Alarm
		break;
	    case Alarm.EVENT_LIMBO_FROM_GOOD:
		//In Warning band from Nominal
		break;
	    case Alarm.EVENT_RESET:
		//Back to Nominal
		break;
            }
            return true;
	}
        return false;
    }


    class VacuumAlarmResponse implements Runnable {
        @Override
        public void run() {
	    // place holder
        }
    }


    private static AlertState getAlarmStateFromEvent(int event) {
        switch (event) {
            //The High Alarm treshold has been crossed.
	case Alarm.EVENT_TRIP:
	    return AlertState.ALARM;
	case Alarm.EVENT_LIMBO_FROM_ERROR:
	    return AlertState.WARNING;
	case Alarm.EVENT_LIMBO_FROM_GOOD:
	    return AlertState.WARNING;
	case Alarm.EVENT_RESET:
	    return AlertState.NOMINAL;
        }
        throw new IllegalArgumentException("Unknown event: id " + event);
    }
 
    
    @Override
    public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState state) {

        for ( PathfinderAlerts pathfinderAlert : PathfinderAlerts.values() ) {
            if ( pathfinderAlert.getId().equals(alert.getAlertId() ) ) {
                Alarm a = getAlarmById(pathfinderAlert.ordinal());
                if ( a != null ) {
                    a.clearState();
                    return ClearAlertCode.CLEAR_ALERT;
                }
            }
        }

        return ClearAlertCode.UNKNOWN_ALERT;
    }
    
    
}
