package org.lsst.ccs.subsystem.pathfinder;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
//import org.lsst.ccs.subsystem.common.actions.PathfinderSharedVacState;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.twistorr.TwisTorr84;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.Monitor.AlarmHandler;
import org.lsst.ccs.monitor.MonitorLogUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.pathfinder.alerts.PathfinderAlerts;
import org.lsst.ccs.subsystem.pathfinder.constants.PLCState;
import org.lsst.ccs.subsystem.pathfinder.constants.ConditionState;
import org.lsst.ccs.subsystem.pathfinder.constants.LatchState;
import org.lsst.ccs.subsystem.pathfinder.constants.DeviceState;
//import org.lsst.ccs.subsystem.vacuum.constants.ImageChannel;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchEnable;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchState;
import org.lsst.ccs.subsystem.pathfinder.data.PathfinderAgentProperties;
import org.lsst.ccs.subsystem.pathfinder.constants.VacuumState;
import org.lsst.ccs.subsystem.pathfinder.data.VacuumException;
import org.lsst.ccs.subsystem.pathfinder.data.VacSysState;
import org.lsst.ccs.bus.messages.StatusMessage;

import org.lsst.ccs.utilities.logging.Logger;

//import org.lsst.ccs.common.devices.power;
import org.lsst.ccs.subsystem.common.devices.turbopump.TwisTorr84Device;
import org.lsst.ccs.subsystem.common.devices.refrigeration.AVCCryoDevice;
import org.lsst.ccs.subsystem.common.devices.vacuum.GPVacMon835Device;
//import org.lsst.ccs.subsystem.common.devices.power.distribution.APC7900Device;
//import org.lsst.ccs.subsystem.common.devices.power.distribution.state.PduOutletState;

import org.lsst.ccs.subsystem.common.devices.refrigeration.data.RefrigerationConfig;

/**
 * The PathfinderSubsystem is designed specifically for running ComCam
 * thermal and vacuum control,
 */
//public class PathfinderSubsystem implements HasLifecycle, Monitor.AlarmHandler {
public class PathfinderSubsystem extends Subsystem implements HasLifecycle, Monitor.AlarmHandler, ClearAlertHandler, AlertListener {

    // Instances of the foreline pressure appearing below when there
    // is no foreline pressure gauge intended for ComCamare for placeholders
    // for a future similar check
    
    private static final double PRESS_ATMOS = 759.0,
            PRESS_TURBO_LOW = 5.0,
            PRESS_FORELINE_LOW = 5.0,
            PRESS_DIFF_LOW = 0.09,
            PRESS_DIFF_HIGH = 20.0,
            PRESS_ION_OFF = 1.0e-5,
            PRESS_ION_ENABLE = 1.2e-6,
            PRESS_VACUUM = 1.0e-7,
            COMCAM_PRESS_HIGH = 20.0,
            TURBO_MAX = 81000,
            TURBO_LOW = 0.1 * TURBO_MAX,
            TURBO_HIGH = 0.5 * TURBO_MAX;
    private static final int[] switchChannels = new int[VacSysState.NUM_SWITCHES];

    static {
        switchChannels[VacSysState.SW_HX_ION_PUMP1] = IonPumpDevice.CHAN_CIP2;
        switchChannels[VacSysState.SW_OR_ION_PUMP] = IonPumpDevice.CHAN_OIP;
        switchChannels[VacSysState.SW_HX_VALVE] = VacPlutoDevice.SW_OPEN_VHX00;

    }
    private static final Map<TwisTorr84.PumpStatus, DeviceState> turboStateMap = new HashMap<>();

    static {
        turboStateMap.put(TwisTorr84.PumpStatus.STOP, DeviceState.STOPPED);
        turboStateMap.put(TwisTorr84.PumpStatus.WAIT_INTLK, DeviceState.WAITING);
        turboStateMap.put(TwisTorr84.PumpStatus.STARTING, DeviceState.STARTNG);
        turboStateMap.put(TwisTorr84.PumpStatus.NORMAL, DeviceState.NORMAL);
        turboStateMap.put(TwisTorr84.PumpStatus.BRAKING, DeviceState.BRAKING);
        turboStateMap.put(TwisTorr84.PumpStatus.FAIL, DeviceState.FAILED);
        turboStateMap.put(TwisTorr84.PumpStatus.AUTO_TUNING, DeviceState.AUTOTUN);
    }

    private static final Map<Integer, PathfinderAlerts> alertMap = new HashMap<>();
    static {
        alertMap.put(VacSysState.LATCH_HX_VACUUM, PathfinderAlerts.HX_VACUUM_BAD);
        alertMap.put(VacSysState.LATCH_HX_GATE_NFC, PathfinderAlerts.HX_GATE_FORCED_SHUT);
        alertMap.put(VacSysState.LATCH_HX_GATE_AO, PathfinderAlerts.HX_GATE_CANNOT_OPEN);
        alertMap.put(VacSysState.LATCH_HX_PUMP, PathfinderAlerts.HX_TURBO_PUMP_BAD);
    }
    private static final Map<String, Integer> revAlertMap = new HashMap<>();
    static {
        for (int cond : alertMap.keySet()) {
            revAlertMap.put(alertMap.get(cond).getId(), cond);
        }
    }


    @LookupName
    private String name;

//    private static final Logger LOG = Logger.getLogger("org.lsst.ccs.subsystem.pathfinder");
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private AlarmHandler alarmHandler;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService apts;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService ass;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private VacPlutoDevice plutoDevc = null;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private CryoTurboDevice cryoTurboDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private IonPumpDevice ionPumpDevc;
//    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
//    private AVCCryoDevice cryotelDevc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    Map<String, AVCCryoDevice> avcMap = new HashMap<>();

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> channelMap = new LinkedHashMap<>();

    // From Groovy file
    private String pressChan1, pressChan2, turboSpeedChan, forelinePressChan, CryoTempChan,
            Cold1TempChan, Cold2TempChan, IonPumpVoltageChan;
    private List<Integer> switches;

    //    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*20.*")
    //    private APC7900Device pdu20Devc;
    
    //    @LookupField(strategy = LookupField.Strategy.DESCENDANTS, pathFilter = ".*15.*")
    //    private APC7900Device pdu15Devc;
    
//    Map<String, APC7900Device> pduMap = new HashMap<>();
//    private APC7900Device pdu20Devc = pduMap.get("PDU20");
    
    // General

    private static final Logger LOG = Logger.getLogger(PathfinderSubsystem.class.getName());

    private final VacSysState vacState = new VacSysState();
//    private final PathfinderSharedVacState pathfinderSharedVacState = new PathfinderSharedVacState();
    private Set<Integer> switchSet;
    private final Device[] switchDevices = new Device[VacSysState.NUM_SWITCHES];
    private Channel cryoPressure, HXPressure, turboSpeed, forelinePressure, CryoTemp, Cold1Temp, Cold2Temp;
    private boolean running = true;
    private final Map<String, Integer> switchNameMap = new LinkedHashMap<>();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private boolean plcActive = true;

    private volatile boolean alertResponseEnabled = true;

    private boolean monAlertEnabledVATShut = false;

//    private final Map<String, ChanParams> monChans = new HashMap<>();
//    private final ClientFactory clientFactory;
//    private PathfinderCommands vacuumCommands;
    private String last_alertStr;

//    private final GlobalProc globalProc = new GlobalProc();
    private boolean useInfoAlerts = System.getProperty("org.lsst.ccs.subsystem.pathfinderPowerInfoAlerts", "false").contains("true");

    @LookupField(strategy = LookupField.Strategy.TREE)
    AgentStatusAggregatorService aggregatorService;

    /**
     * Main constructor for the PathfinderSubsystem, normally invoked from the
     * corresponding groovy file.
     *
     */

    public PathfinderSubsystem() {
        super("pathfinder", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

    @Override
    public void postInit() {

	// Add alert listener
        alertService.addListener(this);

//        vacuumCommands = 
        //Set a property to define that this Agent is a vacuum subsystem.
        agentPropertiesService.setAgentProperty(PathfinderAgentProperties.PATHFINDER_TYPE_AGENT_PROPERTY, PathfinderSubsystem.class.getCanonicalName());

        if (switches != null) {
            switchSet = new HashSet<>(switches);
        } else {
            MonitorLogUtils.reportConfigError(LOG, name, "Switch list", "not specified");
        }
	//hh
        if (plutoDevc != null) {
            for (int cond : plutoDevc.getLatchIds()) {
                vacState.addLatch(cond);
                vacState.setLatch(cond, LatchState.CLEAR);
		LOG.info("Cleared latch cond  "+cond);
            }
             for (int cond : plutoDevc.getConditionIds()) {
                vacState.addCondition(cond);
                vacState.setCondition(cond, ConditionState.NO);
            }
        } else {
            MonitorLogUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }

        if (cryoTurboDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "PF turbo pump device", "not specified");
        }
        /*
        if (hxTurboDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "HX turbo pump device", "not specified");
        }
         */
        if (ionPumpDevc == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Ion pump device", "not specified");
        }
        if (forelinePressChan != null) {
            forelinePressure = channelMap.get(forelinePressChan);
        }
        if (pressChan2 != null) {
            cryoPressure = channelMap.get(pressChan2);
        }
        if (cryoPressure == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Cryo pressure channel", "not specified or not defined");
        }
        if (pressChan1 != null) {
            HXPressure = channelMap.get(pressChan1);
        }
        if (HXPressure == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "PF pressure channel", "not specified or not defined");
        }
        if (turboSpeedChan != null) {
            turboSpeed = channelMap.get(turboSpeedChan);
        }
        if (turboSpeed == null) {
            MonitorLogUtils.reportConfigError(LOG, name, "Turbo pump speed channel", "not specified or not defined");
        }

        List<Integer> ipChannels = ionPumpDevc.getChannelNumbers();
	LOG.info("ipChannel Names = " + ionPumpDevc.getChannelNames());
	LOG.info("ipChannel Numbers = " + ionPumpDevc.getChannelNumbers());
        Iterator swIter = switchSet.iterator();
        while (swIter.hasNext()) {
            int sw = (Integer) swIter.next();
            switch (sw) {
                case VacSysState.SW_HX_TURBO:
                    switchDevices[sw] = cryoTurboDevc;
                    break;
		case VacSysState.SW_HX_VALVE:
		    //                case VacSysState.SW_OR_FPP_VALVE:
		    //                case VacSysState.SW_OR_FH_VALVE:
		    //                case VacSysState.SW_OR_L3H_VALVE:
		    //                case VacSysState.SW_OR_L3_VALVE:
		    //                    switchDevices[sw] = plutoDevc; // HN - not connected to plutoDevc
                    switchDevices[sw] = null;
                    break;
		case VacSysState.SW_HX_ION_PUMP1:
		//                case VacSysState.SW_OR_ION_PUMP:
                    switchDevices[sw] = ionPumpDevc;
                    if (!ipChannels.contains(switchChannels[sw])) {
                        swIter.remove();
                    }
                    break;
		    //                case VacSysState.SW_KOOLANCE:
		    //                    switchDevices[sw] = pdu20Devc;
		    //                    break;
//                case VacSysState.SW_KOOLCOLD:
//                    switchDevices[sw] = pdu15Devc;
//                    break;
//                case VacSysState.SW_CRYOTEL_PWR:
//                    switchDevices[sw] = pdu20Devc;
//                    break;
            }
        }

        for (int sw : switchSet) {
            vacState.addSwitch(sw);
            vacState.setSwitchState(sw, SwitchState.OFFLINE);
            vacState.setSwitchEnable(sw, SwitchEnable.OFF);
        }

        for (String swName : SwitchNames.NAME_MAP.keySet()) {
            int sw = SwitchNames.NAME_MAP.get(swName);
            if (switchSet.contains(sw)) {
                switchNameMap.put(swName, sw);
            }
        }

        GPVacMon835Device vqm = new GPVacMon835Device();

//        subsys.addCommandsFromObject(vqm, "");
        LOG.info("Configured the status aggregator to listen to subsystems: " + System.getProperty("org.lsst.ccs.subsystem.teststand", "ts") + " " + System.getProperty("org.lsst.ccs.subsystem.rebps", "ccs-rebps") + " " + System.getProperty("org.lsst.ccs.subsystem.pathfinder", "pathfinder"));
    }

    @Override
    public void build() {
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("vacuum-state",
                () -> updateVacuumState()).withPeriod(Duration.ofMillis(1000));
        apts.scheduleAgentPeriodicTask(pt);

        ass.registerState(VacuumState.class, "ComCam Vacuum state", this);
        ass.updateAgentState(VacuumState.UNKNOWN);

        for (PathfinderAlerts alert : PathfinderAlerts.values()) {
            activeAlertMap.put(alert.getId(), false);
        }
    }

    @Override
    public void postStart() {
        LOG.info("Pathfinder subsystem started");
    }

    @Override
    public boolean processAlarm(int event, int parm, String cause, String alarmName) {
        if (alarmHandler != null) {
            return alarmHandler.processAlarm(event, parm, cause, alarmName);
        }
        return false;
    }

    /**
     ***************************************************************************
     **
     ** Sleep - what a waste
     * **************************************************************************
     */
    public void Sleep(double secs) {
        if (secs <= 0) {
            return;
        }
        try {
            Thread.sleep((int) (secs * 1000));
        } catch (InterruptedException ex) {
            LOG.error("Rude awakening!" + ex);
        }
    }

    /**
     * Gets the state of the Vacuum system.
     *
     * @return The vacuum state
     */
    @Command(type = Command.CommandType.QUERY, description = "Get the vacuum system state")
    public VacSysState getVacuumState() {
        vacState.setTickMillis(getTickPeriod());
        return vacState;
    }

    /**
     * Gets the shared state of the Vacuum system.
     *
     * @return The vacuum state
     */
/*
    @Command(type = Command.CommandType.QUERY, description = "Get the vacuum system state")
    public PathfinderSharedVacState getPathfinderSharedVacState() {
        pathfinderSharedVacState.setTickMillis(getTickPeriod());
        return pathfinderSharedVacState;
    }
*/
    /**
     * Gets the list of switch names.
     *
     * @return The switch names.
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.QUERY, description = "Get switch names")
    public List<String> getSwitchNames() throws VacuumException {
        return new ArrayList(switchNameMap.keySet());
    }

    /**
     * Turns a switch on or off.
     *
     * @param sw The switch number.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Turn on/off a switch")
    public void setSwitchOn(@Argument(description = "The switch number") int sw,
            @Argument(description = "Whether to turn on") boolean on) throws VacuumException {
        try {
            if (!switchSet.contains(sw)) {
                throw new VacuumException("Invalid switch number: " + sw);
            }
            setSwitch(sw, on);
        } finally {
            publishState();
        }

    }

    /**
     * Turns a named switch on or off.
     *
     * @param name The switch name.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    @Command(type = Command.CommandType.ACTION, description = "Turn on/off a named switch")
    public void setNamedSwitchOn(@Argument(description = "The switch name") String name,
            @Argument(description = "Whether to turn on") boolean on) throws VacuumException {
        Integer sw = switchNameMap.get(name);
        try {
            if (sw == null) {
                throw new VacuumException("Invalid switch name: " + name);
            }
            setSwitch(sw, on);
        } finally {
            publishState();
        }
    }

    /**
     * Turns a switch on or off.
     *
     * @param sw The switch number.
     * @param on Whether to turn on or off
     * @throws VacuumException
     */
    private void setSwitch(int sw, boolean on) throws VacuumException {
        SwitchState state = vacState.getSwitchState(sw);
        LOG.info("setSwitch called for sw = " + sw);
        if (state == SwitchState.OFFLINE) {
            LOG.info("Switch sw = " + sw + " is offline");
//	    if (!(sw == vacState.SW_CRYO_VALVE))
//		return;
//	    else
//		LOG.info("proceeding exceptionally for the VAT valve switch which will directly set the state as a request"); // HN

        }
        
        SwitchEnable enable = vacState.getSwitchEnable(sw);
        // setting switch on is rejected here if enable not on
        if (on && enable != SwitchEnable.ON && enable != SwitchEnable.WAS_ON) {
            LOG.info("enable = " + enable + " SwitchEnable.ON = " + SwitchEnable.ON + " SwitchEnable.WAS_ON = " + SwitchEnable.WAS_ON);
            return;
        }
        LOG.info("proceeding with setting switch sw = " + sw);
        Device swDevice = switchDevices[sw];
        try {
            if (swDevice instanceof TwisTorr84Device) {
                if (on) {
                    ((TwisTorr84Device) swDevice).startTurboPump();
                } else {
                    ((TwisTorr84Device) swDevice).stopTurboPump();
                }
		//            } else if (swDevice == plutoDevc) {
		//                plutoDevc.setSwitchOn(switchChannels[sw], on);
            } else if (swDevice == ionPumpDevc) {
                ionPumpDevc.setChannelOn(switchChannels[sw], on);
            } else if (sw == VacSysState.SW_HX_VALVE) {

		LOG.info("Setting VAT valve REQUEST state");

		if (on) {
		    vacState.setSwitchState(sw, SwitchState.ON);
//		    setMonAlertEnabledVATShut(false); // allow VAT valve opening after closing by monitoring
		} else {
		    vacState.setSwitchState(sw, SwitchState.OFF);
		}

            } else {
                LOG.info("swDevice not identified for sw=" + sw);
            }
        } catch (DriverException e) {
            throw new VacuumException(e);
        }
    }

    /**
     * Gets whether a switch is on.
     *
     * @param sw The switch number.
     * @return Whether the switch is on
     * @throws VacuumException
     */
    private Boolean isSwitchOn(int sw) {
        Boolean value = false;
        Device swDevice = switchDevices[sw];
        if (swDevice instanceof TwisTorr84Device) {
            try {
                DeviceState st = turboStateMap.get(((TwisTorr84Device) swDevice).readTurboStatus());
                value = st != DeviceState.STOPPED && st != DeviceState.BRAKING;
            } catch (DriverException e) {
                value = null;
            }

        } else if (swDevice == plutoDevc) {
	    if (plutoDevc!=null) 
		value = plutoDevc.isSwitchOn(switchChannels[sw]);

        } else if (swDevice == ionPumpDevc) {
            value = ionPumpDevc.isChannelOn(switchChannels[sw]);

        } else if (sw == VacSysState.SW_HX_VALVE) {

	    value = null;  // HN
	    if (vacState.getSwitchState(sw) != SwitchState.OFFLINE) // HN
		value = vacState.getSwitchState(sw) == SwitchState.ON;  // HN

        }
        return value;
    }
    /*
    /**
     * Clears a condition.
     *
     * @param cond The condition number.
     * @throws VacuumException
     * /
    @Command(type = Command.CommandType.ACTION, description = "Clear a condition")
    public void clearCondition(@Argument(description = "The condition number") int cond) throws VacuumException {
        try {
            plutoDevc.clearCondition(cond);
        } finally {
            publishState();
        }
    }
*/

    /**
     *  Clears a latched condition.
     *
     *  @param  cond  The condition number.
     *  @throws  VacuumException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a condition")
    public void clearLatch(@Argument(description="The condition number") int cond) throws VacuumException
    {
        try {
            plutoDevc.clearLatch(cond);
        }
        finally {
            publishState();
        }
    }


    /**
     * Sets the update period.
     *
     * @param value The update period (milliseconds) to set.
     */
    @Command(type = Command.CommandType.ACTION, description = "Set the update interval")
    public void setUpdatePeriod(@Argument(description = "The tick period (ms)") int value) {
        setTickPeriod(value);
        vacState.setTickMillis(getTickPeriod());
        publishState();
    }

    /**
     * Updates the vacuum system state periodically.
     *
     * The vacuum state consists mainly of the state of the switches (lines)
     * being controlled, along with whether they can be turned on.
     *
     * Pressures and pump speeds are read to determine which switches are
     * enabled and/or need to be turned off. If the state changes it is
     * published.
     */
    private void updateVacuumState() {
//        LOG.info("updateVacuumState called");


	if (!running) return;

        Boolean plcActive = plutoDevc.isPLCActive();
        if (plcActive == null) {
            if (vacState.getPlcState() != PLCState.OFFLINE) {
                raiseAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is offline");
                vacState.setPlcState(PLCState.OFFLINE);
            }
        }
        else if (!plcActive) {
            if (vacState.getPlcState() != PLCState.DEAD) {
                raiseAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC has died");
                vacState.setPlcState(PLCState.DEAD);
            }
        }
        else {
            if (vacState.getPlcState() != PLCState.ALIVE) {
                lowerAlert(PathfinderAlerts.VACUUM_PLC_NOT_ALIVE, "Vacuum PLC is alive");
                vacState.setPlcState(PLCState.ALIVE);
            }
        }


        double forelinePr = 0.0;
        if (this.forelinePressChan != null) {
            forelinePressure.readValue();
        }
        double cryoPr = cryoPressure.readValue();
	LOG.info("cryoPr = " + cryoPr);
	double hx1Pr = HXPressure.readValue();
	LOG.info("current hx1Pr = " + hx1Pr);
	//       double hx1Pr = cryoPr;

        double turboSp = turboSpeed.readValue();
        boolean changed = false;
	boolean forcedVATclosing = false;

        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if (!vacState.hasSwitch(sw)) {
//                LOG.info("sw = "+sw+" is not a switch");
                continue;
            }
//            try {
//                LOG.info("Updating status for switch "+sw);
            //+" name = "+this.getSwitchNames().get(sw));
//            } catch (VacuumException ex) {
//                LOG.error(ex);
//            }

            Boolean enable = false;
            boolean turnOff = false;
            DeviceState devState = null;
            switch (sw) {

                case VacSysState.SW_HX_VALVE:
		    /*                    
                    // When foreline pressure is available, insure that the pressure difference is
                    // low enogh to enable opening valve with difference limits depending on the turbo running or not.
                    // For ComCam, hx1Pr == hx1Pr and the following has no effect.
                    if (!Double.isNaN(cryoPr) && !Double.isNaN(hx1Pr) && !Double.isNaN(turboSp)) {
                        double prDiff = Math.abs(hx1Pr - cryoPr);
                        enable = (prDiff <= PRESS_DIFF_HIGH && turboSp < TURBO_LOW)
                                || (prDiff <= PRESS_DIFF_LOW && turboSp > TURBO_HIGH);
                    }
                    */
		    enable = true;
                    // to be anded with conditions that either the foreline press be low or the turbo speed be low
                    boolean enable2 = !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)
                            && (forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW);
                    enable &= enable2;
                    
                    // since ComCam doesn't have a foreline pressure reading, impose the following condition as well
                    boolean enable3 = hx1Pr > COMCAM_PRESS_HIGH || turboSp > TURBO_HIGH;
                    enable &= enable3;
                    
                    // do not close valve just based on a bad reading (added for ComCam)
                    if (!Double.isNaN(hx1Pr)) {
                        turnOff = !enable2;
                    }
		    forcedVATclosing = turnOff;

		    //		    LOG.info("VAT - enable = " + enable + " enable2 = " + enable2 + " enable3 = " + enable3); 

                    break;

                case VacSysState.SW_HX_TURBO:
                    LOG.fine("hx1Pr = " + hx1Pr + " forelinePr = " + forelinePr + " turboSp = " + turboSp);
		    /*
                    if (!Double.isNaN(hx1Pr) && !Double.isNaN(forelinePr) && !Double.isNaN(turboSp)) {
                        enable = hx1Pr < PRESS_TURBO_LOW && (forelinePr < PRESS_FORELINE_LOW || turboSp < TURBO_LOW);
                        LOG.fine("turbo enable = " + enable + " hx1Pr = " + hx1Pr + " PRESS_TURBO_LOW =" + PRESS_TURBO_LOW);
			}*/
                    if (!Double.isNaN(hx1Pr) && !Double.isNaN(turboSp)) {
                        enable = hx1Pr < PRESS_TURBO_LOW;
                        LOG.fine("turbo enable = " + enable + " hx1Pr = " + hx1Pr + " PRESS_TURBO_LOW =" + PRESS_TURBO_LOW);
                    }
                    // do not turn off just based on a single bad reading
                    if (!Double.isNaN(hx1Pr)) {
                        turnOff = !enable;
                    }
                    try {
                        devState = turboStateMap.get(((CryoTurboDevice) switchDevices[sw]).readTurboStatus());
                    } catch (DriverException e) {
                    }
                    break;

                case VacSysState.SW_HX_ION_PUMP1:
                    if (!Double.isNaN(cryoPr)) {
                        enable = cryoPr < PRESS_ION_ENABLE;
                        turnOff = cryoPr >= PRESS_ION_OFF;
                    } else {
                        turnOff = true;
                    }
                    break;
            }

            SwitchState oldState = vacState.getSwitchState(sw);
            if (turnOff && oldState == SwitchState.ON) {
                try {
                    setSwitch(sw, false);
                } catch (VacuumException e) {
                    LOG.error("Error setting switch: " + e);
                }
            }
            Boolean isOn = isSwitchOn(sw);
            SwitchState state = isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
            SwitchEnable enabled = enable ? SwitchEnable.ON : SwitchEnable.OFF;
            if (state != oldState || enabled != vacState.getSwitchEnable(sw)) {
                LOG.info("Enabling switch sw=" + sw + " enabled = " + enabled);
                vacState.setSwitchState(sw, state);
                vacState.setSwitchEnable(sw, enabled);
                changed = true;
            }
            if (devState != vacState.getDeviceState(sw)) {
                vacState.setDeviceState(sw, devState);
                changed = true;
            }
        }
	/*
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if (!vacState.hasCondition(cond)) {
                continue;
            }
	    Boolean active = false;
	    Boolean latched = false;
	    //hh
	    if (plutoDevc != null) {
		active = plutoDevc.isConditionActive(cond);
		latched = plutoDevc.isConditionLatched(cond);
	    }

            ConditionState state = active == null || latched == null ? ConditionState.OFFLINE
                    : latched ? ConditionState.LATCHED
                            : active ? ConditionState.ACTIVE : ConditionState.CLEAR;
            if (state != vacState.getCondition(cond)) {
                vacState.setCondition(cond, state);
                changed = true;
            }
        }
	*/
	/* ---- added 2020/1/28 ---- */
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if (!vacState.hasLatch(cond)) continue;
            Boolean active = plutoDevc.isLatchActive(cond);
            Boolean latched = plutoDevc.isLatchLatched(cond);

	    if (cond == VacSysState.LATCH_CR_GATE_NFC || cond == VacSysState.LATCH_CR_GATE_AO) {
		active = forcedVATclosing;
	    }

            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
		latched ? LatchState.LATCHED :
		active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = vacState.getLatch(cond);
            if (state != oldState) {
                vacState.setLatch(cond, state);
                PathfinderAlerts alert = alertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlert(alert, "Vacuum PLC error condition set");
		}
		else if (state != LatchState.OFFLINE && oldState == LatchState.ACTIVE) {
                    lowerAlert(alert, "Vacuum PLC error condition cleared");
                }
		changed = true;
            }
        }

	/*
	  conditions currently in VacSysState:
    public static final int
        COND_CR_FORELINE_VAC = 0,
        COND_CR_TRB_PUMP_OFF = 1,
        COND_CR_TRB_PRESS_10 = 2,
        COND_CR_VACUUM_01    = 3,
        COND_CR_VACUUM_001   = 4,
        COND_HX_FORELINE_VAC = 5,
        COND_HX_TURBO_OFF    = 6,
        COND_HX_TRB_PRESS_10 = 7,
        COND_HX_VACUUM_01    = 8,
        COND_HX_VACUUM_001   = 9,
        COND_OR_FPP_VALVE    = 10,
        COND_OR_FH_VALVE     = 11,
        COND_OR_L3H_VALVE    = 12,
        COND_OR_L3_VALVE     = 13,
        NUM_CONDITIONS       = 14;

	*/



        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if (!vacState.hasCondition(cond)) continue;
	    /* The ComCam vacuum pluto PLC is not currently (nor foreseen) to be connected to the vacuum gauge
               the vacuum conditions are being set by CCS */

            Boolean active = plutoDevc.isConditionActive(cond);

	    if ( cond == VacSysState.COND_CR_TRB_PRESS_10)
		if (cryoPr < 10.0 ) 
		    active = true ;
		else
		    active = false ;

	    if ( cond == VacSysState.COND_CR_VACUUM_01)
		if (cryoPr < 0.1 ) 
		    active = true ;
		else
		    active = false ;

	    if ( cond == VacSysState.COND_CR_VACUUM_001)
		if (cryoPr < 0.001 ) 
		    active = true ;
		else
		    active = false ;

            ConditionState state = active == null ? ConditionState.OFF :
		active ? ConditionState.YES : ConditionState.NO;
            if (state != vacState.getCondition(cond)) {
		vacState.setCondition(cond, state);
		changed = true;
            }
        }

	/* ----- */

        VacuumState vState;
//	PathfinderSharedVacState.VacuumState cvState;
        if (Double.isNaN(cryoPr)) {
            vState = VacuumState.UNKNOWN;
//            cvState = PathfinderSharedVacState.VacuumState.UNKNOWN;
        } else if (cryoPr <= PRESS_VACUUM && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.VACUUM;
//            cvState = PathfinderSharedVacState.VacuumState.VACUUM;
        } else if (vacState.getSwitchState(VacSysState.SW_HX_ION_PUMP1) == SwitchState.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.ION_ON;
//            cvState = PathfinderSharedVacState.VacuumState.ION_ON;
        } else if (vacState.getSwitchEnable(VacSysState.SW_HX_ION_PUMP1) == SwitchEnable.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.ION_OFF;
//            cvState = PathfinderSharedVacState.VacuumState.ION_OFF;
        } else if (vacState.getSwitchState(VacSysState.SW_HX_TURBO) == SwitchState.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_ON;
//            cvState = PathfinderSharedVacState.VacuumState.TURBO_ON;
        } else if (vacState.getSwitchEnable(VacSysState.SW_HX_TURBO) == SwitchEnable.ON && vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON) {
            vState = VacuumState.TURBO_OFF;
//            cvState = PathfinderSharedVacState.VacuumState.TURBO_OFF;
        } else if (cryoPr < PRESS_ATMOS) {
            vState = VacuumState.FORELINE;
//            cvState = PathfinderSharedVacState.VacuumState.FORELINE;
        } else {
            vState = VacuumState.OFF;
//            cvState = PathfinderSharedVacState.VacuumState.OFF;
        }
        if (vState != vacState.getVacuumState()) {
            vacState.setVacuumState(vState);
            ass.updateAgentState(vState);
            changed = true;
        }

        if (changed) {
            publishState();
        }

	// publish the shared state
/*
        if (cvState != pathfinderSharedVacState.getVacuumState()) {
            pathfinderSharedVacState.setVacuumState(cvState);
        }
	pathfinderSharedVacState.setPressure(cryoPr);

	pathfinderSharedVacState.setForceValveClosing(forcedVATclosing || isMonAlertEnabledVATShut());
*/	
	if (forcedVATclosing || isMonAlertEnabledVATShut())
	    vacState.setSwitchState(VacSysState.SW_HX_VALVE, SwitchState.OFF);

/*
        if (vacState.getSwitchState(VacSysState.SW_HX_VALVE) != SwitchState.OFFLINE) {
	    pathfinderSharedVacState.setValveOpenStateRequested(vacState.getSwitchState(VacSysState.SW_HX_VALVE) == SwitchState.ON);
	} else {
	    pathfinderSharedVacState.setValveOpenStateRequested(null);
	}
	pathfinderSharedVacState.set_COND_CR_TRB_PRESS_10(vacState.getCondition(VacSysState.COND_CR_TRB_PRESS_10) == ConditionState.YES);
	pathfinderSharedVacState.set_COND_CR_VACUUM_01(vacState.getCondition(VacSysState.COND_CR_VACUUM_01) == ConditionState.YES);
	pathfinderSharedVacState.set_COND_CR_VACUUM_001(vacState.getCondition(VacSysState.COND_CR_VACUUM_001) == ConditionState.YES);
	pathfinderSharedVacState.setValveEnable(vacState.getSwitchEnable(VacSysState.SW_HX_VALVE) == SwitchEnable.ON);

        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(PathfinderSharedVacState.KEY, getPathfinderSharedVacState()));
*/
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable alert response")
    public void enableAlertResponse(boolean enable) {
        this.alertResponseEnabled = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isAlertResponseEnabled() {
        return alertResponseEnabled;
    }

    @Command(type = Command.CommandType.ACTION, description = "Enable/disable vacuum monitoring VAT shutting action")
    public void setMonAlertEnabledVATShut(boolean enable) {
        this.monAlertEnabledVATShut = enable;
    }

    @Command(type = Command.CommandType.QUERY, description = "Get true/false if the alert response is enabled")
    public boolean isMonAlertEnabledVATShut() {
        return monAlertEnabledVATShut;
    }

    /**
     *  Raises an alert.
     *
     *  @param  alert  The vacuum alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlert(PathfinderAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
        activeAlertMap.put(alert.getId(), true);
    }


    /**                                                                                                      
     *  Checks whether an alert has been raised.
     *
     *  @param  alert  The vacuum alert to check
     *  @return  Whether it has been raised
     */
    private boolean isAlertRaised(PathfinderAlerts alert)
    {
        return activeAlertMap.get(alert.getId()).equals(Boolean.TRUE);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The vacuum alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(PathfinderAlerts alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
        activeAlertMap.put(alert.getId(), false);
    }


    /**
     * Callback to clear an {@code Alert} instance.
     *
     * @param alert The Alert instance to clear. 
     * @param alertState The AlertState for the provided Alert.
     * @return A ClearAlertCode to indicate which action is to be taken
     *         by the framework.
     */
    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState)
    {
        Boolean active = activeAlertMap.get(alert.getAlertId());
        return active == null ? ClearAlertCode.UNKNOWN_ALERT
	    : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    /**
     *  Alert event handler. 
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revAlertMap.get(id);
            if (cond != null) {
                try {
                    clearLatch(cond);
                }
                catch (VacuumException e) {
                    LOG.log(Level.SEVERE, "Error clearing latched PLC condition ({0}): {1}", new Object[]{cond, e});
                }
            }
        }
    }


    /**
     * Publishes the state of the vacuum system.
     *
     * This is intended to be called whenever any element of the state is
     * changed.
     */
    private void publishState() {
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(VacSysState.KEY, getVacuumState()));
//        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(PathfinderSharedVacState.KEY, getPathfinderSharedVacState()));

    }

    /**
     * Sets the monitoring publishing period.
     */
    private void setTickPeriod(long period) {
        apts.setPeriodicTaskPeriod("monitor-publish", Duration.ofMillis(period));
    }

    /**
     * Gets the monitoring publishing period.
     */
    private int getTickPeriod() {
        return (int) apts.getPeriodicTaskPeriod("monitor-publish").toMillis();
    }

}
