package org.lsst.ccs.subsystem.pathfinder.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.logging.Logger;
import javax.swing.border.TitledBorder;
import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import org.lsst.ccs.subsystem.common.ui.TextFieldX;
//import org.lsst.ccs.subsystem.pathfinder.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.jas.CommandSender;
import org.lsst.ccs.subsystem.common.ui.UiConstants;
import org.lsst.ccs.subsystem.common.ui.UiUtilities;
import org.lsst.ccs.subsystem.pathfinder.constants.LatchState;
import org.lsst.ccs.subsystem.pathfinder.constants.ConditionState;
import org.lsst.ccs.subsystem.pathfinder.constants.DeviceState;
import org.lsst.ccs.subsystem.pathfinder.constants.PLCState;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchEnable;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchState;
import org.lsst.ccs.subsystem.pathfinder.data.VacSysState;

/**
 *  Implements the utility monitoring panel.
 *
 *  @author Owen Saxton
 */
//public class PathfinderControlPanel extends javax.swing.JPanel {
public class PathfinderControlPanel extends JPanel implements UiUtilities.ActionHandler, CommandSender.ReplyHandler {

    static final Font FONT = new java.awt.Font("Tahoma", 1, 12);
    static final Color RED = new Color(175, 0, 0), GREEN = new Color(0, 175, 0),
                       BLUE = new Color(0, 75, 175), PURPLE = new Color(175, 0, 175);
    static final int SWTP_PUMP = 0, SWTP_VALVE = 1;
    static final String[] switchNames = new String[VacSysState.NUM_SWITCHES];
    static {
	switchNames[VacSysState.SW_HX_TURBO] = "Hx Turbo Pump";
	switchNames[VacSysState.SW_HX_ION_PUMP1] = "Hx Ion Pump";
        switchNames[VacSysState.SW_HX_VALVE] = "Hx Gate Valve";
    }
    static final int[] switchTypes = new int[VacSysState.NUM_SWITCHES];
    static {
	switchTypes[VacSysState.SW_HX_TURBO] = SWTP_PUMP;
	switchTypes[VacSysState.SW_HX_ION_PUMP1] = SWTP_PUMP;
        switchTypes[VacSysState.SW_HX_VALVE] = SWTP_VALVE;
    }

    private static final String[] latchNames = new String[VacSysState.NUM_LATCHES];
    static {
        latchNames[VacSysState.LATCH_HX_GATE_AO]  = "Hx Gate Can't Open";
        latchNames[VacSysState.LATCH_HX_GATE_NFC] = "Hx Gate Forced Shut";
        latchNames[VacSysState.LATCH_HX_VACUUM]   = "Hx Vacuum Bad";
        latchNames[VacSysState.LATCH_HX_PUMP]     = "Hx Turbo Pump Bad";
    }
    private static final String[] conditionNames = new String[VacSysState.NUM_CONDITIONS];
    static {
        conditionNames[VacSysState.COND_HX_FORELINE_VAC] = "HX Foreline OK";
        conditionNames[VacSysState.COND_HX_TRB_PRESS_10] = "HX Turbo Pr < 10";
        conditionNames[VacSysState.COND_HX_TURBO_OFF]    = "HX Turbo Pump Off";
        conditionNames[VacSysState.COND_HX_VACUUM_001]   = "HX Vacuum < 0.001";
        conditionNames[VacSysState.COND_HX_VACUUM_01]    = "HX Vacuum < 0.1";
    }

    private static final int sysStateWidth = UiUtilities.maxLabelWidth(new String[]{"RUNNING", "STOPPED"}, "");
    private static final int plcStateWidth = UiUtilities.maxEnumLabelWidth(PLCState.class);
    private static final int switchStatusWidth = Math.max(UiUtilities.maxEnumLabelWidth(SwitchState.class),
                                                          UiUtilities.maxEnumLabelWidth(DeviceState.class));
    private static final int latchStatusWidth = UiUtilities.maxEnumLabelWidth(LatchState.class);
    private static final int condStatusWidth = UiUtilities.maxEnumLabelWidth(ConditionState.class);

    /*
    static final String[] conditionNames = new String[VacSysState.NUM_CONDITIONS];
    static {
        conditionNames[VacSysState.COND_CR_GATE_AO] = "VAT Valve Can't Open";
        conditionNames[VacSysState.COND_CR_GATE_NFC] = "VAT Valve Forced Shut";
        conditionNames[VacSysState.COND_CR_VACUUM] = "Cryo Vacuum Bad";
        conditionNames[VacSysState.COND_CR_PUMP] = "Cryo Turbo Pump Bad";
        conditionNames[VacSysState.COND_HX_GATE_AO] = "Hx Gate Can't Open";
        conditionNames[VacSysState.COND_HX_GATE_NFC] = "Hx Gate Forced Shut";
        conditionNames[VacSysState.COND_HX_VACUUM] = "Hx Vacuum Bad";
        conditionNames[VacSysState.COND_HX_PUMP] = "Hx Turbo Pump Bad";
    }
    */

    final private CommandSender sender;
    private String subsys;
    private double mainPeriod = 5.0;


    private int switchMask = 0, latchMask = 0, conditionMask = 0;

    private final UiUtilities uiUtils;

    private final JPanel mainPanel = new JPanel();

    private final JPanel headPanel = new JPanel();
    private JLabel sysStateValue;
    private JLabel plcStateValue;
    private TextFieldX periodTextField;


    private final JLabel stateLabel = new JLabel("System State:  ");
    private final JLabel stateStatus = new JLabel("STOPPED");

    private final JPanel periodPanel = new JPanel();
    private final JLabel periodLabel = new JLabel("Update Period: ");
    //    private final JTextField periodTextField = new JTextField();
    private final JLabel periodUnitsLabel = new JLabel("sec");

    //private final JPanel[] switchPanel = new JPanel[VacuumState.NUM_SWITCHES];
    private final JPanel switchPanel = new JPanel();
    private final JLabel[] switchLabel = new JLabel[VacSysState.NUM_SWITCHES];
    private final JLabel[] switchStatus = new JLabel[VacSysState.NUM_SWITCHES];
    private final ButtonGroup[] switchBG = new ButtonGroup[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOffRB = new JRadioButton[VacSysState.NUM_SWITCHES];
    private final JRadioButton[] switchOnRB = new JRadioButton[VacSysState.NUM_SWITCHES];


    private JPanel latchPanel;
    private final JLabel[] latchLabel = new JLabel[VacSysState.NUM_LATCHES];
    private final JLabel[] latchStatus = new JLabel[VacSysState.NUM_LATCHES];
    private final JButton[] latchResetBtn = new JButton[VacSysState.NUM_LATCHES];


    private JPanel conditionPanel;
    private final JLabel[] conditionLabel = new JLabel[VacSysState.NUM_CONDITIONS];
    private final JLabel[] conditionStatus = new JLabel[VacSysState.NUM_CONDITIONS];

    private final JButton[] conditionResetBtn = new JButton[VacSysState.NUM_CONDITIONS];

    public PathfinderControlPanel(String subsys) {
        uiUtils = new UiUtilities(this);
        sender = new CommandSender(subsys, this);
        initComponents();
        (new DisablePanel()).run();
    }
    /*    public PathfinderControlPanel(CommandSender cons) {
        this.sender = cons;
        uiUtils = new UiUtilities(this);
	//        sender = new CommandSender(subsys, this);

        initComponents();
        (new UpdateVacStatus(new VacSysState())).run();
        (new DisableSystem()).run();
    }
    */

    public void setSubsystem(String name) {
        this.subsys = name;
    }

    private void initComponents() {

        // System status line                                                                                               
        sysStateValue = UiUtilities.newLabel("X", sysStateWidth);
        plcStateValue = UiUtilities.newLabel("X", plcStateWidth);
        periodTextField = uiUtils.newTextFieldX("00000.0", "U", TextFieldX.TYPE_DOUBLE);

         // System state
        stateLabel.setFont(FONT);
        stateStatus.setFont(FONT);
        Dimension d = stateStatus.getPreferredSize();
        d.width = 150;
        stateStatus.setMinimumSize(d);
        stateStatus.setPreferredSize(d);

        // Update period
        periodLabel.setFont(FONT);
        d = periodTextField.getPreferredSize();
        d.width = 45;
        periodTextField.setMinimumSize(d);
        periodTextField.setPreferredSize(d);
        periodTextField.setHorizontalAlignment(SwingConstants.CENTER);
        periodTextField.setText("...");
        periodTextField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent evt) {
                setUpdatePeriod();
            }
        });
        periodUnitsLabel.setFont(FONT);

        // Switch states
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            JLabel label = switchLabel[sw] = new JLabel(switchNames[sw]);
            label.setFont(FONT);
            d = label.getPreferredSize();
            d.width = 130;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            label = switchStatus[sw] = new JLabel("XXX");
            label.setFont(FONT);
            d = label.getPreferredSize();
            d.width = 80;
            label.setMinimumSize(d);
            label.setPreferredSize(d);
            int swType = switchTypes[sw];
            JRadioButton buttonOff = switchOffRB[sw] = new JRadioButton(swType == SWTP_PUMP ? "Off" : "Shut");
            buttonOff.setFont(FONT);
            //buttonOff.setText("Off");
            buttonOff.setFocusable(false);
            buttonOff.setName(Integer.toString(sw));
            buttonOff.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), false);
                }
            });
            JRadioButton buttonOn = switchOnRB[sw] = new JRadioButton(swType == SWTP_PUMP ? "On" : "Open");
            buttonOn.setFont(FONT);
            //buttonOn.setText("On");
            buttonOn.setFocusable(false);
            buttonOn.setName(Integer.toString(sw));
            buttonOn.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent evt) {
                    String name = ((JRadioButton)evt.getSource()).getName();
                    setSwitchOn(Integer.valueOf(name), true);
                }
            });
            ButtonGroup bg = switchBG[sw] = new ButtonGroup();
            bg.add(buttonOff);
            bg.add(buttonOn);
        }

        // Latched condition states                                                                                                                      
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            latchLabel[cond] = UiUtilities.newLabel(latchNames[cond] + ":", 0);
            latchStatus[cond] = UiUtilities.newLabel("X", latchStatusWidth);
            latchResetBtn[cond] = uiUtils.newButton("Reset", "R" + cond, true);
	}

        // Condition states                                                                                                                              
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            conditionLabel[cond] = UiUtilities.newLabel(conditionNames[cond] + ":", 0);
            conditionStatus[cond] = UiUtilities.newLabel("X", condStatusWidth);
        }


        // Lay out all the sub-panels
        periodPanel.add(periodTextField);
        periodPanel.add(periodUnitsLabel);


        // System status line (panel)                                                                                       
        headPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbh = new GridBagConstraints();
        gbh.insets = new Insets(5, 0, 5, 0);
        gbh.gridx = 0;
        gbh.gridy = 0;
        gbh.anchor = GridBagConstraints.WEST;
	gbh.insets.left = 5;
        headPanel.add(UiUtilities.newLabel("System State: ", 0), gbh);
        gbh.gridx++;
        headPanel.add(sysStateValue, gbh);
        gbh.gridx++;
        gbh.insets.left = 25;
        headPanel.add(UiUtilities.newLabel("PLC State: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
        headPanel.add(plcStateValue, gbh);
	gbh.gridx++;
	gbh.insets.left = 25;
        headPanel.add(UiUtilities.newLabel("Update Period: ", 0), gbh);
        gbh.gridx++;
        gbh.insets.left = 5;
	headPanel.add(periodTextField, gbh);
	gbh.gridx++;
        gbh.insets.right = 5;
        headPanel.add(UiUtilities.newLabel("sec", 0), gbh);


        // Prepare the switch panel
        TitledBorder border = BorderFactory.createTitledBorder("Switches");
        border.setTitleJustification(TitledBorder.CENTER);
        switchPanel.setBorder(border);
        switchPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.gridx = 0;
        gbs.gridy = 0;
        gbs.insets.bottom = -8;
        switchPanel.add(new JLabel(" "), gbs);

	// Prepare the latched conditions panel                                                                                                           
        latchPanel = UiUtilities.newBorderedPanel("PLC Error Conditions");
        GridBagConstraints gbl = new GridBagConstraints();
        gbl.gridx = 0;
        gbl.gridy = 0;
	gbl.insets.bottom = -8;
        latchPanel.add(new JLabel(" "), gbl);

	// Prepare the conditions panel                                                                                                                   
        conditionPanel = UiUtilities.newBorderedPanel("PLC Running Conditions");
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
	gbc.gridy = 0;
        gbc.insets.bottom = -8;
	conditionPanel.add(new JLabel(" "), gbc);


        // Lay out the main panel
        mainPanel.setLayout(new GridBagLayout());
        GridBagConstraints gbm = new GridBagConstraints();
        gbm.insets = new Insets(0, 0, 0, 0);
        gbm.anchor = GridBagConstraints.NORTH;
        gbm.gridx = 0;
        gbm.gridy = 0;
        mainPanel.add(headPanel, gbm);
        gbm.gridy++;
        mainPanel.add(switchPanel, gbm);
        gbm.gridy++;
        mainPanel.add(latchPanel, gbm);
        gbm.gridy++;
        mainPanel.add(conditionPanel, gbm);

        // Lay out the complete panel
        add(mainPanel);

    }

    private void updateSwitchLayout(int switches) {
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switchMask & (1 << sw)) != 0) {
                switchPanel.remove(switchLabel[sw]);
                switchPanel.remove(switchStatus[sw]);
                switchPanel.remove(switchOffRB[sw]);
                switchPanel.remove(switchOnRB[sw]);
            }
        }
        GridBagConstraints gbs = new GridBagConstraints();
        gbs.anchor = GridBagConstraints.NORTHWEST;
        gbs.gridy = 0;
        int colm = 0;
        for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
            if ((switches & (1 << sw)) != 0) {
                if (colm == 0) {
                    gbs.gridx = 0;
                    gbs.gridy++;
                }
                gbs.insets.top = 0;
                gbs.insets.left = colm == 0 ? 4 : 50;
                //gbs.anchor = GridBagConstraints.NORTHEAST;
                switchPanel.add(switchLabel[sw], gbs);
                gbs.insets.left = 0;
                gbs.gridx++;
                //gbs.anchor = GridBagConstraints.NORTHWEST;
                switchPanel.add(switchStatus[sw], gbs);
                gbs.gridx++;
                gbs.insets.top = -4;
                switchPanel.add(switchOffRB[sw], gbs);
                gbs.gridx++;
                gbs.insets.left = 6;
                gbs.insets.right = 3;
                switchPanel.add(switchOnRB[sw], gbs);
                gbs.insets.right = 0;
                gbs.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        switchMask = switches;
    }


    private void updateLatchLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((latchMask & (1 << cond)) != 0) {
                latchPanel.remove(latchLabel[cond]);
                latchPanel.remove(latchStatus[cond]);
                latchPanel.remove(latchResetBtn[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
                gbc.insets.left = colm == 0 ? 4 : 40;
                latchPanel.add(latchLabel[cond], gbc);
                gbc.insets.left = 4;
                gbc.gridx++;
                latchPanel.add(latchStatus[cond], gbc);
                gbc.gridx++;
                latchPanel.add(latchResetBtn[cond], gbc);
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        latchMask = conditions;
    }

    private void updateConditionLayout(int conditions) {
        for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
            if ((conditionMask & (1 << cond)) != 0) {
                conditionPanel.remove(conditionLabel[cond]);
                conditionPanel.remove(conditionStatus[cond]);
            }
        }
        GridBagConstraints gbc = new GridBagConstraints();
        gbc.anchor = GridBagConstraints.WEST;
        gbc.insets = new Insets(0, 0, 4, 4);
        gbc.gridy = -1;
        int colm = 0;
        for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
            if ((conditions & (1 << cond)) != 0) {
                if (colm == 0) {
                    gbc.gridx = 0;
                    gbc.gridy++;
                }
                gbc.insets.top = gbc.gridy == 0 ? 4 : 0;
                gbc.insets.left = colm == 0 ? 4 : 40;
                conditionPanel.add(conditionLabel[cond], gbc);
                gbc.gridx++;
                gbc.insets.left = 4;
                conditionPanel.add(conditionStatus[cond], gbc);
                gbc.gridx++;
                colm = (colm + 1) % 2;
            }
        }
        conditionMask = conditions;
    }



    private void setUpdatePeriod() {
        try {
            double value = Double.valueOf(periodTextField.getText());
            sender.sendCommand(subsys, "setUpdatePeriod", (int)(1000 * value));
            periodTextField.setEnabled(false);
        }
        catch(NumberFormatException nfe) {
            periodTextField.setText(String.valueOf(mainPeriod));
        }
    }

    private void setSwitchOn(int sw, boolean value) {
        switchOffRB[sw].setEnabled(false);
        switchOnRB[sw].setEnabled(false);
        sender.sendCommand(subsys, "setSwitchOn", sw, value);
    }
    
    private void clearCondition(int cond) {
        conditionResetBtn[cond].setEnabled(false);
        sender.sendCommand(subsys, "clearCondition", cond);
    }
    
    public void initPanel() {
	sender.sendCommand(true, null, "getVacuumState");
    }

    @Override
    public void onCommandReply(Object reply, String path, String command, Object[] args) {
	updatePanel((VacSysState)reply);
    }

    public void updatePanel(VacSysState rs) {
        SwingUtilities.invokeLater(new UpdateVacStatus(rs));
    }

    public void disablePanel() {
        SwingUtilities.invokeLater(new DisablePanel());
    }
    /*
    public void disableSystem() {
        SwingUtilities.invokeLater(new DisableSystem());
    }
    */

    class UpdateVacStatus implements Runnable {

        private final VacSysState vs;
	private final Logger sLog = Logger.getLogger("org.lsst.ccs.subsystem.pathfinder.ui.PathfinderControlPanel");



        UpdateVacStatus(VacSysState vs) {
            this.vs = vs;
        }

        @Override
        public void run() {

            sysStateValue.setText("RUNNING");
            sysStateValue.setForeground(UiConstants.GREEN);
            PLCState plcState = vs.getPlcState();
	    plcStateValue.setText(plcState.name());
            plcStateValue.setForeground(plcState == PLCState.ALIVE ? UiConstants.GREEN :
                                        plcState == PLCState.OFFLINE ? UiConstants.BLUE : UiConstants.RED);
            plcStateValue.setEnabled(true);
            periodTextField.update(vs.getTickMillis() / 1000.0, true);

	    /*            stateStatus.setText("RUNNING");
            stateStatus.setForeground(GREEN);
            periodTextField.setEnabled(true);
            mainPeriod = vs.getTickMillis() / 1000.0;
            periodTextField.setText(String.valueOf(mainPeriod));
	    */

            int switches = 0;
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                if (!vs.hasSwitch(sw)) continue;
                switches |= (1 << sw);
                SwitchState state = vs.getSwitchState(sw);

                String text;
                Color color;
                SwitchEnable enable = vs.getSwitchEnable(sw);
                if (enable == SwitchEnable.OFF || enable == SwitchEnable.WAS_OFF) {
                    text = "DISABLD";
                    color = RED;
                }
                else {
                    text = state.name();
                    if (switchTypes[sw] == SWTP_VALVE) {
                        text = state == SwitchState.OFF ? "SHUT" : state == SwitchState.ON ? "OPEN" : text;
                    }
                    color = state == SwitchState.OFF ? Color.black : state == SwitchState.ON ? GREEN : BLUE;
                    DeviceState devState = vs.getDeviceState(sw);
                    if (devState != null && state != SwitchState.OFFLINE) {
                        text = devState.name();
                        if (devState == DeviceState.FAILED) {
                            color = RED;
                        }
                    }
                }
                switchStatus[sw].setText(text);
                switchStatus[sw].setForeground(color);

                JRadioButton selButton = state == SwitchState.ON ? switchOnRB[sw] : switchOffRB[sw];
                selButton.setSelected(true);
                switchStatus[sw].setEnabled(true);
                switchOffRB[sw].setEnabled(true);
                switchOnRB[sw].setEnabled(true);
            }
            if (switches != switchMask) {
                updateSwitchLayout(switches);
            }



            int conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
                if (!vs.hasLatch(cond)) continue;
                conditions |= (1 << cond);
                LatchState state = vs.getLatch(cond);
		//		if ((latchMask & (1 << cond)) != 0) {

		    sLog.info("PathfinderControlPanel: cond = "+cond+" name = "+state.name()+" length of latchStatus is "+latchStatus.length);

		    sLog.info("PathfinderControlPanel: latchStatus = "+latchStatus);

		    latchStatus[cond].setText(state.name());
		    latchStatus[cond].setForeground(state == LatchState.OFFLINE ? UiConstants.BLUE :
                                                state == LatchState.CLEAR ? UiConstants.GREEN :
						state == LatchState.ACTIVE ? UiConstants.RED : UiConstants.PURPLE);
		    latchStatus[cond].setEnabled(true);
		    latchResetBtn[cond].setEnabled(true);
		    //		}
            }
            if (conditions != latchMask) {
                updateLatchLayout(conditions);
            }
            conditions = 0;
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                if (!vs.hasCondition(cond)) continue;
                conditions |= (1 << cond);
                ConditionState state = vs.getCondition(cond);
		conditionStatus[cond].setText(state.name());
                conditionStatus[cond].setForeground(state == ConditionState.OFF ? UiConstants.BLUE :
                                                    state == ConditionState.NO ? Color.BLACK : UiConstants.GREEN);
                conditionStatus[cond].setEnabled(true);
	    }
            if (conditions != conditionMask) {
                updateConditionLayout(conditions);
            }



        }

    }

    class DisablePanel implements Runnable {

	@Override
        public void run() {

            sysStateValue.setText("STOPPED");
            sysStateValue.setForeground(UiConstants.RED);
            plcStateValue.setEnabled(false);
            periodTextField.setDisabled();
            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
		switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_LATCHES; cond++) {
		try {
		    latchStatus[cond].setEnabled(false);
		    latchResetBtn[cond].setEnabled(false);
		} catch(NullPointerException npe) {
		    break;
		} 
            }
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
		try {
		    conditionStatus[cond].setEnabled(false);
		} catch(NullPointerException npe) {
		    break;
		} 
            }
        }
    }

    /*
class DisableSystem implements Runnable {

        @Override
        public void run() {

            sysStateValue.setText("STOPPED");
            sysStateValue.setForeground(UiConstants.RED);
            plcStateValue.setEnabled(false);
	    periodTextField.setDisabled();

	    /*
            stateStatus.setText("STOPPED");
            stateStatus.setForeground(RED);
            periodTextField.setEnabled(false);
	    * /

            for (int sw = 0; sw < VacSysState.NUM_SWITCHES; sw++) {
                switchStatus[sw].setEnabled(false);
                switchOffRB[sw].setEnabled(false);
                switchOnRB[sw].setEnabled(false);
            }
            for (int cond = 0; cond < VacSysState.NUM_CONDITIONS; cond++) {
                conditionStatus[cond].setEnabled(false);
                conditionResetBtn[cond].setEnabled(false);
            }
        } 
    }
*/
    private static final long serialVersionUID = 1L;
}
