package org.lsst.ccs.subsystem.pathfinder.data;

import java.util.Arrays;

import org.lsst.ccs.subsystem.pathfinder.constants.ConditionState;
import org.lsst.ccs.subsystem.pathfinder.constants.LatchState;
import org.lsst.ccs.subsystem.pathfinder.constants.PLCState;
import org.lsst.ccs.subsystem.pathfinder.constants.DeviceState;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchState;
import org.lsst.ccs.subsystem.pathfinder.constants.SwitchEnable;
import org.lsst.ccs.subsystem.pathfinder.constants.VacuumState;
import org.lsst.ccs.subsystem.pathfinder.constants.FanState;
import org.lsst.ccs.subsystem.pathfinder.constants.UtilTrunkFans;
import java.io.Serializable;

/**
 *  Contains the state of the vacuum subsystem
 *
 *  @author Owen Saxton
 */
public class VacSysState implements Serializable {
    
    public VacSysState() {
	Arrays.fill(fanState, FanState.OFFLINE);
	Arrays.fill(fanBaseState, FanState.SPEED);
    }

    public static final String KEY = "PathFinderState";
    public static final int
        SW_CRYO_TURBO     = 0,
        SW_HX_TURBO       = 1,
        SW_HX_ION_PUMP1   = 2,
        SW_HX_VALVE       = 3,
        SW_OR_ION_PUMP    = 4,
        NUM_SWITCHES      = 5;

    public static final int
        LATCH_CR_VACUUM   = 0,
        LATCH_CR_GATE_NFC = 1,
        LATCH_CR_GATE_AO  = 2,
        LATCH_CR_PUMP     = 3,
        LATCH_HX_VACUUM   = 4,
        LATCH_HX_GATE_NFC = 5,
        LATCH_HX_GATE_AO  = 6,
        LATCH_HX_PUMP     = 7,
        NUM_LATCHES       = 8;

    public static final int
        COND_CR_FORELINE_VAC = 0,
        COND_CR_TRB_PUMP_OFF = 1,
        COND_CR_TRB_PRESS_10 = 2,
        COND_CR_VACUUM_01    = 3,
        COND_CR_VACUUM_001   = 4,
        COND_HX_FORELINE_VAC = 5,
        COND_HX_TURBO_OFF    = 6,
        COND_HX_TRB_PRESS_10 = 7,
        COND_HX_VACUUM_01    = 8,
        COND_HX_VACUUM_001   = 9,
        COND_OR_FPP_VALVE    = 10,
        COND_OR_FH_VALVE     = 11,
        COND_OR_L3H_VALVE    = 12,
        COND_OR_L3_VALVE     = 13,
        NUM_CONDITIONS       = 14;

    /*
    public static final int
        COND_CR_VACUUM   = 0,
        COND_CR_GATE_NFC = 1,
        COND_CR_GATE_AO  = 2,
        COND_CR_PUMP     = 3,
        COND_HX_VACUUM   = 4,
        COND_HX_GATE_NFC = 5,
        COND_HX_GATE_AO  = 6,
        COND_HX_PUMP     = 7,
        NUM_CONDITIONS   = 8;
    */

    private int tickMillis;           // Data status update period (ms)
    private VacuumState vacuumState = VacuumState.UNKNOWN;  // The vacuum state
    private PLCState plcState = PLCState.ALIVE;  // The PLC state
    private int switchMask;           // Bit mask of defined switches
    private int conditionMask;        // Bit mask of defined conditions
    private final SwitchState[] switchState = new SwitchState[NUM_SWITCHES];  // Switch state array
    private final SwitchEnable[] switchEnable = new SwitchEnable[NUM_SWITCHES];  // Switch enable array
    private final DeviceState[] deviceState = new DeviceState[NUM_SWITCHES];  // Extra device state array
    private final ConditionState[] conditionState = new ConditionState[NUM_CONDITIONS];

    private final FanState[] fanState = new FanState[UtilTrunkFans.NUM_FANS];
    private final FanState[] fanBaseState = new FanState[UtilTrunkFans.NUM_FANS];
    private final double[] fanSpeed = new double[UtilTrunkFans.NUM_FANS];
    private final double[] deltaTemp = new double[UtilTrunkFans.NUM_FANS];


    private int latchMask;            // Bit mask of defined latched conditions
    private final LatchState[] latchState = new LatchState[NUM_LATCHES];  // Latched PLC conditions array


    
    public void setTickMillis(int value) {
        tickMillis = value;
    }

    public int getTickMillis() {
        return tickMillis;
    }

    public void setFanState(int fan, FanState state) {
        fanState[fan] = state;
    }

    public FanState getFanState(int fan) {
        return fanState[fan];
    }

    public void setFanBaseState(int fan, FanState state) {
        fanBaseState[fan] = state;
    }

    public FanState getFanBaseState(int fan) {
        return fanBaseState[fan];
    }

    public void setFanSpeed(int fan, double speed) {
        fanSpeed[fan] = speed;
    }

    public double getFanSpeed(int fan) {
        return fanSpeed[fan];
    }

    public void setDeltaTemp(int fan, double temp) {
        deltaTemp[fan] = temp;
    }

    public double getDeltaTemp(int fan) {
        return deltaTemp[fan];
    }





    public void setVacuumState(VacuumState value) {
        vacuumState = value;
    }

    public VacuumState getVacuumState() {
        return vacuumState;
    }

    public void setPlcState(PLCState value) {
        plcState = value;
    }

    public PLCState getPlcState() {
        return plcState;
    }

    public void addSwitch(int sw) {
        switchMask |= (1 << sw);
    }

    public boolean hasSwitch(int sw) {
        return (switchMask & (1 << sw)) != 0;
    }

    public void setSwitchState(int sw, SwitchState state) {
        switchState[sw] = state;
    }

    public SwitchState getSwitchState(int sw) {
        return switchState[sw];
    }

    public void setSwitchEnable(int sw, SwitchEnable enable) {
        switchEnable[sw] = enable;
    }

    public SwitchEnable getSwitchEnable(int sw) {
        return switchEnable[sw];
    }

    public void addLatch(int latch) {
        latchMask |= (1 << latch);
    }

    public boolean hasLatch(int latch) {
        return (latchMask & (1 << latch)) != 0;
    }

    public void setLatch(int latch, LatchState state) {
        latchState[latch] = state;
    }

    public LatchState getLatch(int latch) {
        return latchState[latch];
    }

    public void setDeviceState(int sw, DeviceState state) {
        deviceState[sw] = state;
    }

    public DeviceState getDeviceState(int sw) {
        return deviceState[sw];
    }

    public void addCondition(int cond) {
        conditionMask |= (1 << cond);
    }

    public boolean hasCondition(int cond) {
        return (conditionMask & (1 << cond)) != 0;
    }

    public void setCondition(int cond, ConditionState state) {
        conditionState[cond] = state;
    }

    public ConditionState getCondition(int cond) {
        return conditionState[cond];
    }

    private static final long serialVersionUID = 2081368367194546251L;
}
