package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.sal.camera.states.CCSCommandStateEvent.CCSCommandState;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;

/**
 *
 * @author tonyj
 */
public class OCSCommandExecutor {

    private static final Logger LOG = Logger.getLogger(OCSBridge.class.getName());
    private final OCSBridge bridge;

    OCSCommandExecutor(OCSBridge bridge) {
        this.bridge = bridge;
    }

    void executeCommand(OCSExecutor command) {
        State<CCSCommandState> commandState = bridge.getCommandState();
        if (!commandState.isInState(CCSCommandState.IDLE)) {
            rejectCommand(command, "Command state not idle");
        } else {
            try {
                Duration timeout = command.testPreconditions();
                commandState.setState(CCSCommandState.BUSY);
                if (!timeout.isZero()) {
                    acknowledgeCommand(command, timeout);
                }
                command.execute();
                reportComplete(command);
            } catch (PreconditionsNotMet ex) {
                rejectCommand(command, ex.getMessage());
            } catch (Exception ex) {
                reportError(command, ex);
            } finally {
                commandState.setState(CCSCommandState.IDLE);
            }
        }
    }

    protected void rejectCommand(OCSExecutor command, String reason) {
        LOG.log(Level.INFO, "Reject command: {0} because {1}", new Object[]{command, reason});
    }

    protected void acknowledgeCommand(OCSExecutor command, Duration timeout) {
        LOG.log(Level.INFO, "Acknowledge command: {0} timeout {1}", new Object[]{command, timeout});
    }

    protected void reportError(OCSExecutor command, Exception ex) {
        LOG.log(Level.WARNING, "Command failed: " + command, ex);
    }

    protected void reportComplete(OCSExecutor command) {
        LOG.log(Level.INFO, "Command complete: {0}", command);
    }

    /**
     * Send event, designed to be overridden in subclasses.
     */
    void sendEvent(CameraEvent event) {
        LOG.log(Level.INFO, "OCS Event -> {0}", event.toString());
    }
    
    void sendTelemetry(CameraTelemetry telemetry) {
        LOG.log(Level.INFO, "OCS Telemetry -> {0}", telemetry.toString());
    }

    /**
     * A base class for all OCS command executors
     *
     * @author tonyj
     */
    public static abstract class OCSExecutor {

        private final CameraCommand cmd;
        
        OCSExecutor(CameraCommand command) {
            this.cmd = command;
        }

        /**
         * Check preconditions, and estimate the command duration.
         *
         * @throws PreconditionsNotMet If the preconditions are not met
         * @return The estimated duration of the command (can be ZERO)
         */
        abstract Duration testPreconditions() throws PreconditionsNotMet;

        /**
         * Actually perform the command
         */
        abstract void execute() throws Exception;

//        public int getCmdId() {
//            return cmd.getCmdId();
//        }

        public CameraCommand getCommand() {
            return cmd;
        }

        @Override
        public String toString() {
            return cmd.toString();
        }

    }

    static class PreconditionsNotMet extends Exception {

        private static final long serialVersionUID = 2388630285363617708L;

        public PreconditionsNotMet(String reason) {
            super(reason);
        }
    }
}
