package org.lsst.ccs.subsystem.ocsbridge.util;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.subsystem.ocsbridge.sim.FilterChanger.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.sal.camera.CameraStateChangeEvent;
import org.lsst.sal.camera.states.CCSCommandStateEvent;
import org.lsst.sal.camera.states.CCSCommandStateEvent.CCSCommandState;
import org.lsst.sal.camera.states.CalibrationDetailedStateEvent;
import org.lsst.sal.camera.states.FilterChangerDetailedStateEvent;
import org.lsst.sal.camera.states.ImageReadinessDetailedStateEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.RaftsDetailedStateEvent;
import org.lsst.sal.camera.states.ShutterDetailedStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent;

/**
 * The OCSStateEventConverter is used to convert state events as sent by CCS, 
 * into the equivalent CameraStateChangeEvent.
 * @author Farrukh Azfar
 *
 */
public class OCSStateEventConverter {

    private final Map< Class<? extends Enum>, Class< ? extends Enum>> mapEnums;
    private final Map< Class<? extends Enum>, Class< ? extends CameraStateChangeEvent>> mapEnumToClass;

    public OCSStateEventConverter() {
        this.mapEnums = new HashMap<>();
        this.mapEnumToClass = new HashMap<>();

        mapEnums.put(SummaryStateEvent.SummaryState.class, SummaryStateEvent.SummaryState.class);
        mapEnums.put(Shutter.ShutterState.class, ShutterDetailedStateEvent.ShutterState.class);
        mapEnums.put(FilterState.class, FilterChangerDetailedStateEvent.FilterChangerState.class);
        mapEnums.put(CCSCommandState.class, CCSCommandStateEvent.CCSCommandState.class);
        mapEnums.put(MCM.TakeImageReadinessState.class, ImageReadinessDetailedStateEvent.ImageReadinessState.class);
        mapEnums.put(FocalPlane.RaftsState.class, RaftsDetailedStateEvent.RaftsState.class);
        mapEnums.put(MCM.CalibrationState.class, CalibrationDetailedStateEvent.CalibrationState.class);
        mapEnums.put(OfflineState.class, OfflineDetailedStateEvent.OfflineState.class);

        mapEnumToClass.put(SummaryStateEvent.SummaryState.class, SummaryStateEvent.class);
        mapEnumToClass.put(Shutter.ShutterState.class, ShutterDetailedStateEvent.class);
        mapEnumToClass.put(FilterState.class, FilterChangerDetailedStateEvent.class);
        mapEnumToClass.put(CCSCommandState.class, CCSCommandStateEvent.class);
        mapEnumToClass.put(MCM.TakeImageReadinessState.class, ImageReadinessDetailedStateEvent.class);
        mapEnumToClass.put(FocalPlane.RaftsState.class, RaftsDetailedStateEvent.class);
        mapEnumToClass.put(MCM.CalibrationState.class, CalibrationDetailedStateEvent.class);
        mapEnumToClass.put(OfflineState.class, OfflineDetailedStateEvent.class);
    }

    /**
     * Convert a CCS Enum into the equivalent OCS Enum. This class takes
     * into account that the order of the declarations between the two Enums
     * (and thus the ordinals) may not be the same.
     * @param ccsStateEvent The input Enum to convert
     * @return The resulting Enum, or <code>null</code> if the input enum is not known.
     * @throws RuntimeException if the Enum cannot be successfully mapped.
     */
    private Enum getEnum(Enum ccsStateEvent) {

        Class<? extends Enum> ocsEvent = mapEnums.get(ccsStateEvent.getClass());
        if (ocsEvent == null) return null;
        
        String name = ccsStateEvent.name();
        for (Enum ocsEnum : ocsEvent.getEnumConstants()) {
            
            if (ocsEnum.name().equals(name)) {
                return ocsEnum;
            }
        }

        throw new RuntimeException("Could not find equivalent Enum for input: "+ccsStateEvent);
    }

    /** 
     * Convert a CCS Enum into the equivalent CameraStateChangeEvent
     * 
     * @param when When the state change occurred
     * @param ccsEnum The input CCS Enum
     * @param priority The priority to assign to the returned CameraStateChangeEvent.
     * @return The generated CameraStateChangeEvent, or <code>null</code> if the input Enum is unknown.
     * @throws RuntimeException If an error occurs during the conversion process.
     */
    public CameraStateChangeEvent convert(CCSTimeStamp when, Enum ccsEnum, int priority) throws RuntimeException {

        Enum enumForOCS = getEnum(ccsEnum);
        Class<? extends CameraStateChangeEvent> ocsStateChangeEvent = mapEnumToClass.get(ccsEnum.getClass());        

        if (enumForOCS == null || ocsStateChangeEvent==null) return null;
        
        try {
            // TODO: We need to include the timestamp in the OCS Event
            Constructor<? extends CameraStateChangeEvent> constructor = ocsStateChangeEvent.getConstructor(int.class, enumForOCS.getClass());
            return constructor.newInstance(priority, enumForOCS);
        } catch (ReflectiveOperationException x) {
            throw new RuntimeException("Error constructing CameraStateChangeEvent for ccsEnum "+ccsEnum,x);  
        }
    }

}
