package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import org.lsst.ccs.imagenaming.ImageName;
import static org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.CLEAR_TIME;
import static org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.DISCARD_TIME;
import static org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.READOUT_TIME;
import org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 *
 * @author tonyj
 */
public class FocalPlaneSimulation implements FocalPlaneInterface {

    private final State raftsState;
    private final CCS ccs;

    public FocalPlaneSimulation(CCS ccs, State raftsState) {
        this.raftsState = raftsState;
        this.ccs = ccs;
    }

    @Override
    public void clear(int nClears) {
        raftsState.checkState(RaftsState.QUIESCENT, RaftsState.NEEDS_CLEAR);
        raftsState.setState(FocalPlane.RaftsState.CLEARING);
        ccs.schedule(CLEAR_TIME.multipliedBy(nClears), () -> {
            raftsState.setState(FocalPlane.RaftsState.QUIESCENT);
        });
    }
    
    @Override
    public void discardRows(int nRows) throws ExecutionException {
        raftsState.checkState(RaftsState.INTEGRATING);
        raftsState.setState(FocalPlane.RaftsState.DISCARDING);
        ccs.schedule(DISCARD_TIME.multipliedBy(nRows), () -> {
            raftsState.setState(FocalPlane.RaftsState.INTEGRATING);
        });
    }

    @Override
    public void endIntegration(boolean readout, Duration exposure) throws ExecutionException {
        raftsState.checkState(RaftsState.INTEGRATING);
        if (readout) {
            raftsState.setState(FocalPlane.RaftsState.READING_OUT);
            ccs.schedule(READOUT_TIME, () -> {
                raftsState.setState(FocalPlane.RaftsState.QUIESCENT);
            });
        } else {
            raftsState.setState(FocalPlane.RaftsState.NEEDS_CLEAR);
        }
    }

    @Override
    public void start(String configName) throws ExecutionException {
        // NOOP
    }

    @Override
    public void startIntegration(ImageName imageName, Map<String, String> parsedKeyValueData, LocationSet locations, String annotation) throws ExecutionException {
        raftsState.checkState(RaftsState.QUIESCENT);
        raftsState.setState(FocalPlane.RaftsState.INTEGRATING);
    }

    @Override
    public void play(String playlist, boolean repeat) {
        // Nothing to do at the moment
    }

    @Override
    public void definePlaylist(String playlist, String folder, String... images) {
        // Nothing to do at the moment
    }

}
