package org.lsst.ccs.subsystem.ocsbridge;

import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.subsystem.ocsbridge.OCSBridgeConfig.Device;
import org.lsst.ccs.subsystem.ocsbridge.util.GenericConverter;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription;
import org.lsst.sal.camera.CameraEvent;

/**
 * Handler receipt of configuration from CCS, conversion into equivalent SAL
 * messages, and send to SAL.
 *
 * @author tonyj
 */
public abstract class ConfigurationSender {


    private static final Logger LOG = Logger.getLogger(ConfigurationSender.class.getName());
    private OCSEventSender sender;

    static ConfigurationSender create(Device device, OCSEventSender sender) {
        switch (device) {
            case COMCAM:
                return new ComCamConfigurationSender(sender);
            case AUXTEL:
                return new AuxTelConfigurationSender(sender);
            case CAMERA:
                return new CameraConfigurationSender(sender);
            default:
                throw new IllegalArgumentException("Unsupported device: " + device);
        }

    }

    protected ConfigurationSender(OCSEventSender sender) {
        this.sender = sender;
    }

    abstract GenericConverter getConverter();

    public void send(Map<String, SALClassDescription> salClassMapInfo, StatusConfigurationInfo data) {
        try {
            List<CameraEvent> converted = getConverter().settingsAppliedEventConverter(salClassMapInfo, data);
            if (converted.isEmpty()) {
                LOG.log(Level.INFO, "No converted configuration {0}", data.getOriginAgentInfo().getName());
            } else {
                for (CameraEvent t : converted) {
                    t = applyAfterBurner(t);
                    LOG.log(Level.INFO, "Sending {0} {1}", new Object[]{data.getOriginAgentInfo().getName(), t});
                    sender.sendEvent(t);
                }
            }
        } catch (ReflectiveOperationException ex) {
            LOG.log(Level.WARNING, String.format("Problem converting configuration subsytem: %s key: %s", data.getOriginAgentInfo().getName(), data.getConfigurationInfo()));
        }
    }

    /**
     * Allow subclasses to modify sent events if necessary
     * @param t The event to be sent.
     * @return The input event or modified event
     */
    protected CameraEvent applyAfterBurner(CameraEvent t) {
        return t;
    }
    
    void setSender(OCSCommandExecutor ocs) {
        this.sender = ocs;
    }

    abstract String getConfigurationEvents();
}
