package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Very simple shutter simulation
 *
 * @author tonyj
 */
public class Shutter {

    /**
     * Time needed to get the shutter ready
     */
    static final Duration PREP_TIME = Duration.ofMillis(150);
    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME = Duration.ofMillis(4000);

    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME_FOR_SEVERAL_IMAGES = Duration.ofMillis(10000);

    /**
     * Time needed to move a shutter blade
     */
    static final Duration MOVE_TIME = Duration.ofMillis(980);

    public boolean imageSequence = false;
    private final MCMConfig config;

    public enum ShutterReadinessState {

        NOT_READY, READY, GETTING_READY
    };

    public enum ShutterState {

        CLOSED, OPENING, OPEN, CLOSING
    };

    private final State shutterReadinessState;
    private final State shutterState;
    private ScheduledFuture<?> notReadyFuture;
    private ShutterInterface shutter;

    private final CCS ccs;

    Shutter(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        shutterReadinessState = new State(ShutterReadinessState.NOT_READY);
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        ccs.getAggregateStatus().add(now, shutterReadinessState);
        shutterState = new State(ShutterState.CLOSED);
        ccs.getAggregateStatus().add(now, shutterState);
        // When the shutter is closed, we only keep the motors powered up for a limited time
        shutterState.addStateChangeListener((when, state, oldState) -> {
            if (state == ShutterState.CLOSED) {
                scheduleNotReady();
            } else {
                cancelNotReady();
            }
        });

        // By default assumes simulation
        shutter = new ShutterSimulation(ccs, shutterState);
    }

    public void startImageSequence() {

        imageSequence = true;
    }

    public void endImageSequence() {

        imageSequence = false;
    }

    private void scheduleNotReady() {

        Duration ready_time;
        if (imageSequence == true) {
            ready_time = READY_TIME_FOR_SEVERAL_IMAGES;
        } else {
            ready_time = READY_TIME;

        }

        notReadyFuture = ccs.schedule(ready_time, () -> {
            shutterReadinessState.setState(Shutter.ShutterReadinessState.NOT_READY);
        });
    }

    private void cancelNotReady() {
        if (notReadyFuture != null) {
            notReadyFuture.cancel(false);
        }
    }

    void prepare() {
        cancelNotReady();
        shutterReadinessState.setState(ShutterReadinessState.GETTING_READY);
        ccs.schedule(PREP_TIME, () -> {
            shutterReadinessState.setState(ShutterReadinessState.READY);
            scheduleNotReady();
        });
    }

    void expose(Duration exposureTime) throws ExecutionException {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutter.expose(exposureTime);
    }

    void start(String configName) throws ExecutionException {
        shutter.start(configName);
    }

    void open() throws ExecutionException {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutter.open();
    }

    void close() throws ExecutionException {
        if (!shutterState.isInState(ShutterState.CLOSED)) {
            shutter.close();
        }
    }

    void registerMCMSubsystem(MCMSubsystem mcm) {
        shutter = new ShutterSubsystemLayer(mcm, ccs, config);
    }
}
