package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.IOException;
import java.io.Serializable;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.UnauthorizedLevelException;
import org.lsst.ccs.services.UnauthorizedLockException;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;

/**
 * A base class for subsystems which are actively controlled by the MCM
 *
 * @author tonyj
 */
class ControlledSubsystem {

    private static final Logger LOG = Logger.getLogger(ControlledSubsystem.class.getName());

    protected final MCMConfig config;
    protected final MCMCommandSender commandSender;
    protected final CCS ccs;

    public ControlledSubsystem(Subsystem mcm, String targetSubsystem, CCS ccs, MCMConfig config) {
        this.config = config;
        this.commandSender = new MCMCommandSender(targetSubsystem, mcm.getMessagingAccess(), mcm.getAgentService(AgentLockService.class));
        this.ccs = ccs;

        mcm.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (agent.getName().equals(targetSubsystem)) {
                        // TODO: It would be better to get the locally cached state from the core, rather than get it from the
                        // remote subsystem to avoid need to deal with exceptions
                        try {
                            StateBundle initialState = commandSender.sendCommand(StateBundle.class, "getState");
                            // TODO: Perhaps better to call state change after onConnect?
                            onConnect(agent, initialState);
                        } catch (ExecutionException x) {
                            LOG.log(Level.SEVERE, "Failed to get initial state from subsystem "+targetSubsystem, x);
                        }
                    }
                }
            }

            @Override
            public void disconnected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (agent.getName().equals(targetSubsystem)) {
                        onDisconnect(agent);
                    }
                }
            }
        });

        Predicate<BusMessage<? extends Serializable, ?>> targetSubsystemFilter = BusMessageFilterFactory.messageOrigin(targetSubsystem);
        mcm.getMessagingAccess().addStatusMessageListener((msg) -> {
            if (msg instanceof StatusStateChangeNotification) {
                onStateChange((StatusStateChangeNotification) msg);
            } else if (msg instanceof StatusHeartBeat) {
                // Ignored
            } else {
                onEvent(msg);
            }
        }, targetSubsystemFilter);
    }

    public void start(String configName) throws ExecutionException {
        try {
            commandSender.lock(10);
            commandSender.sendCommand("publishConfigurationInfo");
        } catch (IOException | UnauthorizedLevelException | UnauthorizedLockException x) {
            throw new ExecutionException("Failed to lock subsystem " + config.getShutterSubsystemName(), x);
        }
    }

    /**
     * This method is called whenever the target subsystems connects, including
     * initially when the the MCM itself starts up if the target subsystem is
     * already running.
     *
     * @param agent
     */
    protected void onConnect(AgentInfo agent, StateBundle initialState) {

    }

    protected void onDisconnect(AgentInfo agent) {
        //TODO: If a subsystem goes away while it is locked that should cause the MCM to enter fault state.
    }

    protected void onStateChange(StatusStateChangeNotification statusChange) {
        //TODO: If a controlled subsystem goes into fault state while it is locked, the MCM should go into fault state
    }

    protected void onEvent(StatusMessage msg) {
    }
}
