package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 *
 * @author tonyj
 */
public class SALClassDescription {

    private final String className;
    private final Map<String, SALVariable> variables = new TreeMap<>();
    private final int level;
    private final String category;

    /**
     * Create new SALClassDescription
     * @param className The name of the SAL class
     * @param level The level at which the conversion was done. 
     */
    SALClassDescription(String className, int level, String category) {
        this.className = className;
        this.level = level;
        this.category = category;
    }
    
    SALVariable getVariable(String variableName) {
        return variables.get(variableName);
    }
    
    void add(SALVariable var) {
        variables.put(var.getVariableName(), var);
    }

    public String getClassName() {
        return className;
    }

    public List<SALVariable> getVariables() {
        return new ArrayList(variables.values());
    }

    public int getLevel() {
        return level;
    }

    public String getCategory() {
        return category;
    }

    @Override
    public String toString() {
        return "SALClassDescription{" + "className=" + className + ", level=" + level + ", category=" + category +
               getVariables().stream().map(v->v.toString()).collect(Collectors.joining("\n\t","\n\t","\n"))        
               + '}';
    }

    /** 
     * Rename the given variable
     * @param variable The variable to rename
     * @param newName The new name
     */
    void renameVariable(SALVariable variable, String newName) {
        String oldName = variable.getVariableName();
        variables.remove(oldName);
        variable.setName(newName);
        variables.put(newName, variable);
    }
     
    public static abstract class SALVariable {

        private String variableName;
        private final String type;
        private final String units;
        private final String description;

        public SALVariable(String variableName, String type, String units, String description) {
            this.variableName = variableName;
            this.type = type;
            this.units = units;
            this.description = description;
        }

        public String getVariableName() {
            
            return variableName;
        }

        public String getType() {
            return type;
        }

        public String getUnits() {
            return units;
        }

        public String getDescription() {
            return description;
        }
        
        public int getCount() {
            return 1;
        }

        @Override
        public String toString() {
            return "variableName=" + variableName + ", type=" + type + ", units=" + units + ", description=\"" + description+"\"";
        }

        private void setName(String newName) {
            this.variableName = newName;
        }
    }

    /**
     * A SAL variable which does not match a pattern
     */
    public static class SimpleSALVariable extends SALVariable {

        private final BusVariable busVariable;

        public SimpleSALVariable(String variableName, String type, String units, String description, BusVariable busVariable) {
            super(variableName, type, units, description);
            this.busVariable = busVariable;
        }

        public BusVariable getBusVariable() {
            return busVariable;
        }

        @Override
        public String toString() {
            return "SimpleSALVariable{" + super.toString() + " busVariable=" + busVariable +  '}';
        }
    }

    /**
     * A SAL variable that has been fabricated to keep track of locations for variables which match patterns. 
     * It does not have a corresponding bus variable
     */
    public static class LocationVariable extends SALVariable {


        public LocationVariable(String variableName, String description) {
            super(variableName, "String", "unitless", description);
        }


        @Override
        public String toString() {
            return "LocationVariable{" + super.toString() + '}';
        }
    }
    
    public static class PatternMatchedSALVariable extends SALVariable {

        private final String patternName;
        private final String pattern;
        private final List<BusVariable> busVariables = new ArrayList<>();
        private final LocationVariable locationVariable;

        public PatternMatchedSALVariable(String patternName, String pattern, LocationVariable location, String variableName, String type, String units, String description) {
            super(variableName, type, units, description);
            this.patternName = patternName;
            this.pattern = pattern;
            this.locationVariable = location;
        }
        
    
        
        void addBusVariable(BusVariable var) {
            busVariables.add(var);
        }

        public String getPatternName() {
            return patternName;
        }

        public String getPattern() {
            return pattern;
        }

        public List<BusVariable> getBusVariables() {
            return busVariables;
        }

        public LocationVariable getLocationVariable() {
            return locationVariable;
        }
        
        
        
        @Override
        public int getCount() {
            return busVariables.size();
        }

        @Override
        public String toString() {
            return "PatternMatchedSALVariable{"  + super.toString() + " patternName=" + patternName + ", pattern=" + pattern + ", locationVariable=" + locationVariable +
                  busVariables.stream().map(v->v.toString()).collect(Collectors.joining("\n\t\t","\n\t\t","\n\t")) + 
            '}';
        }
    }

    public static class BusVariable {
        private final String pathAndNameOnBuses;

        public BusVariable(String pathAndNameOnBuses) {
            this.pathAndNameOnBuses = pathAndNameOnBuses;
        }

        public String getPathAndNameOnBuses() {
            return pathAndNameOnBuses;
        }

        @Override
        public String toString() {
            return "BusVariable{" + "pathAndNameOnBuses=" + pathAndNameOnBuses + '}';
        }
        
    }
    
    
    
    // Get count for shared location - experimental - Farrukh 25 September 2021
    // we receive a variable name (String) and we want to know the array size
  
    int getSharedLocationCount(SALVariable var) {

        // we extract the associate SALvariable
        SALVariable sv = variables.get(var.getVariableName());
        int count = 0;

        // only enter the if block if it is a PatternMatchedSALVariable
        if (sv instanceof PatternMatchedSALVariable) {
            // create a list which will contain all the PatternMatchedVariables in this SALClassDescription 
            List<PatternMatchedSALVariable> pmsvList = new ArrayList<>();
            PatternMatchedSALVariable pmv = (PatternMatchedSALVariable) sv;
            String location = pmv.getLocationVariable().getVariableName();
            pmsvList.add(pmv);

            // loop over all SALVariables contained in this SAL class description 
            for (Map.Entry<String, SALVariable> entry : variables.entrySet()) {
                // make we select only PatternMatchedSALVariables
                if (entry.getValue() instanceof PatternMatchedSALVariable) {
                    // these cannot be the given variable its already in the list 
                    if (!pmv.equals(entry.getValue())
                            // we make sure the location variable is the same 
                            && location.equals(((PatternMatchedSALVariable) entry.getValue()).getLocationVariable().getVariableName())) {
                        // we add it to the list 
                        pmsvList.add((PatternMatchedSALVariable) entry.getValue());
                    }
                }

                // now set count to the max value 
                count = Collections.max(pmsvList.stream().map(eachpmv -> eachpmv.getCount()).collect(Collectors.toList()));
            }
            // now that we have the list of all PatternMatchedSALVariables in this SALClassDescription that share the same location
            // variable we find the one with the maximum number of associated busVariables - this will then be what the
            // array size will be set to 

        } // if the variable name passed on to us is not a PatternMatchedVariable then 
        else {
            // in which case the count is equal to 1 
            count = 1;
        }

        return count;
    }
    
    
}
