package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDefinePlaylistCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPlayCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStandbyCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;

/**
 * Run the simulated MCM as a CCS subsystem, able to send/receive commands using
 * CCS buses.
 *
 * @author tonyj
 */
public class MCMSubsystem extends Subsystem implements HasLifecycle {

    private final CCS ccs = new CCS();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private MCMConfig config;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    private MCM mcm;

    @SuppressWarnings("OverridableMethodCallInConstructor")
    public MCMSubsystem() {
        super("mcm", AgentType.MCM);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }

    @Override
    public void init() {
        agentStateService.registerState(Shutter.ShutterState.class, "Shutter state", this);
        agentStateService.registerState(MCM.TakeImageReadinessState.class, "Image readiness State", this);
        agentStateService.registerState(FilterChanger.FilterState.class, "Filter State", this);
        agentStateService.registerState(MCM.CalibrationState.class, "Calibration State", this);
        agentStateService.registerState(Shutter.ShutterReadinessState.class, "Shutter readiness State", this);
        agentStateService.registerState(FocalPlane.RaftsState.class, "Focal plane state", this);
        agentStateService.registerState(MCM.StandbyState.class, "Standby state", this);
    }
    
    @Override
    public void postStart() {

        mcm = new MCM(ccs, config, imageNameService);
        mcm.registerMCMSubsystem(this);

        if (config.hasFocalPlaneSubsystem()) {
            mcm.getFocalPlane().registerMCMSubsystem(this);
        } 
        
        if (config.getShutterSubsystemName() != null && !config.getShutterSubsystemName().trim().isEmpty()) {
            mcm.getShutter().registerMCMSubsystem(this);
        }
        
        if (config.hasFilterChanger() && config.getFilterChangerSubsystemName() != null && !config.getFilterChangerSubsystemName().trim().isEmpty()) {
            mcm.getFilterChanger().registerMCMSubsystem(this);
        }      

        ccs.addStateChangeListener((when, state, oldState) -> {
            // publish whenever state change occurs
            agentStateService.updateAgentState(when, state);
        });

        // publish initial states
        List<Enum> initialStates = new ArrayList<>();
        ccs.getAggregateStatus().getStates().forEach((state) -> {
            initialStates.add(state.getState());
        });
        agentStateService.updateAgentState(initialStates.toArray(new Enum[initialStates.size()]));

        ccs.addStatusMessageListener((StatusMessage msg) -> {
            // This should be called when a simulated subsystem generates trending/config
            // data.
            this.getMessagingAccess().sendStatusMessage(msg);
        });
        
        ccs.addEventListener((CCSEvent event) -> {
            this.publishSubsystemDataOnStatusBus(new KeyValueData("CCSEvent",event));
        });
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void initImage(@Argument(description="Time in seconds") double delta) throws Exception {
        CCSInitImageCommand initImage = new CCSInitImageCommand(delta);
        executeAndHandleResponse(initImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void startImage(boolean shutter, String sensors, String keyValueData, String annotation, double timeout) throws Exception {
        CCSStartImageCommand startImage = new CCSStartImageCommand(shutter, sensors, keyValueData, annotation, timeout);
        executeAndHandleResponse(startImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void takeImages(double expTime, int numImages, boolean shutter, String sensors, String keyValueData, String annotation) throws Exception {
        CCSTakeImagesCommand takeImages = new CCSTakeImagesCommand(expTime, numImages, shutter, sensors, keyValueData, annotation);
        executeAndHandleResponse(takeImages);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setFilter(String name) throws Exception {
        CCSSetFilterCommand setFilter = new CCSSetFilterCommand(name);
        executeAndHandleResponse(setFilter);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void initGuiders(String roiSpec) throws Exception {
        CCSInitGuidersCommand initGuiders = new CCSInitGuidersCommand(roiSpec);
        executeAndHandleResponse(initGuiders);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void endImage() throws Exception {
        CCSEndImageCommand endImage = new CCSEndImageCommand();
        executeAndHandleResponse(endImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clear(int nClears) throws Exception {
        CCSClearCommand clear = new CCSClearCommand(nClears);
        executeAndHandleResponse(clear);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void play(String playlist, @Argument(defaultValue = "false") boolean repeat) throws Exception {
        CCSPlayCommand play = new CCSPlayCommand(playlist, repeat);
        executeAndHandleResponse(play);
    }    

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void definePlaylist(String playlist, String daqFolder, String... images) throws Exception {
        CCSDefinePlaylistCommand definePlaylist = new CCSDefinePlaylistCommand(playlist, daqFolder, images);
        executeAndHandleResponse(definePlaylist);
    }  
    
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void discardRows(int nRows) throws Exception {
        CCSDiscardRowsCommand discardRows = new CCSDiscardRowsCommand(nRows);
        executeAndHandleResponse(discardRows);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void start(String configuration) throws Exception {
        CCSStartCommand start = new CCSStartCommand(configuration);
        executeAndHandleResponse(start);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void standby() throws Exception {
        CCSStandbyCommand standby = new CCSStandbyCommand();
        executeAndHandleResponse(standby);
    }
    
    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void enableCalibration() throws Exception {
        CCSEnableCalibrationCommand enableCalibration = new CCSEnableCalibrationCommand();
        executeAndHandleResponse(enableCalibration);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void disableCalibration() throws Exception {
        CCSDisableCalibrationCommand disableCalibration = new CCSDisableCalibrationCommand();
        executeAndHandleResponse(disableCalibration);
    }

    private void executeAndHandleResponse(CCSCommand command) throws Exception {
        CCSCommandResponse response = mcm.execute(command);
        CCSAckOrNack can = response.waitForAckOrNack();
        if (can.isNack()) {
            MCMSubsystem.this.sendNack(can.getReason());
        } else {
            // Note: We add 100mS to the estimate to keep ShellCommandConsole from timing
            // out, especially on zero length commands.
            MCMSubsystem.this.sendAck(can.getDuration().plus(Duration.ofMillis(1000)));
            response.waitForCompletion();
        }
    }
}
