package org.lsst.ccs.subsystem.ocsbridge.xml;

import static java.lang.System.exit;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;

/**
 * Given a data provider dictionary create a map of processed info for a given
 * subsystem
 *
 * @author farrukh, tonyj
 */
public class ProcessedInfoMaker {

    private final Set<String> patternSubstitutionRecord;
    private final Map<String, List<ProcessedInfo>> processedInfo;
    private final boolean DEBUG_PRINT = true;

    /**
     *
     * @param info The data provider dictionary
     * @param subsystemName The subsystem name
     * @param componentName The component name
     * @param level The level at which conversion should be done
     * @param mapping The regular expression map to use.
     * @param verbose If verbose output should be printed
     */
    public ProcessedInfoMaker(StatusDataProviderDictionary info, String subsystemName, String componentName, int level, Mapping mapping, String attributeType, String resultType) {

        boolean trending = "MONITORING".equals(attributeType);
        DataProviderDictionary innards = info.getDataProviderDictionary();
        List<DataProviderInfo> dpInfoList = innards.getDataProviderInfos();

        List<ProcessedInfo> preliminaryProcessedInfo = new ArrayList<ProcessedInfo>();
        
        for (DataProviderInfo dpi : dpInfoList) {
            if (dpi.getAttributeValue(DataProviderInfo.Attribute.DATA_TYPE).trim().equals(attributeType)) {
                String type = dpi.getAttributeValue(DataProviderInfo.Attribute.TYPE);
                String description = dpi.getAttributeValue(DataProviderInfo.Attribute.DESCRIPTION);
                String units = dpi.getAttributeValue(DataProviderInfo.Attribute.UNITS);
                String category = dpi.getAttributeValue(DataProviderInfo.Attribute.CONFIG_CATEGORY);
                
                // Still needed?
                if (trending) {
                    type = "double";
                    category = "";
                }

                
                String originalPathName = dpi.getAttributeValue(DataProviderInfo.Attribute.PUBLISHED_PATH);
                // should be using getPath() according to Max so be prepared 
                if(null == originalPathName) originalPathName = dpi.getFullPath().replace("main/","").trim();
                originalPathName = originalPathName.trim();
                if(DEBUG_PRINT){
                 System.out.println(" *************** (1) ********************************");
                 System.out.println(" ProcessedInfoMaker:Original Path  (from getPath() no main replacement) " + dpi.getPath());
                 System.out.println(" ProcessedInfoMaker:Original Path (from getFullPath() no main replacement) " + dpi.getFullPath());
                 System.out.println(" ProcessedInfoMaker:Published Path " + dpi.getAttributeValue(DataProviderInfo.Attribute.PUBLISHED_PATH));
                 System.out.println(" ProcessedInfoMaker:What is the final assigned originalPath Name ? " + originalPathName);
                 System.out.println(" ProcessedInfoMaker:Type " + type);
                 System.out.println(" *************** (1) ********************************");
                }
                
                boolean foundPattern = false;

                
                for(Map.Entry<Pattern, String> patternKey : mapping.getPatterns().entrySet()){
                 Matcher m = patternKey.getKey().matcher(originalPathName);
                 if(m.matches()){ 
                    String rawSALClassParameterName = patternKey.getValue() + "/" + m.group(m.groupCount());
                    String salClassParameterName = makeSALClassParameterName(rawSALClassParameterName.trim(), level);
                    String associatedClassNameM = makeClassName(level, subsystemName, componentName, resultType, rawSALClassParameterName, category);
                    if(DEBUG_PRINT){
                     System.out.println("\n *************** (2) ********************************");
                     System.out.println(" Class name generated "+ associatedClassNameM);
                     System.out.println(" Original path name " + originalPathName);
                     System.out.println(" Sal Parameter Name " + salClassParameterName);
                     System.out.println("*************** (2) ********************************");
                    }
                    
                    ProcessedInfo addThis = new ProcessedInfo(originalPathName, salClassParameterName, patternKey.getValue(), patternKey.getKey().toString(),associatedClassNameM, type, units, description, category);

                    // maybe a location string array to store location values - 
       
                         String locationString = "";
                              if (patternKey.getValue().trim().equals("ccd")) {
                                    //both manSerName and name will give us an entry into ccdLocations ... so use one of them
                                    locationString = m.group(1) + m.group(3);
                                     if(DEBUG_PRINT){
                                    System.out.println("\n *************** (3) ********************************");
                                     System.out.println(" ProcessedInfoMaker:CCD location string " + locationString);
                                     System.out.println(" ProcessedInfoMaker:PatternKey " + patternKey.getKey());
                                     System.out.println(" ProcessedInfoMaker:Original Path Name " + originalPathName);
                                     System.out.println(" *************** (3) ********************************");
                                   }
                                } else if (patternKey.getValue().trim().equals("reb")) {
                                    // if the matched pattern associated value is reb then the definition of the location value is as follows
                                    locationString = m.group(1) + m.group(2);
                                    if(DEBUG_PRINT){
                                    System.out.println("\n *************** (3) ********************************");
                                     System.out.println(" ProcessedInfoMaker:Reb location string " + locationString);
                                     System.out.println(" ProcessedInfoMaker:PatternKey " + patternKey.getKey());
                                     System.out.println(" ProcessedInfoMaker:Original Path Name " + originalPathName);
                                     System.out.println(" group 1 " + m.group(1) + " group 2 " + m.group(2));
                                     System.out.println(" *************** (3) ********************************");
                                   }
                                } else if (patternKey.getValue().trim().equals("raft")) {
                                    // if the matched pattern associated value is raft then the definition of the location value is as follows
                                    locationString = m.group(1);
                                   if(DEBUG_PRINT){
                                    System.out.println(" *************** (3) ********************************");
                                    System.out.println(" ProcessedInfoMaker:Raft location string " + locationString);
                                    System.out.println(" ProcessedInfoMaker:PatternKey " + patternKey.getKey());
                                    System.out.println(" ProcessedInfoMaker:Original Path Name " + originalPathName);
                                    System.out.println(" *************** (3) ********************************");
                                   }
                                    
                                } else if (patternKey.getValue().trim().equals("rebps")) {
                                  // if the matched pattern associated value is rebps then the definition of the location value is as follows
                                    locationString = m.group(1);
                                  if(DEBUG_PRINT){
                                    System.out.println(" *************** (3) ********************************");
                                    System.out.println(" ProcessedInfoMaker:Rebps location string " + locationString);
                                    System.out.println(" ProcessedInfoMaker:PatternKey " + patternKey.getKey());
                                    System.out.println(" ProcessedInfoMaker:Original Path Name " + originalPathName);
                                    System.out.println(" *************** (3) ********************************");
                                   }
                                } else if (patternKey.getValue().trim().equals("segment")) {
                                 // if the matched pattern associated value is segment then the definition of the location value is as follows
                                    locationString = m.group(1);
                                     if(DEBUG_PRINT){
                                    System.out.println(" *************** (3) ********************************");
                                    System.out.println(" ProcessedInfoMaker:Segment location string " + locationString);
                                    System.out.println(" ProcessedInfoMaker:PatternKey " + patternKey.getKey());
                                    System.out.println(" ProcessedInfoMaker:Original Path Name " + originalPathName);
                                    System.out.println(" *************** (3) ********************************");
                                   }
                                    
                                }
                        addThis.setLocationString(locationString);
                                   
                        preliminaryProcessedInfo.add(addThis);
                        foundPattern = true;
                       
                    break;
                } 
                    
                
             }
                
                if(!foundPattern){
                 String associatedClassName = makeClassName(level, subsystemName, componentName, resultType, originalPathName, category);
            
                 preliminaryProcessedInfo.add(new ProcessedInfo(originalPathName, makeSALClassParameterName(originalPathName, level), "NOMATCHEDPATTERN", "NOMATCHEDPATTERN", associatedClassName, type, units, description, category));
                }
            }
        }
        this.patternSubstitutionRecord = new HashSet<>();
        this.processedInfo = classNameVariableProcessor(level, subsystemName, componentName, resultType, preliminaryProcessedInfo, mapping);

        
        // Sort alphabetically
        for (Map.Entry<String, List<ProcessedInfo>> key : processedInfo.entrySet()) {
            ProcessedInfo.sortProcessedInfo(key.getValue());
        }
    }
   
    Set<String> getPatternSubstitutionRecord() {
        return patternSubstitutionRecord;
    }

    public Map<String, List<ProcessedInfo>> getProcessedInfo() {
        return processedInfo;
    }

    @Override
    public String toString() {
        List<String> result = new ArrayList<>();
        for (Map.Entry<String, List<ProcessedInfo>> e : processedInfo.entrySet()) {
            result.add("\n" + e.getKey() + "\n");
           
            for (ProcessedInfo pi : e.getValue()) {
                result.add(String.format("%-30s %-30s %-30s %5d %-12s %-12s $-%-40s", pi.getOriginalPathName(), pi.getMatchedPattern(), pi.getPatternReplacement(), pi.getCounts(), pi.getSALClassParameterType(), pi.getSALClassParameterUnits(), pi.getSALClassParameterDescription()));
                List<String> originalPathsCorrespondingToArray = pi.getOriginalPathsCorrespondingToArray();
                if (originalPathsCorrespondingToArray != null) {
                    for (String path : originalPathsCorrespondingToArray) {
                        result.add(String.format("\t%s", path));
                    }                    
                }
                List<String> locationList = pi.getLocationList();
                if (locationList != null) {
                    for (String location : locationList) {
                        result.add(String.format("\t%s", location));
                    }
                }
            }
        }
        result.add(patternSubstitutionRecord.toString());
        return result.stream().collect(Collectors.joining("\n"));
    }

   
    private Map<String, List<ProcessedInfo>> classNameVariableProcessor(int level, String subsystem, String component, String salType, List<ProcessedInfo> preliminaryProcessedInfo, Mapping mapping) {

        Map<String, List<ProcessedInfo>> classesAndVariablesMap = new LinkedHashMap<>();
        Set<String> recordPatternSubstitution = new HashSet<>();

        
        Set locationEntry = new HashSet();


// I've received a list of processedInfo one for each path name encountered. A class name, matched pattern, parameter name etc have all been added
// we now need to group them as a class so I have a double loop 
        for (ProcessedInfo inputPI1 : preliminaryProcessedInfo) { 
     
           // get the class name associated to this parameter (parameter begins at the path name we encounter in the dictionary) 
            String className = inputPI1.getAssociatedClassName();
            List<ProcessedInfo> thisClassesVariables = new ArrayList<>();

            for (ProcessedInfo inputPI2 : preliminaryProcessedInfo) {
          
            // check to see if the className associated with the parameter in the inner loop matches the same in the outer loop
            if (className.trim().equals(inputPI2.getAssociatedClassName().trim())) {
        
   
                    String salClassParameterName = inputPI2.getSALClassParameterName();

                  // if a pattern match is recorded the information goes into recordPatternSubstitution this could be changed to a better way
                   if(!inputPI2.getPatternReplacement().contains("NOMATCHEDPATTERN")){
                    recordPatternSubstitution.add(inputPI2.getAssociatedClassName() + "::"+inputPI2.getSALClassParameterName()+"::"+inputPI2.getPatternReplacement());
                   }
                    if (salClassParameterName.isEmpty()) {
                        String tmp = "";
                        String[] tmpArr = className.split("_");

                        for (int i = 1; i < className.split("_").length; i++) {
                            tmp = tmp + tmpArr[i];
                        }

                        tmp = tmp.trim();
                        if(DEBUG_PRINT) {                                 
                           System.out.println("******************(4)**************");                     
                           System.out.println(" ProcessedInfoMaker:Class Name " +inputPI2.getAssociatedClassName());
                           System.out.println(" ProcessedInfoMaker:Parameter Name " +inputPI2.getSALClassParameterName());
                           System.out.println(" ProcessedInfoMaker:Parameter Type " +inputPI2.getSALClassParameterType());
                           System.out.println(" ProcessedInfoMaker:Path from Dictionary " + inputPI2.getOriginalPathName());
                           System.out.println(" ProcessedInfoMaker:Location String " +inputPI2.getLocationString());
                           System.out.println(" ProcessedInfoMaker:Parameter Description " + inputPI2.getSALClassParameterDescription());
                           System.out.println("******************(4)**************");                             

                        }                     
                        salClassParameterName = tmp.substring(0, 1).toLowerCase() + tmp.substring(1);
                    }
                    // pattern matched or unmatched parameter this is added into a list   
                    thisClassesVariables.add(inputPI2);
                }

            }

            // and then the classes and variables map gets the class name and all associated parameters. Those that have been matched
            // have not yet been grouped into arrays. 
            classesAndVariablesMap.put(className, thisClassesVariables);
        }

        
        // created the final map containing a string which is a class name and a list of processedInfo that is every parameter in the class
        // this will be filled in loop that follows
        Map<String, List<ProcessedInfo>> finalReprocessedClassInfo = new LinkedHashMap<>();

        // loop over the classes and variables map 
        for (Map.Entry<String, List<ProcessedInfo>> mykey : classesAndVariablesMap.entrySet()) {

            // get the list containing all the processed info for this class 
            List<ProcessedInfo> info = mykey.getValue();
            //list that contains all the like-named parameters grouped into arrays 
            List<ProcessedInfo> reprocessedInfo = new ArrayList<>();
            // a record of parameter names that have already been added to an array for this class  
            List<String> alreadyAdded = new ArrayList<>();

            String salClassParameterName;
            String variableType;
            String variableUnits;
            String variableDescription;
            String piCategory;
            
  
            // loop over this classes processed info 
            for (int i = 0; i < info.size(); i++) {

               // we're going to detect arrays based on repeated parameter names for each classes the associated boolean and array size is below
                boolean isArray = false;

                int arrayCount = 0;

                salClassParameterName = info.get(i).getSALClassParameterName();

                variableType = info.get(i).getSALClassParameterType();
                variableUnits = info.get(i).getSALClassParameterUnits();
                variableDescription = info.get(i).getSALClassParameterDescription();
                piCategory = info.get(i).getCategory();
                
                // this array list contains the old path names to which something has been matched
                List<String> matchedOldPathNames = new ArrayList<>();
                List<String> locationValueList = new ArrayList<>();
                matchedOldPathNames.add(info.get(i).getOriginalPathName());
                if(!(info.get(i).getLocationString()==null)){
                  locationValueList.add(info.get(i).getLocationString());
                }
                //inner loop over all the processed info counting the number of occurances of this SALClass parameter name 
                for (int j = i + 1; j < info.size(); j++) {

                    // here we are comparing the names of two parameters 
                    if (salClassParameterName.equals(info.get(j).getSALClassParameterName())) {
                        arrayCount++;
                        isArray = true;
                        matchedOldPathNames.add(info.get(j).getOriginalPathName());
                        if(!info.get(j).getLocationString().isEmpty()){
                         locationValueList.add(info.get(j).getLocationString());
                        }
                    }

                }

                // Now we've determined that this is an array - we create the associated array parameter record (processedInfo)for the SAL class 
                if (isArray && !alreadyAdded.contains(salClassParameterName)) {

                    // must modify processedInfo to carry the array size count 
                    int countThatMustBeProvided = arrayCount;
                    variableType = "list<" + variableType + ">";

                    ProcessedInfo addMe = new ProcessedInfo(info.get(i).getOriginalPathName(), salClassParameterName, info.get(i).getPatternReplacement(), info.get(i).getMatchedPattern(),info.get(i).getAssociatedClassName(),variableType, variableUnits, variableDescription, piCategory);
                    addMe.setCounts(arrayCount + 1);
                    addMe.setLocationString(info.get(i).getLocationString());
                    addMe.setOriginalPathsCorrespondingToArray(matchedOldPathNames);
                    reprocessedInfo.add(addMe);
                    alreadyAdded.add(salClassParameterName);

                    
                    
                    // now we create the location parameter 
                    for (String it : recordPatternSubstitution) {                   

                        // So here we are looking in the list recordPatternSubstitution which contains a concatenate of the class name the SAL class parameter name and the 
                        // matched pattern 
                        
                                        
                        if (it.split("::")[0].trim().equals(mykey.getKey().trim()) && it.split("::")[1].trim().equals(salClassParameterName.trim())) {

                            // first we name the location parameter with its default name ie "Location"
                            String location = "Location";
                            // now if the class parameter name (array element [1] below) contains the replacement pattern (array element [2]) -so I am guessing this is like reb rebps etc
                            // then we pop change "Location" into rebLocation etc (ie add the replacement pattern name before the location) 
                            if (it.split("::")[1].trim().contains(it.split("::")[2].trim())) {
                                location = it.split("::")[2].trim() + location;
                            }
                            if (!alreadyAdded.contains(location)) {
                                String localVarName = it.split("::")[2].trim() + "Location".trim();
                                String localVarType = "string";
                                String localVarUnits = "unitless";
                               
                                String localVarDescr = it.split("::")[2].substring(0, 1).toUpperCase() + it.split("::")[2].substring(1).toLowerCase() + " Location";
                                
                                ProcessedInfo addMeLocation = new ProcessedInfo(location, location, info.get(i).getPatternReplacement(), info.get(i).getMatchedPattern(), info.get(i).getAssociatedClassName(),localVarType, localVarUnits, localVarDescr, piCategory);
                                addMeLocation.setLocationList(locationValueList);// 
                                // corresponding old path names to this array ...
                                addMeLocation.setOriginalPathsCorrespondingToArray(matchedOldPathNames);
                               // reprocessedInfo.add(new ProcessedInfo(location, location, info.get(i).getPatternReplacement(), info.get(i).getMatchedPattern(), info.get(i).getAssociatedClassName(),localVarType, localVarUnits, localVarDescr, piCategory));
                               reprocessedInfo.add(addMeLocation);
                               alreadyAdded.add(location);
                            }
                        }
                    }

                } else if (!isArray && !alreadyAdded.contains(salClassParameterName)) {
                    reprocessedInfo.add(info.get(i));

                    for (String it : recordPatternSubstitution) {
                        if (it.split("::")[0].trim().equals(mykey.getKey().trim()) && it.split("::")[1].trim().equals(salClassParameterName.trim())) {

                            // when the uncommented code follows we'll be creating the location parameter
                            // perhaps this is where we should be creating the VALUE of the location parameter itself and adding it into
                            // an array carried by ProcessedInfo - the value of the location parameter is based on information we already have
                            // so probably no need to do this in GenericConverter - we could attach an array of strings to ProcessedInfo called "LocationValue" or
                            // some such thing and use the generic converter code to fill it up using the matched groups.
                            
                            String location = "Location";
                            if (it.split("::")[1].trim().contains(it.split("::")[2].trim())) {
                                location = it.split("::")[2].trim() + location;
                            }
                            if (!alreadyAdded.contains(location)) {
                                String localVarName = it.split("::")[2].trim() + "Location".trim();
                                String localVarType = "string";
                                String localVarUnits = "unitless";
                                if(DEBUG_PRINT){
                                 System.out.println("******************(5)**************"); 
                                 System.out.println(" Class Name  " + info.get(i).getAssociatedClassName());
                                 System.out.println(" Parameter Name  " + info.get(i).getSALClassParameterName());
                                 System.out.println(" ProcessedInfoMaker:Parameter Type " +info.get(i).getSALClassParameterType());
                                 System.out.println(" ProcessedInfoMaker:Path from Dictionary " + info.get(i).getOriginalPathName());
                                 System.out.println(" ProcessedInfoMaker:Location String " +info.get(i).getLocationListSALFormat());    
                                 System.out.println(" ProcessedInfoMaker:DAMN !!!");
                                 System.out.println("*************(5)*******************");
                                }
                                String localVarDescr = it.split("::")[2].substring(0, 1).toUpperCase() + it.split("::")[2].substring(1).toLowerCase() + " Location";
                                                            
                                ProcessedInfo addMeSingleLocation = new ProcessedInfo(location, location, info.get(i).getPatternReplacement(), info.get(i).getMatchedPattern(), info.get(i).getAssociatedClassName(), localVarType, localVarUnits, localVarDescr, piCategory);
                                addMeSingleLocation.setLocationString(info.get(i).getLocationString());
                                //reprocessedInfo.add(new ProcessedInfo(location, location, info.get(i).getPatternReplacement(), info.get(i).getMatchedPattern(), info.get(i).getAssociatedClassName(), localVarType, localVarUnits, localVarDescr, piCategory));
                                reprocessedInfo.add(addMeSingleLocation);
                                alreadyAdded.add(location);
                            }

                        }
                    }
                }

                finalReprocessedClassInfo.put(mykey.getKey(), reprocessedInfo);

            }
            //ensure that this is cleared before we move on to the next class  
            alreadyAdded.clear();
        } 
        
          return finalReprocessedClassInfo;
    }

    private String makeClassName(int level, String subsystem, String component, String salType, String path, String category) {

        if (subsystem.toLowerCase().contains("comcam")) {
            subsystem = "CCCamera";
        }


//       if(subsystem.toLowerCase().equals("ats")){ subsystem="Vacuum";}
//       if(subsystem.toLowerCase().equals("ats-power")){subsystem="WrebPower";}
       
        // debug
        String subsystemAndComponent = "";
        if (!subsystem.isEmpty()) {

            subsystemAndComponent = subsystem.substring(0, 1).toUpperCase() + subsystem.substring(1);
        } // there should always be a subsystem 
        component = component.trim();

        if (!component.isEmpty()) {
            subsystemAndComponent += "_" + component.substring(0, 1).toLowerCase() + component.substring(1);
        }

        String className = "";

        if (salType.toLowerCase().contains("settingsapplied")) {
            className = subsystem + "_" + "logevent" + "_" + component.toLowerCase();
        } else {
            className = subsystemAndComponent;
        }

       
        if (path.contains("main/")) {
            path = path.replace("main/", "");
            className = subsystemAndComponent;// and thats it - 

        } else {// go upto level - 

            for (int i = 0; i < level; i++) {
                className = className + "_" + capitalize(path.split("/")[i]);
            }

        }

        if (!(category == null)) {
            if (!category.isEmpty()) {
                className = className + category;
            }

        }

        // check to see if last character is underscore if so remove it 
        if (className.endsWith("_")) {
            className = className.substring(0, className.length() - 1);
        }

        return className;
    }

      private String makeSALClassParameterName1(String path, int level) {
        path = path.replace("main/", "");
        int pathLength = path.split("/").length;
        String variableName = "";

        // level better be less than or at most equal to path-length 
        if (level <= pathLength) {
            for (int i = level; i < pathLength; i++) {

                variableName = variableName + "_" + path.split("/")[i];

            }

            if (variableName.startsWith("_")) {
                variableName = variableName.substring(1);
            }
        } else {
            //TODO: We probably don'y want to exit now wwe are planning to use this class in more places
            System.out.println("!!You have specified level to be longer than the variable path + name!!");
            exit(-1);
        }

        return uncapitalize(variableName).trim();
    } 
    
    
    String makeSALClassParameterName(String path, int level) {
        path = path.replace("main/", "");
        int pathLength = path.split("/").length;
        String variableName = "";

       
        if(path.contains("/")){
        // level better be less than or at most equal to path-length 
        if (level <= pathLength) {
            for (int i = level; i < pathLength; i++) {

                variableName = variableName + "_" + path.split("/")[i];

            }

            if (variableName.startsWith("_")) {
                variableName = variableName.substring(1);

            }
        } else {
            if(DEBUG_PRINT){
             System.out.println("!!You have specified level to be longer than the variable path + name!!");
            }
            exit(-1);

           
        }}
        else{
        variableName = path;
        }
        
        return uncapitalize(variableName).trim();
    }

    private String capitalize(String in) {
        if (in.length() > 1) {
            return in.substring(0, 1).toUpperCase() + in.substring(1);
        } else {
            return in.toUpperCase();
        }
    }

    private String uncapitalize(String in) {
        if (in.length() > 1) {
            return in.substring(0, 1).toLowerCase() + in.substring(1);
        } else {
            return in.toLowerCase();
        }
    }

}
