package org.lsst.ccs.subsystem.ocsbridge.util;

import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.subsystem.ocsbridge.xml.Mapping;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription;
import org.lsst.ccs.subsystem.ocsbridge.xml.SALClassDescription.SALVariable;

/**
 *
 * @author Farrukh Azfar
 */
public class GenericConverter {

    private final Map<String, String> nameChangeTelemetry;
    private final Map<String, String> nameChangeSettingsApplied;

    //keep a record of when data is missing on the bus ...
    private final List<String> mia = new ArrayList<>();
    private final Map<Class, List<String>> alreadyWarned = new HashMap<>();
    private final Map<String, Class> classMap;
    private final List<String> subsystemToConvert;
    private final Mapping mapping;
    // print telemetry data processing
    private static final boolean DEBUG_PRINT_TRENDING = false;

    // print settings applied processing 
    private static final boolean DEBUG_PRINT_SETTINGS = false;

    // print reflection processing 
    private static final boolean DEBUG_PRINT_REFLECTION = false;

    // print location value processing 
    private static final boolean DEBUG_PRINT_BADVALUES = false;

    boolean badBValue = false;
    double badDValue = Double.NaN;
    int badIValue = -111111;
    float badFValue = Float.NaN;
    String badStringValue = "NOTFOUND";
    long badLValue = -1111111;

    /**
     * Create a new generic converter for converting CCS classes to SAL classes
     *
     * @param subsystemsToConvert The list of subsystem this generic converter
     * should handle
     * @param classMap A map to be used in case the class name as created by
     * XMLMaker/ProcessedInfo to the actual SAL generated class
     * @param nameChangeTelemetry A map to be used in the case that telemetry
     * variable names have changed
     * @param nameChangeSettingsApplied A map to be used in the case that
     * settings applied variable names have changed.
     */
    public GenericConverter(List<String> subsystemsToConvert, Map<String, Class> classMap, Mapping mapping, Map<String, String> nameChangeTelemetry, Map<String, String> nameChangeSettingsApplied) {
        this.subsystemToConvert = subsystemsToConvert;
        this.classMap = classMap;
        this.nameChangeTelemetry = nameChangeTelemetry;
        this.nameChangeSettingsApplied = nameChangeSettingsApplied;
        this.mapping = mapping;
    }

    public List<CameraTelemetry> telemetryConverter(Map<String, SALClassDescription> givenSalClassMapInfo, StatusSubsystemData data) throws ReflectiveOperationException {

        ArrayList<CameraTelemetry> salTelemetry = new ArrayList<>();
        Map<String, SALClassDescription> salClassMapInfo = givenSalClassMapInfo;

        // extract subsystem name  and get data
        String subsystemName = data.getOriginAgentInfo().getName();
        KeyValueData subsystemData = data.getSubsystemData();
        List<KeyValueData> values = (List<KeyValueData>) subsystemData.getValue();
        // data is contained in the list directly above ...

        // identify this class's printout
        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ");
            System.out.println("***** Start Printout : GenericConverter3:telemetryConverter ******");
        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ");
            System.out.println("****** GenericConverter3:telemetryConverter : Start Filling Received Bus Data *******");
        }

        Map<String, Object> dataNamesAndValues = new HashMap<>();

        for (KeyValueData kvd : values) {
            String pathFromBus = kvd.getKey();

            // For now we ignore the states in the passed in telemetry
            if (!pathFromBus.contains("/state")) {

                // fill in bus data ...
                dataNamesAndValues.put(pathFromBus, kvd.getValue());
                if (DEBUG_PRINT_TRENDING) {
                    System.out.println(" Variable name from bus " + kvd.getKey());
                    System.out.println(" Variable type from bus " + kvd.getType());
                    System.out.println(" Variable value from bus " + kvd.getValue());
                }
            }

        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println("****** GenericConverter3:telemetryConverter: End Filling Received Bus Data *******");
        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ********GenericConverter3:telemetryConverter SalClassMap ************");
            for (Map.Entry<String, SALClassDescription> scd : givenSalClassMapInfo.entrySet()) {
                System.out.println(" Class Name from SALClassDescription " + scd.getKey());
                System.out.println(" Complete SAL Class Description " + scd.getValue().toString());
            }
        }

        for (String subsystem : subsystemToConvert) {
            for (Map.Entry<String, SALClassDescription> scd : givenSalClassMapInfo.entrySet()) {
                String scdSALClassName = scd.getKey();

                if (DEBUG_PRINT_TRENDING) {

                    System.out.println(" Class Name from SALClassDescription " + scdSALClassName);

                }

                // actual SAL Class ...
                Class realSALClass = classMap.get(scdSALClassName);

                if (realSALClass == null) {

                    //System.out.println(" Printout of sal class name change follows : ");

                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" No entry found for " + scdSALClassName);
                        for (Map.Entry<String, Class> ck : classMap.entrySet()) {
                            System.out.println(" SalClassDescriptionName " + scdSALClassName + " Entry Key " + ck.getKey() + " class entry " + ck.getValue().getName());
                        }

                    }

                }

                // if the subsystemName from this batch of telemetry data is one of the subsystems we've been requested and if
                // the real class is not null we proceed. 
                if (subsystemName.equals(subsystem) && !(realSALClass == null)) {

                    // get the parameters of the first constructor - (there could be more so potentially this is something to look out for)    
                    Parameter[] parameters = realSALClass.getConstructors()[0].getParameters();

                    CameraTelemetry ct = (CameraTelemetry) reflectedClass(realSALClass, dataNamesAndValues, parameters, scd.getValue());
                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" ");
                        System.out.println(" GenericConverter3:telemetryConverter: Class Received ");
                        System.out.println(ct.toString());

                    }

                    int check[] = checkClassCompleteness(ct);
                    int numMissingValues = check[0];
                    int numGetMethods = check[1];

                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" ");
                        System.out.println(" GenericConverter3:telemetryConverter: Class quality check ");
                        System.out.println(" Number of missing values " + check[0]);
                        System.out.println(" Number of get functions " + check[1]);
                    }

                    if (DEBUG_PRINT_TRENDING) {
                        System.out.println(" ");
                        System.out.println("**** GenericConverter3:telemetryConverter: Missing values from reflected class ******");
                        System.out.println("Class name " + realSALClass.getName() + " Number of missing values " + numMissingValues + " Number of methods " + numGetMethods);
                    }

                    if (!(numMissingValues == numGetMethods)) {
                        salTelemetry.add(ct);
                    }
                }

            }

        }

        if (DEBUG_PRINT_TRENDING) {
            System.out.println(" ");
            System.out.println(" Number of classes in SAL Telemetry " + salTelemetry.size());
            System.out.println("****************** End of Printout from GenericConverter3:telemetryConverter");
        }

        return salTelemetry;
    }

    public List<CameraEvent> settingsAppliedEventConverter(Map<String, SALClassDescription> givenSalClassMapInfo, StatusConfigurationInfo sdata) throws ReflectiveOperationException {

        ConfigurationInfo data = sdata.getConfigurationInfo();

        String subsystemFromStatusConfiguration = sdata.getOriginAgentInfo().getName();

        Map<String, SALClassDescription> salClassMapInfo = givenSalClassMapInfo;

        ArrayList<CameraEvent> salSettings = new ArrayList<>();

        List<ConfigurationParameterInfo> cpinfo = data.getAllParameterInfo();

        Map<String, Map<String, Object>> categoryIndexedData = new HashMap<>();

        if (DEBUG_PRINT_SETTINGS) {
            System.out.println(" ");
            System.out.println("**** GenericConverter3:settingsAppliedConverter Data from Bus Message ****");

            for (ConfigurationParameterInfo dinfo : data.getAllParameterInfo()) {
                System.out.println(" Category from Bus : " + dinfo.getCategoryName() + " ,Data from bus: " + dinfo.getPathName() + " = " + dinfo.getConfiguredValueObject());
            }
        }

        if (DEBUG_PRINT_SETTINGS) {
            System.out.println(" ");
            System.out.println("**** GenericConverter3:settingsAppliedConverter SALClassDescription information ****");
            for (Map.Entry<String, SALClassDescription> scd : salClassMapInfo.entrySet()) {
                String scdSALClassName = scd.getKey();
                System.out.println(" Class Name from SALClassDescription " + scdSALClassName);
                System.out.println(" Printout of SALClassDescription " + scd.getValue().toString());
            }
        }
        // only one sybsystem per configuration status information 
        boolean legitSubsystem = false;
        if (subsystemToConvert.contains(sdata.getOriginAgentInfo().getName())) {
            legitSubsystem = true;
        }

        // can I turn this into a Map ? <actualSALClassName, category> is this guaranteed to be 1-1 ? 
        List<String> categoriesFromReceivedClassNames = new ArrayList<>();

        for (String category : data.getCategorySet()) {
            Map<String, Object> thisCategoriesData = new HashMap<>();

            for (ConfigurationParameterInfo cp : cpinfo) {

                if (category.equals(cp.getCategoryName()) && !(category == null)) {
                    thisCategoriesData.put(cp.getPathName(), cp.getConfiguredValueObject());
                }
            }

            if (!(category == null) && !category.isEmpty() && !(thisCategoriesData.isEmpty())) {
                categoryIndexedData.put(category, thisCategoriesData);
            }
        }

        // subsystems have matched so all the categories for this subsystem have been filled 
        Map<String, Object> localSettingsData = new HashMap<>();

        for (Map.Entry<String, SALClassDescription> scd : givenSalClassMapInfo.entrySet()) {

            String salClassDescriptionName = scd.getKey();
            System.out.println(" ");
            System.out.println("**** GenericConverter3:settingsAppliedConverter - Class Name from SALClass Description  ****");
            System.out.println(" SAL description class name " + salClassDescriptionName);

            Class realSALClass = classMap.get(salClassDescriptionName);
            if (DEBUG_PRINT_SETTINGS && realSALClass == null) {
                System.out.println(" ");
                System.out.println("**** GenericConverter3:settingsAppliedConverter - we did not find the SAL class  ****");
                System.out.println(" SAL Class from Class description " + salClassDescriptionName + " Whole map folows below ");
                for (Map.Entry<String, Class> clskey : classMap.entrySet()) {
                    System.out.println(" SAL Class Description Name  " + clskey.getKey() + " SAL Class in Map " + clskey.getValue().getName());
                }
            }

            if (!(realSALClass == null)) {

                System.out.println(" ");
                System.out.println("**** GenericConverter3:settingsAppliedConverter - changed class name  ****");
                System.out.println(" real class name " + realSALClass.getName());

                String categoryFromSALClassDescription = scd.getValue().getCategory();

                // any of the members of the ProcessedInfo list could be 
                if (DEBUG_PRINT_SETTINGS) {

                    System.out.println(" ");
                    System.out.println("**** GenericConverter3:settingsAppliedConverter - category obtained  ****");
                    System.out.println(" Category from SALClass Description " + scd.getValue().getCategory());
                    System.out.println("*****************");
                }

                localSettingsData = categoryIndexedData.get(categoryFromSALClassDescription);

                System.out.println(" ");
                System.out.println("**** GenericConverter3:settingsAppliedConverter - Local settings data from category "
                        + scd.getValue().getCategory() + " ****");
                for (Map.Entry<String, Object> key : localSettingsData.entrySet()) {

                    System.out.println(" Local Settings Data Name " + key.getKey() + " local Settings data type " + key.getValue().getClass().getTypeName()
                            + " local settings data value " + key.getValue());
                }
                System.out.println("*****************");

                Parameter[] parameters = realSALClass.getConstructors()[0].getParameters();

                if (DEBUG_PRINT_SETTINGS) {
                    Method[] premethods = realSALClass.getMethods();
                    System.out.println(" ");
                    System.out.println("**** GenericConverter3:settingsAppliedConverter Preliminary printout of methods and parameter names ****");

                    for (Method premethod : premethods) {
                        System.out.println(" Method name of reflected class before constructor invocation- compare to whats on the bus " + premethod.getName());
                    }
                    for (Parameter parameter : parameters) {
                        System.out.println(" Parameter name of reflected class before constructor invocation- compare to whats on the bus " + parameter.getName());
                    }
                }

                CameraEvent ce = (CameraEvent) reflectedClass(realSALClass, localSettingsData, parameters, scd.getValue());

                if (DEBUG_PRINT_SETTINGS) {
                    System.out.println(" ");
                    System.out.println(" GenericConverter3:settingsAppliedConverter: Class Received ");
                    System.out.println(" Forcing recompilation ");
                    System.out.println(ce.toString());
                    System.out.println("***********(3)***************\n");
                }

                int check[] = checkClassCompleteness(ce);
                int numMissingValues = check[0];
                int numGetMethods = check[1];

                if (DEBUG_PRINT_SETTINGS) {
                    System.out.println(" ");
                    System.out.println(" GenericConverter3:settingsAppliedConverter: Class quality check ");
                    System.out.println(" Number of missing values " + check[0]);
                    System.out.println(" Number of get functions " + check[1]);
                }

                if (!(numMissingValues == numGetMethods)) {
                    salSettings.add(ce);
                }

                if ((numMissingValues == numGetMethods)) {
                    if (DEBUG_PRINT_SETTINGS) {
                        System.out.println(" ");
                        System.out.println(" GenericConverter2:SettingsAppliedConverter");
                        System.out.println(" no values have been found on the bus");
                        System.out.println(" Values in the following have been extracted from ConfigInfo ");
                        System.out.println(" and inserted into the Map dataNamesAndValues ");
                        System.out.println(ce.toString());

                        for (String tospath : localSettingsData.keySet()) {

                            // For now we ignore the states in the passed in telemetry
                            if (!tospath.contains("/state")) {

                                System.out.println(tospath);
                                System.out.println(localSettingsData.get(tospath));
                            }

                        }
                        System.out.println(" *********************** ");
                    }

                }

            }
        }

        // still needs to be assigned a value - return SalSettings
        return salSettings;
    }

    public Object reflectedClass(Class cls, Map<String, Object> dataNamesAndValues, Parameter[] parameters, SALClassDescription salClassDescriptionList) throws ReflectiveOperationException {

        // arguments for this class's constructor     
        ArrayList<Object> constructorArgs = new ArrayList<>();
        Map<String, List<SALClassDescription.BusVariable>> locationBusVariableCorrespondence = new HashMap<>();

        for (SALVariable sv : salClassDescriptionList.getVariables()) {
            if (sv instanceof SALClassDescription.PatternMatchedSALVariable) {
                SALClassDescription.PatternMatchedSALVariable pv = (SALClassDescription.PatternMatchedSALVariable) sv;
                if (!pv.getBusVariables().isEmpty()) {
                    locationBusVariableCorrespondence.put(pv.getLocationVariable().getVariableName(), pv.getBusVariables());
                }
            }
        }

        if (DEBUG_PRINT_REFLECTION) {
            System.out.println("");
            System.out.println("****** Generic Converter 3 : SAL Class Description Information ****");
            for (SALVariable psv : salClassDescriptionList.getVariables()) {
                System.out.println(" SAL Variable " + psv.toString());
            }
        System.out.println(" ");
        }
        
        boolean found = false;

        // loop over received constructor parameters
        for (Parameter parameter : parameters) {

            found = false;

            if (DEBUG_PRINT_REFLECTION) {
                System.out.println(" ");
                System.out.println(" ***** START GenericConverter3:reflectedClass called by both telemetryConverter or settingsAppliedConverter ******");
                System.out.println(" Looping over SAL class parameters and dealing with parameter " + parameter.getName() + " type " + parameter.getType());
            }

            if (parameter.getName().trim().equals("priority")) {
                constructorArgs.add(1);
                if (DEBUG_PRINT_REFLECTION) {
                    System.out.println(" Priority found and set ");
                }
                found = true;

            }

            if (parameter.getName().trim().equals("version")) {
                constructorArgs.add(1);
                if (DEBUG_PRINT_REFLECTION) {
                    System.out.println(" Version found and set ");
                }
                found = true;

            }
            // if this was the priority variable no need to consider it further assign
            // 1 to priority and break 

            if (!found) {
                for (SALVariable sv : salClassDescriptionList.getVariables()) {

                    // what's the parameter Type ?
                    String parameterType = parameter.getType().toString();
                    boolean parameterIsArray = parameter.getType().isArray();

                    // do I need to check any of the following or am I ok  ?
                    boolean isAnArrayList = parameterType.contains("List") || parameterType.contains("List");
                    boolean isAnArray = (!isAnArrayList) && parameterType.contains("[");
                    boolean isAString = parameterType.contains("String") || parameterType.contains("String");
                    boolean isAMap = parameterType.contains("Map");

                    if (DEBUG_PRINT_REFLECTION) {
                        System.out.println(" ");
                        System.out.println(" ***** GenericConverter3:reflectedClass Comparing ******");
                        System.out.println(" Comparing SAL Class parameter " + parameter.getName() + " ,with SAL class description " + sv.getVariableName());
                        System.out.println(" SAL Class Parameter type " + parameter.getType() + " ,SAL Class description parameter type " + sv.getType());
                    }

                    // we're checking to see if the SALClassDescription has used patternMatching                    
                    boolean involvesPatternMatching = sv instanceof SALClassDescription.PatternMatchedSALVariable;
                    if (involvesPatternMatching) {
                        if (DEBUG_PRINT_REFLECTION) {
                            System.out.println(" ");
                            System.out.println(" ***** GenericConverter3:reflectedClass involved pattern matching  ******");
                            System.out.println(" Comparing SAL Class parameter " + parameter.getName() + " ,with SAL class description " + sv.getVariableName());
                            System.out.println(" SAL Class Parameter type " + parameter.getType() + " ,SAL Class description parameter type " + sv.getType());
                        }
                    }

                    String svVariableName = "";
                    if (sv instanceof SALClassDescription.LocationVariable) {
                        svVariableName = ((SALClassDescription.LocationVariable) sv).getVariableName();
                        //System.out.println(" Location variable found " + svVariableName + " ,Compare to just variable name " + sv.getVariableName());
                    }

                    // check that the variable name on the SALClassDescription is the same as the parameter name in the class constructor
                    if (sv.getVariableName().toLowerCase().trim().equals(parameter.getName().toLowerCase().trim())) {

                        if (DEBUG_PRINT_REFLECTION) {
                            System.out.println(" ");
                            System.out.println(" ***** GenericConverter3:reflectedClass Match found ******");
                            System.out.println(" SAL Class parameter " + parameter.getName() + " SAL variable description " + sv.getVariableName());
                            System.out.println(" SAL Class parameter type " + parameter.getType() + " SAL variable description type " + sv.getType());
                        }

                        // array was built from pattern matching and is not a location variable also the corresponding SAL class variable is 
                        // an array
                        if (parameterIsArray && involvesPatternMatching && !parameter.getName().trim().toLowerCase().contains("location")) {

                            Object arrayArg = java.lang.reflect.Array.newInstance(parameter.getType().getComponentType(),
                                    ((SALClassDescription.PatternMatchedSALVariable) sv).getCount());

                            for (int i = 0; i < ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().size(); i++) {

                                SALClassDescription.BusVariable busVarName = ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(i);

                                if (!(null == returnBusData(busVarName.getPathAndNameOnBuses(), dataNamesAndValues))) {
                                    if (DEBUG_PRINT_REFLECTION) {
                                        System.out.println(" ");
                                        System.out.println("***** GenericConverter3:reflectedClass Filling an array ***");
                                        System.out.println(" Array SAL Class Name " + parameter.getName() + " SAL Description Name " + sv.getVariableName());
                                        System.out.println(" Bus type " + returnBusData(busVarName.getPathAndNameOnBuses(),dataNamesAndValues).getClass().getTypeName());
                                        System.out.println(" Bus variable name " + busVarName.getPathAndNameOnBuses());
                                        System.out.println(" Array element index " + i + " Array element value from bus " + returnBusData(busVarName.getPathAndNameOnBuses(), dataNamesAndValues));
                                    }
                                    Object returnedData = returnBusData(busVarName.getPathAndNameOnBuses(), dataNamesAndValues);
                                    if(returnedData.getClass().isArray()) {
                                      arrayArg = returnedData; 
                                    } else{
                                      java.lang.reflect.Array.set(arrayArg, i, returnedData);
                                    }
                                } else {
                                    java.lang.reflect.Array.set(arrayArg, i, -111111);
                                }
                            }

                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println(" ***** GenericConverter3:reflectedClass arrayArg **** ");
                                System.out.println(" Adding  argument to constructor list : SAL class parameter  name " + parameter.getName() + " ,SAL class parameter type " + parameter.getClass().getTypeName());
                                System.out.println(" SAL class description parameter name " + sv.getVariableName() + " SAL class description parameter type " + sv.getType());
                                System.out.println(" Constructor array argument value " + arrayArg.toString() + " constructor array argument type " + arrayArg.getClass().getTypeName());
                            }
                            constructorArgs.add(arrayArg);
                            found = true;

                        }

                        // deal with locations - we have to be sure that SAL classes NEVER will have a string array in the constructor.
                        if (!parameterIsArray && parameter.getName().trim().toLowerCase().contains("location")) {
                            // we've already checked that the names of the parameters in the class and the ProcessedInfo are the same 
                            // here we have created the value of the location variable inside the processed Info so no issues.

                            List<String> locationVals = new ArrayList<>();

                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("***** GenericConverter3:reflectedClass Location variables ***");
                                System.out.println("  SAL Class parameter name Name " + parameter.getName() + " SAL Description Name " + sv.getVariableName());
                                System.out.println(" SAL Class Description printout " + sv.toString());
                            }

                            if (locationBusVariableCorrespondence.get(sv.getVariableName()) == null) {
                                if (DEBUG_PRINT_REFLECTION) {
                                    System.out.println(" Null from locationBusVariableCorrespondence " + sv.toString());
                                }

                            }

                            for (SALClassDescription.BusVariable bv : locationBusVariableCorrespondence.get(sv.getVariableName())) {
                                locationVals.add((getLocationValue(bv.getPathAndNameOnBuses()).replace("/", "")));
                                if (DEBUG_PRINT_REFLECTION) {
                                    System.out.println(" Added location value " + (getLocationValue(bv.getPathAndNameOnBuses()).replace("/", "")));
                                }
                            }

                            constructorArgs.add(convertStringList(locationVals));// has to be a string so I am not checking 
                            // if its not a string something is seriously wrong ...
                            found = true;

                        }

                        // one of the unusual cases where the bus should have a an array published on it or we didn't add it here .... 
                        if (parameterIsArray && !involvesPatternMatching) {
                            //constructorArgs.add(returnBusData(pil.getOriginalPathName(), values));

                            boolean notmatched = (sv instanceof SALClassDescription.SimpleSALVariable);
                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("****GenericConverter3:reflectedClass : An array not from pattern matching (unusual)");
                                System.out.println(" Should not be match - is it (false is trouble) ? " + notmatched);
                                System.out.println(" SAL parameter name " + parameter.getName() + " ,SALClassDescription parameter " + sv.getVariableName());
                                System.out.println(" SAL parameter type " + parameter.getType() + " ,SALClassDescription parameter type " + sv.getType());
                                System.out.println(" There should really be a Bus variable associated with this and that should be the next print is it ? ");
                                if (notmatched) {
                                    System.out.println(" Good news (see previous line) " + ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses());
                                }
                            }

                            // this will crash if there is no busVariable associated 
                            String busVariableName = ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses();
                            constructorArgs.add(returnBusData(busVariableName, dataNamesAndValues));

                            found = true;
                        }

                        // pattern matching has occured and the parameter is not an array
                        if (involvesPatternMatching && !parameterIsArray && !parameter.getName().toLowerCase().contains("location")) {

                            boolean matched = (sv instanceof SALClassDescription.PatternMatchedSALVariable);
                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("****GenericConverter3:reflectedClass : Not an array but pattern matched (unusual)");
                                System.out.println(" Should not be match  ? " + matched);
                                System.out.println(" SAL parameter name " + parameter.getName() + " ,SALClassDescription parameter " + sv.getVariableName());
                                System.out.println(" SAL parameter type " + parameter.getType() + " ,SALClassDescription parameter type " + sv.getType());
                                System.out.println(" There should really be a Bus variable associated with this and that should be the next print is it ? ");
                                if (matched) {
                                    System.out.println(" Good news (see previous line) " + ((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(0).getPathAndNameOnBuses());
                                }
                            }

                            constructorArgs.add(returnBusData(((SALClassDescription.PatternMatchedSALVariable) sv).getBusVariables().get(0).getPathAndNameOnBuses(), dataNamesAndValues));
                            found = true;
                        }

                        if (!involvesPatternMatching && !parameterIsArray && !parameter.getName().toLowerCase().contains("location")) {

                            boolean notmatched = (sv instanceof SALClassDescription.SimpleSALVariable);
                            if (DEBUG_PRINT_REFLECTION) {
                                System.out.println(" ");
                                System.out.println("****GenericConverter3:reflectedClass : Not an array not pattern matched - a vanilla parameter");
                                System.out.println(" Should not be match - is it (false is trouble) ? " + notmatched);
                                System.out.println(" SAL parameter name " + parameter.getName() + " ,SALClassDescription parameter " + sv.getVariableName());
                                System.out.println(" SAL parameter type " + parameter.getType() + " ,SALClassDescription parameter type " + sv.getType());
                                System.out.println(" There should really be a Bus variable associated with this and that should be the next print is it ? ");
                                if (notmatched) {
                                    System.out.println(" Good news (see previous line) " + ((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses());
                                }
                            }
                            constructorArgs.add(returnBusData(((SALClassDescription.SimpleSALVariable) sv).getBusVariable().getPathAndNameOnBuses(), dataNamesAndValues));
                            found = true;

                        }

                    }
                }

            }

            if (!found) {

                if (DEBUG_PRINT_REFLECTION) {
                    System.out.println(" **** GenericConverter3:reflectedClass Not found parameter information ");
                    System.out.println(" Class Name " + cls.getCanonicalName());
                    System.out.println(" Didn't find " + parameter.getName());
                    System.out.println(" reflectedClass : Printing from stored information Is it here ? ");

                    for (String tospath : dataNamesAndValues.keySet()) {
                        System.out.println(" path name " + tospath);
                        System.out.println(" value " + dataNamesAndValues.get(tospath) + " type " + dataNamesAndValues.get(tospath).getClass().getTypeName());
                    }

                }

                mia.add(parameter.getName());
                alreadyWarned.put(cls, mia);
                int jj = Collections.frequency(alreadyWarned.get(cls), parameter.getName());
                // we should be able to set the number of warnings we want to tolerate
                // how do we check for individual types ? // needs a rework need to check for arrays etc ...
                if (parameter.getType().isArray()) {
                    // here we need to suss out the size of the array and add that many doubles ...
                    constructorArgs.add(new double[0]);
                } else if (parameter.getType().equals(String.class)) {
                    constructorArgs.add("NOTFOUND");
                } else if (parameter.getType().getName().toLowerCase().contains("boolean")) {
                    constructorArgs.add(badBValue);
                } else if (parameter.getType().getName().toLowerCase().contains("double")) {
                    constructorArgs.add(badDValue);
                } else if (parameter.getType().getName().toLowerCase().contains("float")) {
                    constructorArgs.add(badFValue);
                } else {
                    constructorArgs.add(-111111);
                }
            }

        }

        if (DEBUG_PRINT_REFLECTION) {
            System.out.println(" ");
            System.out.println("***** GenericConverter3.settingsAppliedConverter: cross check just before reflection ******");
            System.out.println(" Constructor Argugments and types about to pass these on to constructor ");
            for (int i = 0; i < constructorArgs.size(); i++) {
                System.out.println(" Value and Type " + constructorArgs.get(i) + " " + constructorArgs.get(i).getClass().getTypeName());
            }
            
            System.out.println(" Please compare the with the constructor of the class " + cls.getConstructors()[0].toString());
        }
        // so now constructor args must be full.....or not let's check ....
        try {
            return cls.getConstructors()[0].newInstance(constructorArgs.toArray());

        } catch (ReflectiveOperationException | RuntimeException x) {
            if (DEBUG_PRINT_REFLECTION) {
                System.out.printf("Error calling constructor for %s with args %s\n", cls.getCanonicalName(), cls.getConstructors()[0].toString() + " " + constructorArgs.toString());// betterToString(cls.getConstructors()[0], constructorArgs));
            }
            throw x;
        }

    }

    // given the bus path name and the mapping this function will generate the value of the location 
    String getLocationValue(String busPathName) {
        
        Mapping.Match match = mapping.match(busPathName);
        if (match != null) {
            return match.getLocation();
        } else {
            return "NOTFOUND";
        }
    }

    // return data 
    Object returnBusData(String pathName, Map<String, Object> dataNamesAndValues) { //List<KeyValueData> values) {

        Object returnValue = null;

        for (String tospath : dataNamesAndValues.keySet()) {

            String key = tospath;

            // For now we ignore the states in the passed in telemetry
            if (!key.contains("/state") && pathName.toLowerCase().trim().equals(tospath.toLowerCase().trim())) {

                // change doubles to floats remove this once xml is fixed 
                returnValue = dataNamesAndValues.get(tospath);
                if (returnValue instanceof Double) {
                    returnValue = ((Double) returnValue).floatValue();
                }

                if (returnValue instanceof HashMap) {
                    returnValue = ((HashMap) returnValue).toString();

                    System.out.println(" Hash Map printout " + returnValue + " " + returnValue.getClass().getTypeName());
                    if (returnValue.getClass().getTypeName().contains("[")) {
                        System.out.println(" Is a string array/list " + returnValue);
                    }

                }

                if (returnValue.getClass().getTypeName().toLowerCase().contains("string") && returnValue instanceof String[]) {

                    returnValue = convertStringArray((String[]) returnValue);
                }

                if (returnValue.getClass().getTypeName().toLowerCase().contains("string") && (returnValue instanceof List)) {

                    returnValue = convertStringList((List<String>) returnValue);
                }

            }
        }

        if (returnValue == null) {
            returnValue = badDValue;
        }
        return returnValue;
    }

    // clean up the class name used for xmls
    String cleanupXMLClassName(String xmlClassName) {

        xmlClassName = xmlClassName.replace("comcam-", "");
        String[] blahs = xmlClassName.split("_");
        String cleanedName = "";

        for (int j = 1; j < blahs.length; j++) {
            cleanedName += blahs[j] + "_";
        }

        int lastIndex = cleanedName.lastIndexOf("_");
        cleanedName = cleanedName.substring(0, lastIndex);

        return cleanedName.trim();
    }

    // clean up the SAL class Name for comparision
    String cleanupSALClassName(String salClassName) {

        String cleanedName = salClassName.split("\\.")[salClassName.split("\\.").length - 1];

        cleanedName = cleanedName.replace("ATCamera", "").trim();
        cleanedName = cleanedName.replace("MTCamera", "").trim();
        cleanedName = cleanedName.replace("CCCamera", "").trim();
        cleanedName = cleanedName.replace("Telemetry", "").trim();
        cleanedName = cleanedName.replace("SettingsAppliedEvent", "").trim();

        if (cleanedName.startsWith("_")) {
            cleanedName = cleanedName.replaceFirst("[_]", "").trim();
        }

        return cleanedName.trim();
    }

    boolean badValueOrValues(Object invokedValue, Object badValue, String className, String methodName) {
        boolean bad = true;

        boolean isanArray = invokedValue.getClass().isArray();
        String type = invokedValue.getClass().getTypeName();
        boolean isDouble = false;
        boolean isString = false;
        boolean isInt = false;
        boolean isFloat = false;
        boolean isLong = false;
        boolean isBool = false;

        if (type.toLowerCase().contains("double")) {
            isDouble = true;
            badValue = badDValue;
        } else if (type.toLowerCase().contains("string")) {
            isString = true;
            badValue = badStringValue;
        } else if (type.toLowerCase().contains("int")) {
            isInt = true;
            badValue = badIValue;
        } else if (type.toLowerCase().contains("float")) {
            isFloat = true;
            badValue = badFValue;
        } else if (type.toLowerCase().contains("long")) {
            isLong = true;
            badValue = badLValue;
        } 

        if (isanArray && isDouble) {

            double[] doubleArray = (double[]) invokedValue;
            for (int i = 0; i < doubleArray.length; i++) {
                // even if one value is good the array is passable        
                if (!Double.isNaN(doubleArray[i])) {
                    bad = false;
                }
            }
        }
        if (isanArray && isInt) {

            int[] intArray = (int[]) invokedValue;
            for (int i = 0; i < intArray.length; i++) {
                // even if one value is good the array is passable        
                if (!(intArray[i] == (int) badValue)) {
                    bad = false;
                }
            }
        }
        if (isanArray && isFloat) {

            float[] floatArray = (float[]) invokedValue;
            for (int i = 0; i < floatArray.length; i++) {
                // even if one value is good the array is passable        
                if (!Float.isNaN(floatArray[i])) {
                    bad = false;
                }
            }
        }

        if (!isanArray) {
            if (!invokedValue.equals(badValue)) {
                bad = false;
            }
        }

        if (DEBUG_PRINT_BADVALUES) {
            System.out.println(" ******* GenericConverter3:badValueOrValues ******* ");
            System.out.println(" Class name " + className + " method name " + methodName);
            System.out.println(" Array ? " + isanArray + " type " + invokedValue.getClass().getTypeName());
            System.out.println(" Bad value check " + invokedValue + " " + badValue);
        }
        return bad;
    }

    Map<String, Object> copyStringObjectMap(Map<String, Object> inMap) {

        Map<String, Object> copy = new HashMap<>();

        copy.putAll(inMap);

        return copy;
    }

    private static String betterToString(Constructor constructor, List<Object> data) {
        List<String> result = new ArrayList<>();
        for (int i = 0; i < data.size(); i++) {
            result.add(constructor.getParameters()[i].getName() + "=" + (data.get(i).getClass().isArray() ? arrayToString(data.get(i)) : data.get(i).toString()));
        }
        return "[" + String.join(", ", result) + "]";
    }

    private static String convertStringList(List<String> stringList) {

        String[] localArray = new String[stringList.size()];
        stringList.toArray(localArray);
        return manipulateStringArray(localArray);
    }

    private static String convertStringArray(String[] stringArray) {

        String[] localArray = null;

        return manipulateStringArray(stringArray);

    }

    private static String manipulateStringArray(String[] inputArray) {
        String returnStringI = "";
        String returnStringF = "";
        int lengthOfStringArray = inputArray.length;
        if (!(lengthOfStringArray > 0)) {
            System.out.println("Warning from GenericConverter3:manipulateStringArray : Length of string Array is 0 !");
            returnStringF = "NOTFOUND";
        } else {
            for (int i = 0; i < lengthOfStringArray; i++) {
                returnStringI += inputArray[i] + ":";
                if (i == lengthOfStringArray - 1) {
                    int target = returnStringI.lastIndexOf(":");
                    if (target == -1) {
                        returnStringF = returnStringI;
                    } else {
                        returnStringF = returnStringI.substring(0, target);
                    }
                }
            }
        }
        return returnStringF;
    }

    private static String arrayToString(Object array) {
        List<String> result = new ArrayList<>();
        int l = Array.getLength(array);
        for (int i = 0; i < l; i++) {
            result.add(Array.get(array, i).toString());
        }
        return "[" + String.join(", ", result) + "]";
    }

    int[] checkClassCompleteness(Object cls) throws IllegalAccessException, IllegalArgumentException, InvocationTargetException {

        Parameter[] thisClassParameters = cls.getClass().getConstructors()[0].getParameters();
        Method[] methods = cls.getClass().getDeclaredMethods();
        ArrayList<Method> getMethods = new ArrayList<>();
        int numMissingValues = 0;
        int numGetMethods = 0;
        int[] returnValues = new int[2];

        for (int i = 0; i < methods.length; i++) {
            String methodName = methods[i].getName();

            String[] parts = methodName.split("\\s");
            if (!parts[parts.length - 1].contains("toString") && parts[parts.length - 1].contains("get")) {
                getMethods.add(methods[i]);
                numGetMethods++;
            }
        }

        boolean checkbadvalue = false;

        for (Method method : getMethods) {

            if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("double")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badDValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("string")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badStringValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("float")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badFValue, cls.getClass().getName(), method.getName());
            } else if (method.invoke(cls).getClass().getTypeName().toLowerCase().contains("int")) {
                checkbadvalue = badValueOrValues(method.invoke(cls), badIValue, cls.getClass().getName(), method.getName());
            }

            if (checkbadvalue) {
                numMissingValues++;
            }

            if (!checkbadvalue && method.getName().toLowerCase().contains("location")) {
                numMissingValues++;
            }

        }

        returnValues[0] = numMissingValues;
        returnValues[1] = numGetMethods;
        return returnValues;
    }

}

