package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.IOException;
import java.time.Duration;
import java.util.concurrent.ExecutionException;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.UnauthorizedLevelException;
import org.lsst.ccs.services.UnauthorizedLockException;

/**
 * A wrapper around ConcurrentMessageUtilities with support for strong typing of
 * responses, and more explicit error handling
 *
 * @author tonyj
 */
public class MCMCommandSender {

    private final ConcurrentMessagingUtils cmu;
    private final String target;
    private final AgentLockService lockService;
    private final AgentMessagingLayer agentMessagingLayer;

    MCMCommandSender(String target, AgentMessagingLayer agentMessagingLayer, AgentLockService lockService) {
        this.target = target;
        this.lockService = lockService;
        this.agentMessagingLayer = agentMessagingLayer;
        cmu = new ConcurrentMessagingUtils(agentMessagingLayer);
    }

    /**
     * Send a command with no return and no timeout
     *
     * @param target The subsystem to send to
     * @param command The command
     * @param args The command arguments, if any
     * @throws ExecutionException If something goes wrong
     */
    void sendCommand(String command, Object... args) throws ExecutionException {
        sendCommand(Void.TYPE, null, command, args);
    }

    <T> T sendCommand(Class<T> returnType, String command, Object... args) throws ExecutionException {
        return sendCommand(returnType, null, command, args);
    }

    /**
     * Send a command
     *
     * @param <T> The return type
     * @param target The subsystem to send to
     * @param command The command to send
     * @param returnType The expected return type
     * @param timeout The timeout (or null to use default)
     * @param args The command arguments, if any
     * @return The result
     * @throws ExecutionException If something goes wrong
     */
    <T> T sendCommand(Class<T> returnType, Duration timeout, String command, Object... args) throws ExecutionException {
        CommandRequest request = new CommandRequest(target, command, args);
        try {
            Object result = timeout == null ? cmu.sendSynchronousCommand(request) : cmu.sendSynchronousCommand(request, timeout);
            return returnType.cast(result);
        } catch (Exception ex) {
            throw new ExecutionException(String.format("Error sending command %s to %s", command, target), ex);
        }
    }

    // Move to ControlledSubsystem?
    void lock(int level) throws UnauthorizedLevelException, UnauthorizedLockException, IOException {
        lockService.setLevelForAgent(target, level);
    }

    void unlock() {

    }

}
