package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ScheduledFuture;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * <ol>
 * <li>If focalPlaneSubsystem is null (the default) it simulates all focalPlane
 * operations within this class</li>
 * <li>If focalPlaneSubsystem is specified, it delegates the operations to the
 * specified subsystem</li>
 * </ol>
 *
 * @author tonyj
 */
public class FocalPlane {

    private final CCS ccs;
    private FocalPlaneInterface focalPlane;
    private final State raftsState;
    private final MCMConfig config;

    /**
     * Time to readout the science rafts
     */
    static final Duration READOUT_TIME = Duration.ofMillis(2000);
    /**
     * Time to clear the sensors
     */
    static final Duration CLEAR_TIME = Duration.ofMillis(70);
    /**
     * Idle time before a clear is required
     */
    static final Duration QUIESCENT_BEFORE_CLEAR = Duration.ofMillis(4000);

    public enum RaftsState {
        NEEDS_CLEAR, CLEARING, INTEGRATING, READING_OUT, QUIESCENT
    }

    FocalPlane(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        raftsState = new State(RaftsState.NEEDS_CLEAR);
        ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), raftsState);
        // Whenever we enter quiescent state, we start a timer to indicate when a clear is needed
        // If we exit ready state we cancel the timer. Note, this should move into the focal-plane
        // subsystem itself, after which this code should move into FocalPlaneSimulation
        raftsState.addStateChangeListener(new State.StateChangeListener<RaftsState>() {
            private ScheduledFuture<?> clearFuture;

            @Override
            public void stateChanged(CCSTimeStamp when, RaftsState currentState, RaftsState oldState) {
                if (currentState == RaftsState.QUIESCENT) {
                    clearFuture = ccs.schedule(QUIESCENT_BEFORE_CLEAR, () -> {
                        raftsState.setState(when, RaftsState.NEEDS_CLEAR);
                    });
                } else {
                    if (clearFuture != null) {
                        clearFuture.cancel(false);
                        clearFuture = null;
                    }
                }
            }

        });

        focalPlane = new FocalPlaneSimulation(ccs, raftsState);

    }

    void clear(int nClears) throws ExecutionException {
        focalPlane.clear(nClears);
    }

    void startIntegration(ImageName imageName, Map<String, String> parsedKeyValueData, LocationSet locations, String annotation) throws ExecutionException {
        focalPlane.startIntegration(imageName, parsedKeyValueData, locations, annotation);
    }

    void endIntegration(boolean readout, Duration exposure) throws ExecutionException {
        focalPlane.endIntegration(readout, exposure);
    }

    void play(String playlist, boolean repeat) throws ExecutionException {
        focalPlane.play(playlist, repeat);
    }

    void definePlaylist(String playlist, String folder, String[] images) throws ExecutionException {
        focalPlane.definePlaylist(playlist, folder, images);
    }

    /**
     * Called when the MCM receives a start command.
     *
     * @param configName
     */
    void start(String configName) throws ExecutionException {
        focalPlane.start(configName);
    }

    void setFocalPlaneSubsystem(FocalPlaneSubsystemLayer focalPlaneSubsystem) {
        this.focalPlane = focalPlaneSubsystem;
    }

    void registerMCMSubsystem(MCMSubsystem mcm) {
        focalPlane = new FocalPlaneSubsystemLayer(mcm, ccs, config);
    }
}
