package org.lsst.ccs.subsystem.ocsbridge.xml;

import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.w3c.dom.Comment;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 *
 * @author farrukh
 */
public class XMLMaker {

    private final boolean verbose;
    private final UnitConverter unitConverter = new UnitConverter();
    private enum SALType {
        TELEMETRY("MONITORING", "Telemetry", "Telemetry"), 
        SETTINGS_APPLIED(DataProviderInfo.Type.CONFIGURATION.name(), "SettingsApplied", "Event");

        private final String classNameAppendage;
        private final String configurationName;
        private final String salElementName;
                
        SALType(String configurationName, String classNameAppendage, String salElementName) {
            this.classNameAppendage = classNameAppendage;
            this.configurationName = configurationName;
            this.salElementName = salElementName;
        }

        public String getClassNameAppendage() {
            return classNameAppendage;
        }

        public String getConfigurationName() {
            return configurationName;
        }

        public String getSalElementName() {
            return salElementName;
        }
    }

    XMLMaker(boolean verbose) {
        this.verbose = verbose;
    }

    Document makeXMLFromDataProviderDictionary(StatusDataProviderDictionary info, String subsystemName, String componentName, int level, Mapping mapping) throws ParserConfigurationException, TransformerException {
        final SALType salType = SALType.TELEMETRY;
        ProcessedInfoMaker infoMaker = new ProcessedInfoMaker(info, subsystemName, componentName, level, mapping, salType.getConfigurationName(), salType.getClassNameAppendage());
        return generateXML(subsystemName, componentName, salType, infoMaker.getProcessedInfo(), level);
    }

    Document makeXMLFromConfigDictionary(StatusDataProviderDictionary info, String subsystemName, String componentName, int level, Mapping mapping) throws IOException, ClassNotFoundException, ParserConfigurationException, TransformerException {
        final SALType salType = SALType.SETTINGS_APPLIED;
        ProcessedInfoMaker infoMaker = new ProcessedInfoMaker(info, subsystemName, componentName, level, mapping, salType.getConfigurationName(), salType.getClassNameAppendage());
        return generateXML(subsystemName, componentName, salType, infoMaker.getProcessedInfo(), level);
    }

    private Document createDocumentHeader() throws ParserConfigurationException {
        // ***** This is document builder stuff which belongs only here - Farrukh *****
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        DocumentBuilder db = dbf.newDocumentBuilder();
        return db.newDocument();
    }

    private Element createRootElement(Document dom, SALType salType) {
        Element rootEle = dom.createElement("SAL" + salType.getSalElementName() + "Set");
        rootEle.setAttributeNS("http://www.w3.org/2001/XMLSchema-instance",
                "xsi:noNamespaceSchemaLocation", "http://project.lsst.org/ts/sal_objects/schema/" + salType.getSalElementName() + "Set.xsd");
        dom.appendChild(rootEle);
        return rootEle;
    }
    
    /**
     * Generate XML for all of the classes defined in the processedInfo
     * @param subsystemName
     * @param componentName
     * @param salType
     * @param finalReprocessedClassInfo
     * @param level
     * @return
     * @throws ParserConfigurationException
     * @throws TransformerConfigurationException
     * @throws TransformerException 
     */
    Document generateXML(String subsystemName, String componentName, SALType salType, Map<String, List<ProcessedInfo>> finalReprocessedClassInfo, int level) throws ParserConfigurationException, TransformerConfigurationException, TransformerException {

        if (verbose) {
            for (Map.Entry<String, List<ProcessedInfo>> key : finalReprocessedClassInfo.entrySet()) {
                System.out.println(" Class Name " + key.getKey());
                System.out.println(key.getValue());
            }
        }

        Document dom = createDocumentHeader();
        Element rootEle = createRootElement(dom, salType);

        for (Map.Entry<String, List<ProcessedInfo>> key : finalReprocessedClassInfo.entrySet()) {
            // alphabetization of the ProcessedInfo List before passing it on to createElement
            for(ProcessedInfo pi : key.getValue()){ pi.getSALClassParameterName();};
            List<ProcessedInfo> alphabetizedPI = key.getValue();
            alphabetizedPI.sort(Comparator.comparing(ProcessedInfo::getSALClassParameterName));
            
            createElement(dom, rootEle, salType, subsystemName, key.getKey().trim(), alphabetizedPI, level);

            //createElement(dom, rootEle, salType, subsystemName, key.getKey().trim(), key.getValue(), level);
        }

        boolean debug = true;
        if (debug) {
            TransformerFactory tranFactory = TransformerFactory.newInstance();
            Transformer aTransformer = tranFactory.newTransformer();
            aTransformer.setOutputProperty(OutputKeys.INDENT, "yes");
            Source src = new DOMSource(dom);
            Result dest = new StreamResult(System.out);
            aTransformer.transform(src, dest);

        }

        return dom;
    }

    private void createElement(Document dom, Element rootEle, SALType salType, String subsystemName, String className, List<ProcessedInfo> processedInfoList, int level) {

        if (subsystemName.contains("comcam")) {
            subsystemName = "CCCamera";
        }

        int firstUnderScore = className.indexOf("_");
        char[] charArray = className.toCharArray();
        charArray[firstUnderScore + 1] = Character.toLowerCase(charArray[firstUnderScore + 1]);
        className = String.valueOf(charArray);

        // to tackle SettingsApplied - currently unfilled unless we have SettingsApplied - but there could be others
        if (className.contains("focalplane")) {
            className = className.replace("focalplane", "focalPlane");
        }
        
        String classNameWithAppendage = className + salType.getClassNameAppendage();
        
        String text = "Created by xmlmaker for subsystem "+subsystemName+" class "+classNameWithAppendage+" using level "+level;
        Comment comment = dom.createComment(text);
        rootEle.appendChild(comment);
        Comment comment2 = dom.createComment(String.join("", Collections.nCopies(text.length(), "=")));
        rootEle.appendChild(comment2);
        
        Element salElement = dom.createElement("SAL" + salType.getSalElementName());
        rootEle.appendChild(salElement);

        Element subsystemElement = dom.createElement("Subsystem");
        subsystemElement.setTextContent(subsystemName);
        salElement.appendChild(subsystemElement);

        Element topicElement = dom.createElement("EFDB_Topic");
        topicElement.setTextContent(classNameWithAppendage);
        salElement.appendChild(topicElement);
        // SAL will add priority 
        //just for SettingsApplied or also for Event ?
        if (salType == SALType.SETTINGS_APPLIED) {
            // add version                
            Element version = dom.createElement("item");
            salElement.appendChild(version);
            Element efdbName = dom.createElement("EFDB_Name");
            efdbName.setTextContent("version");
            version.appendChild(efdbName);
            Element desc = dom.createElement("Description");
            desc.setTextContent("Version number of these settings");
            version.appendChild(desc);
            Element type = dom.createElement("IDL_Type");
            type.setTextContent("long");
            version.appendChild(type);
            Element units = dom.createElement("Units");
            units.setTextContent("unitless");
            version.appendChild(units);
            Element count = dom.createElement("Count");
            count.setTextContent("1");
            version.appendChild(count);
        }


        for (ProcessedInfo pi : processedInfoList) {
            int counts = 1;
            String type = pi.getSALClassParameterType();
            // add java.lang.abc 
            if (type.toLowerCase().contains("long")) {
                type = "long long";
            }
            // make it safe against int array ? 
            if (type.toLowerCase().contains("int")) {
                type = type.replace("int", "long");
            }
            if (type.toLowerCase().contains("map")) {
                type = "string";
            }
            //System.out.println(" type from xml maker " + type);

            String description = pi.getSALClassParameterDescription();
            if (description == null || description.isEmpty()) {
                // If all CCS elements had a description this would not be needed
                description = pi.getSALClassParameterName();
            }

            if (pi.getSALClassParameterType().toLowerCase().contains("list")) {

                System.out.println("type=" + pi.getSALClassParameterType());
                String tmp = pi.getSALClassParameterType().split("<")[1];
                tmp = tmp.replace(">", "").trim();
                tmp = tmp.split("\\.")[tmp.split("\\.").length - 1];
                tmp = tmp.toLowerCase();
                if(tmp.contains("long"))tmp = "long long";
                if(tmp.contains("int"))tmp= "long";

                // need to take counts from ProcessedInfo - this is temporary
                if (pi.getCounts() > 0) {
                    counts = pi.getCounts();
                } else {
                    counts = 42;
                }

                if (pi.getSALClassParameterType().toLowerCase().contains("string")) {
                    counts = 1;
                    description = description + " String Array concatenated with character ':'";
                }
                type = tmp;

            }

            // check to see if not
            //System.out.println("<item>");
            Element item = dom.createElement("item");
            salElement.appendChild(item);
//                String variableName = pi.getVariableName();
            String variableName = pi.getSALClassParameterName();
            if (variableName.contains("_")) {

                String temp = "";
                String[] splitVariable = variableName.split("_");
                splitVariable[0] = splitVariable[0].toLowerCase();

                for (int i = 0; i <= splitVariable.length - 1; i++) {
                    temp += splitVariable[i] + "_";
                }

                if (temp.charAt(temp.length() - 1) == '_') {
                    temp = temp.substring(0, temp.length() - 1);
                }

                variableName = temp;
            } else if (variableName.length() > 1) {
                variableName = variableName.substring(0, 1).toLowerCase() + variableName.substring(1);
            } else {
                variableName = variableName.toLowerCase();
            }

            Element efdbName = dom.createElement("EFDB_Name");
            efdbName.setTextContent(variableName);
            item.appendChild(efdbName);

            if (pi.getSALClassParameterUnits() != null) {
                if (pi.getSALClassParameterUnits().toLowerCase().contains("rpm")) {
                    description = description + " (RPM)";
                }
            }

            Element desc = dom.createElement("Description");
            desc.setTextContent(description);
            item.appendChild(desc);

            Element idlType = dom.createElement("IDL_Type");
            idlType.setTextContent(type);
            item.appendChild(idlType);
            String ccsUnits = pi.getSALClassParameterUnits();
            String salUnits = unitConverter.convertCCSUnitToSALUnit(ccsUnits);
            Element unitsElement = dom.createElement("Units");
            unitsElement.setTextContent(salUnits);
            item.appendChild(unitsElement);
            Element countElement = dom.createElement("Count");
            countElement.setTextContent(String.valueOf(counts));
            item.appendChild(countElement);

        }
    }

    private String cleanCapitalize(String inString) {
        String output = inString;
        output = output.toLowerCase();
        output = output.substring(0, 1).toUpperCase() + output.substring(1);
        output = output.trim();
        return output;
    }

    /**
     * Write the XML document to a file
     *
     * @param output The location to write the output
     * @param dom The XML document
     * @throws TransformerConfigurationException
     * @throws TransformerException
     */
    void writeXML(StreamResult output, Document dom) throws TransformerConfigurationException, TransformerException {
        final TransformerFactory transformerFactory = TransformerFactory.newInstance();
        Transformer tr = transformerFactory.newTransformer();
        tr.setOutputProperty(OutputKeys.INDENT, "yes");
        tr.setOutputProperty(OutputKeys.METHOD, "xml");
        tr.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
        tr.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "2");
        // send DOM to file
        tr.transform(new DOMSource(dom), output);
    }

}
