package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 *
 * @author tonyj
 */
public class FilterChangerSimulation implements FilterChangerInterface {

    private enum FilterType {
        U, G, R, I, Z, Y, NONE, OTHER
    };

    // This should all become (simulation only) configuration
    private static final Map<String, FilterType> mainCameraFilters = new LinkedHashMap<>();

    static {
        // These are made up, we should find what the real filters are
        mainCameraFilters.put("NONE", FilterType.NONE);
        mainCameraFilters.put("u-10", FilterType.U);
        mainCameraFilters.put("g-9", FilterType.G);
        mainCameraFilters.put("r-1", FilterType.R);
        mainCameraFilters.put("i-9", FilterType.I);
        mainCameraFilters.put("z-100", FilterType.Z);
        mainCameraFilters.put("y-99", FilterType.Y);
    }

    private static final List<String> mainCameraInstalledFilters = new ArrayList<>(mainCameraFilters.keySet());

    static {
        mainCameraInstalledFilters.remove("y-099");
    }

    private static final Map<String, FilterType> comCamFilters = new LinkedHashMap<>();

    static {
        // These are taken from comcam-fcs configuration
        comCamFilters.put("u_05", FilterType.U);
        comCamFilters.put("g_07", FilterType.G);
        comCamFilters.put("g_01", FilterType.G);
        comCamFilters.put("r_03", FilterType.R);
        comCamFilters.put("i_06", FilterType.I);
        comCamFilters.put("z_03", FilterType.Z);
        comCamFilters.put("z_02", FilterType.Z);
        comCamFilters.put("y_04", FilterType.Y);
        comCamFilters.put("pinhole", FilterType.OTHER);
        comCamFilters.put("NONE", FilterType.NONE);
    }

    private static final List<String> comCamInstalledFilters = new ArrayList<>();
    static {
        comCamInstalledFilters.add("i_06");
        comCamInstalledFilters.add("r_03");
        comCamInstalledFilters.add("g_07");
    }

    static final Duration LOAD_TIME = Duration.ofMillis(15000);
    static final Duration ROTATION_TIME_PER_DEGREE = Duration.ofMillis(100);
    static final Duration UNLOAD_TIME = Duration.ofMillis(15000);

    private final Map<String, FilterType> allFilters;
    private final List<String> installedFilters;

    private final MCMConfig config;
    private String currentFilter;
    private int currentRotationPosition = 0;
    private final CCS ccs;
    private final State filterState;

    FilterChangerSimulation(CCS ccs, MCMConfig config) {

        this.ccs = ccs;
        this.config = config;
        switch (config.getCameraType()) {
            case MAIN_CAMERA:
                allFilters = mainCameraFilters;
                currentFilter = "NONE";
                installedFilters = mainCameraInstalledFilters;
                break;
            case COMCAM:
                allFilters = comCamFilters;
                installedFilters = comCamInstalledFilters;
                currentFilter = comCamInstalledFilters.get(0);
                break;
            default:
                throw new RuntimeException("Filter changer not supported for " + config.getCameraType());
        }
        filterState = new State(FilterChanger.FilterState.UNLOADED);
        ccs.getAggregateStatus().add(CCSTimeStamp.currentTime(), filterState);
    }

    @Override
    public Map<String, String> getAvailableFilters() {
        return allFilters.entrySet().stream().collect(Collectors.toMap(entry -> entry.getKey(), entry -> entry.getValue().name().toLowerCase()));
    }

    @Override
    public List<String> getInstalledFilters() {
        return Collections.unmodifiableList(installedFilters);
    }

    @Override
    public void start(String configName) {
        /// No Op
    }

    @Override
    public String getCurrentFilter() {
        return currentFilter;
    }

    @Override
    public void setFilter(String filterName) {
        try {
            int position = getInstalledFilters().indexOf(filterName);
            if (position < 0) {
                throw new IllegalArgumentException("Invalid or uninstalled filter: " + filterName);
            }
            FilterType requestedFilterType = allFilters.get(filterName);
            ccs.fireEvent(new MCM.CCSSetFilterEvent(filterName, requestedFilterType.name().toLowerCase()));

            int targetRotation = -1;
            if (filterName.equals(currentFilter)) {
                // No-op? -- actually comcam-fcs rejects this, maybe we should too?
            } else if (requestedFilterType == FilterType.NONE) {
                filterState.setState(FilterChanger.FilterState.UNLOADING);
                Future<Void> waitForUnloaded = ccs.waitForStatus(FilterChanger.FilterState.UNLOADED);
                ccs.schedule(UNLOAD_TIME, () -> {
                    filterState.setState(FilterChanger.FilterState.UNLOADED);
                    currentFilter = filterName;
                });
                waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            } else {
                if (allFilters.get(currentFilter) != FilterType.NONE) {
                    filterState.setState(FilterChanger.FilterState.UNLOADING);
                    Future<Void> waitForUnloaded = ccs.waitForStatus(FilterChanger.FilterState.UNLOADED);
                    ccs.schedule(UNLOAD_TIME, () -> {
                        filterState.setState(FilterChanger.FilterState.UNLOADED);
                        currentFilter = filterName;
                    });
                    waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
                }
                targetRotation = (position - 1) * 360 / 5;
                if (currentRotationPosition != targetRotation) {
                    // FIXME: Always rotates in same direction, probably not realistic
                    int degreesToRotate = Math.abs(currentRotationPosition - targetRotation) % 360;
                    filterState.setState(FilterChanger.FilterState.ROTATING);
                    Future<Void> waitForRotation = ccs.waitForStatus(FilterChanger.FilterState.UNLOADED);
                    Duration rotationTime = ROTATION_TIME_PER_DEGREE.multipliedBy(degreesToRotate);
                    ccs.schedule(rotationTime, () -> {
                        filterState.setState(FilterChanger.FilterState.UNLOADED);
                    });
                    waitForRotation.get(rotationTime.toMillis() * 2, TimeUnit.MILLISECONDS);
                    currentRotationPosition = targetRotation;
                }
                filterState.setState(FilterChanger.FilterState.LOADING);
                Future<Void> waitForLoaded = ccs.waitForStatus(FilterChanger.FilterState.LOADED);
                ccs.schedule(LOAD_TIME, () -> {
                    filterState.setState(FilterChanger.FilterState.LOADED);
                    currentFilter = filterName;
                });
                waitForLoaded.get(LOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            }
            ccs.fireEvent(new MCM.CCSSetFilterEvent(filterName, requestedFilterType.name().toLowerCase(), position, targetRotation));
        } catch (InterruptedException | ExecutionException | TimeoutException ex) {
            throw new RuntimeException("Error while changing filter", ex);
        }
    }

    @Override
    public Duration getEstimatedDurationForFilterChange(String filterName) {
        return ROTATION_TIME_PER_DEGREE.multipliedBy(360).plus(LOAD_TIME).plus(UNLOAD_TIME);
    }
}
