package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.io.Serializable;
import java.time.Duration;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.RawCommand;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.subsystem.ocsbridge.util.EventListener;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * An implementation of MCMLayer which uses the CCS buses to communicate with an
 * MCM running in a separate process.
 *
 * @author tonyj
 */
class MCMCCSLayer implements MCMLayer {

    private final Subsystem subsystem;
    private final CCS ccs;
    private final OCSBridgeConfig config;

    MCMCCSLayer(Subsystem subsystem, CCS ccs, OCSBridgeConfig config) {
        this.subsystem = subsystem;
        this.ccs = ccs;
        this.config = config;

        Predicate<BusMessage<? extends Serializable, ?>> stateChangeFilter
                = BusMessageFilterFactory.messageOrigin(config.getMCMName()).and(BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class));

        subsystem.getMessagingAccess().addStatusMessageListener((msg) -> handleMCMStatusChangeMessage(msg), stateChangeFilter);

        Predicate<BusMessage<? extends Serializable, ?>> eventFilter
                = BusMessageFilterFactory.messageOrigin(config.getMCMName()).and(BusMessageFilterFactory.embeddedObjectClass(KeyValueData.class));

        subsystem.getMessagingAccess().addStatusMessageListener((msg) -> handleMCMEvent(msg), eventFilter);

        for (String l : config.getListenSubsystems()) {
            Predicate<BusMessage<? extends Serializable, ?>> subsystemFilter = BusMessageFilterFactory.messageOrigin(l);
            subsystem.getMessagingAccess().addStatusMessageListener((msg) -> handleSubsystemEvent(msg), subsystemFilter );
        }
    }

    @Override
    public CCSCommand.CCSCommandResponse execute(CCSCommand ccsCommand) {
        CCSExecutor executor = new CCSBusesExecutor(ccsCommand);
        return new CCSCommand.CCSCommandResponse(executor);
    }

    @Override
    public void addStateChangeListener(State.StateChangeListener<Enum> stateChangeListener) {
        ccs.addStateChangeListener(stateChangeListener);
    }

    @Override
    public void removeStateChangeListener(State.StateChangeListener<Enum> stateChangeListener) {
        ccs.removeStateChangeListener(stateChangeListener);
    }

    @Override
    public void addEventListener(CCSEvent.CCSEventListener eventListener) {
        ccs.addEventListener(eventListener);
    }

    @Override
    public void removeEventListener(CCSEvent.CCSEventListener eventListener) {
        ccs.removeEventListener(eventListener);
    }

    @Override
    public void addStatusMessageListener(EventListener<StatusMessage> eventListener) {
        ccs.addStatusMessageListener(eventListener);
    }

    @Override
    public void removeStatusMessageListener(EventListener<StatusMessage> eventListener) {
        ccs.removeStatusMessageListener(eventListener);
    }
    
    private void handleMCMStatusChangeMessage(StatusMessage msg) {
        StatusStateChangeNotification statusChange = (StatusStateChangeNotification) msg;
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        CCSTimeStamp when = statusChange.getCCSTimeStamp();
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            ccs.getAggregateStatus().add(when, new State(value));
        });
    }

    private void handleMCMEvent(StatusMessage msg) {
        Object object = msg.getObject();
        if (object instanceof KeyValueData) {
            KeyValueData data = (KeyValueData) object;
            if ("CCSEvent".equals(data.getKey())) {
                CCSEvent evt = (CCSEvent) data.getValue();
                ccs.fireEvent(evt);
                return;
            }
        } 
        ccs.fireEvent(msg);
    }
    
    private void handleSubsystemEvent(StatusMessage msg) {
        ccs.fireEvent(msg);
    }

    /**
     * This class is used by the OCSBridge to send a command to the MCM.
     */
    private class CCSBusesExecutor extends CCSExecutor {

        private final ArrayBlockingQueue queue = new ArrayBlockingQueue(2);
        private final CCSCommand ccsCommand;

        public CCSBusesExecutor(CCSCommand ccsCommand) {
            this.ccsCommand = ccsCommand;
            CommandOriginator originator = new CommandOriginator() {
                @Override
                public void processAck(CommandAck ack) {
                    queue.offer(ack);
                }

                @Override
                public void processResult(CommandResult result) {
                    queue.offer(result);
                }

                @Override
                public void processNack(CommandNack nack) {
                    queue.offer(nack);
                }
            };
            CommandRequest request = new CommandRequest(config.getMCMName(), new RawCommand(ccsCommand.getCommand(), ccsCommand.getArguments()));
            subsystem.getMessagingAccess().sendCommandRequest(request, originator);
        }

        @Override
        protected Duration testPreconditions() throws CCSCommand.CCSPreconditionsNotMet {
            try {
                Object result = queue.take();
                if (result instanceof CommandAck) {
                    return ((CommandAck) result).getTimeout();
                } else if (result instanceof CommandNack) {
                    throw new CCSCommand.CCSPreconditionsNotMet(((CommandNack) result).getReason());
                } else {
                    throw new RuntimeException("Unexpected object received while waiting for ack/nack: "+ ccsCommand.toString());
                }
            } catch (InterruptedException ex) {
                throw new RuntimeException("Unexpected interrupt while waiting for command ack/nack: "+ ccsCommand.toString(), ex);
            }
        }

        @Override
        protected void execute() throws ExecutionException {
            try {
                Object result = queue.take();
                if (result instanceof CommandResult) {
                    CommandResult cr = (CommandResult) result;
                    if (!cr.wasSuccessful()) {
                        Throwable t = (Throwable) cr.getObject();
                        if (t instanceof ExecutionException) {
                            throw (ExecutionException) t;
                        } else {
                            throw new ExecutionException("MCM command failed: " + ccsCommand.toString(), t);                        
                        }
                    }
                } else {
                    throw new RuntimeException("Unexpected object received while waiting for command result: "+ ccsCommand.toString());
                }
            } catch (InterruptedException x) {
                throw new RuntimeException("Interrupt while waiting for command result: "+ ccsCommand.toString(), x);
            }
        }
    }

}
