package org.lsst.ccs.subsystem.ocsbridge.util;

import java.util.Arrays;
import java.util.Map;
import java.util.StringJoiner;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.StreamSupport;

/**
 * A class for converting String[] into delimited strings. This is used whenever
 * a String array needs to be packed into a single string for SAL.
 *
 * @author tonyj
 */
public class DelimitedStringSplitJoin {

    private static final char DEFAULT_DELIMITER = ':';
    private static final char DEFAULT_ESCAPE = '\\';
    private final String escapedDelimiter;
    private final String escapedEscape;
    private final Pattern splitPattern;
    private final Pattern unescapePattern;
    private final char delimiter;

    /**
     * Create a DelimitedStringSplitJoin object with the default delimiter (:)
     * and escape character (\).
     */
    public DelimitedStringSplitJoin() {
        this(DEFAULT_DELIMITER, DEFAULT_ESCAPE);
    }

    /**
     * Create a DelimitedStringSplitJoin object with the specified delimiter and
     * escape character
     *
     * @param delimiter The delimiter character to use
     * @param escape The escape character to use
     */
    public DelimitedStringSplitJoin(char delimiter, char escape) {
        this.delimiter = delimiter;
        this.escapedDelimiter = Matcher.quoteReplacement(String.valueOf(delimiter));
        this.escapedEscape = Matcher.quoteReplacement(String.valueOf(escape));
        this.splitPattern = Pattern.compile(String.format("(?<!%s)%s", this.escapedEscape, this.escapedDelimiter));
        this.unescapePattern = Pattern.compile(String.format("%s(.)", this.escapedEscape));
    }

    /**
     * Split a single string into a string array, using the specified delimiter
     * characters.
     *
     * @param input The string to be split
     * @return The resulting string array
     */
    public String[] split(CharSequence input) {
        return splitPattern.splitAsStream(input)
                .map(s -> unescape(s))
                .toArray(size -> new String[size]);
    }

    /**
     * Join a string array into a single string, using the specified delimiter
     * characters.
     *
     * @param input The input string array
     * @return The resulting string
     */
    public String join(CharSequence[] input) {
        return Arrays.stream(input).collect(joining());
    }

    /**
     * Join a collection if strings into a single string, using the specified
     * delimiter and escape characters.
     *
     * @param input The collection of strings
     * @return The resulting string
     */
    public String join(Iterable<? extends CharSequence> input) {
        return StreamSupport.stream(input.spliterator(), true).collect(joining());
    }

    /**
     * Returns a Collector that concatenates the input elements into a String,
     * in encounter order using the specified delimited and escape characters.
     *
     * @return
     */
    public Collector<CharSequence, ?, String> joining() {
        return Collector.of(() -> new StringJoiner(String.valueOf(delimiter)),
                (joiner, item) -> joiner.add(escape(item)),
                StringJoiner::merge,
                StringJoiner::toString
        );
    }

    private String escape(CharSequence s) {
        return s.toString().replaceAll(escapedEscape, escapedEscape + escapedEscape).replaceAll(escapedDelimiter, escapedEscape + escapedDelimiter);
    }

    private String unescape(String s) {
        return unescapePattern.matcher(s).replaceAll("$1");
    }

    /**
     * Split a set of keys and values, and zip them to create a Map.
     * @param keys
     * @param values
     * @return 
     */

    public Map<String, String> zip(CharSequence keys, CharSequence values) {
        String[] k = split(keys);
        String[] v = split(values);
        if (k.length != v.length) {
            throw new IllegalArgumentException("Inconsistent keys, values "+keys+" "+values);
        }
        return IntStream.range(0, k.length).boxed().collect(Collectors.toMap(i -> k[i], i -> v[i]));
    }
}
