package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.text.ParseException;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.AggregateStatus;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDefinePlaylistCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPlayCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig.Camera;
import org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.KeyValueParser;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A simple implementation of the MCM. Designed to be used either directly
 * within the OCSSimulation, or as a standalone process.
 *
 * @author tonyj
 */
public class MCM {

    private final MCMConfig config;
    private final ImageNameService imageNameService;
    private static final Logger LOG = Logger.getLogger(MCM.class.getName());

    public enum TakeImageReadinessState {

        NOT_READY, GETTING_READY, READY
    };

    public enum CalibrationState {
        DISABLED, ENABLED, INTEGRATING
    }

    private final CCS ccs;
    private final State takeImageReadinessState;
    private final State calibrationState;
    private final Shutter shutter;
    private final FocalPlane focalPlane;
    private final FilterChanger fcs;

    private Instant startIntegrationTime;
    private ScheduledFuture<?> startImageTimeout;
    private ScheduledFuture<?> shutterPrepSchedule;
    private ScheduledFuture<?> raftsClearSchedule;
    private CCSTimeStamp integrationStartTime;
    private boolean isStarted = false;

    public MCM(CCS ccs, MCMConfig config, ImageNameService imageNameService) {
        this.ccs = ccs;
        this.config = config;
        this.imageNameService = imageNameService;
        takeImageReadinessState = new State(TakeImageReadinessState.NOT_READY);
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        ccs.getAggregateStatus().add(now, takeImageReadinessState);
        calibrationState = new State(CalibrationState.DISABLED);
        ccs.getAggregateStatus().add(now, calibrationState);
        shutter = new Shutter(ccs, config);
        focalPlane = new FocalPlane(ccs, config);
        if (config.hasFilterChanger()) {
            fcs = new FilterChanger(ccs, config);
        } else {
            fcs = null;
        }

        // We are ready to take an image only if the rafts have been cleared, and the shutter
        // has been prepared.
        ccs.addStateChangeListener((when, state, oldState) -> {
            AggregateStatus as = ccs.getAggregateStatus();
            if (as.hasState(FocalPlane.RaftsState.QUIESCENT, Shutter.ShutterReadinessState.READY)) {
                takeImageReadinessState.setState(when, TakeImageReadinessState.READY);
            } else if (!as.hasState(TakeImageReadinessState.GETTING_READY)) {
                takeImageReadinessState.setState(when, TakeImageReadinessState.NOT_READY);
            }
            if (oldState == RaftsState.CLEARING || oldState == RaftsState.READING_OUT) {
                integrationStartTime = when;
            }
        });
    }

    CCSCommandResponse execute(CCSCommand command) {
        if (command instanceof CCSInitImageCommand) {
            return execute((CCSInitImageCommand) command);
        } else if (command instanceof CCSTakeImagesCommand) {
            return execute((CCSTakeImagesCommand) command);
        } else if (command instanceof CCSSetFilterCommand) {
            return execute((CCSSetFilterCommand) command);
        } else if (command instanceof CCSInitGuidersCommand) {
            return execute((CCSInitGuidersCommand) command);
        } else if (command instanceof CCSStartImageCommand) {
            return execute((CCSStartImageCommand) command);
        } else if (command instanceof CCSEndImageCommand) {
            return execute((CCSEndImageCommand) command);
        } else if (command instanceof CCSClearCommand) {
            return execute((CCSClearCommand) command);
        } else if (command instanceof CCSDiscardRowsCommand) {
            return execute((CCSDiscardRowsCommand) command);
        } else if (command instanceof CCSStartCommand) {
            return execute((CCSStartCommand) command);
        } else if (command instanceof CCSEnableCalibrationCommand) {
            return execute((CCSEnableCalibrationCommand) command);
        } else if (command instanceof CCSDisableCalibrationCommand) {
            return execute((CCSDisableCalibrationCommand) command);
        } else if (command instanceof CCSPlayCommand) {
            return execute((CCSPlayCommand) command);
        } else if (command instanceof CCSDefinePlaylistCommand) {
            return execute((CCSDefinePlaylistCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    CCSCommandResponse execute(CCSInitImageCommand command) {
        InitImageExecutor executor = new InitImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSTakeImagesCommand command) {
        TakeImagesExecutor executor = new TakeImagesExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSSetFilterCommand command) {
        SetFilterExecutor executor = new SetFilterExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSInitGuidersCommand command) {
        InitGuidersExecutor executor = new InitGuidersExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSStartImageCommand command) {
        StartImageExecutor executor = new StartImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSEndImageCommand command) {
        EndImageExecutor executor = new EndImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSClearCommand command) {
        ClearExecutor executor = new ClearExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDiscardRowsCommand command) {
        DiscardRowsExecutor executor = new DiscardRowsExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSStartCommand command) {
        StartExecutor executor = new StartExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSEnableCalibrationCommand command) {
        EnableCalibrationExecutor executor = new EnableCalibrationExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDisableCalibrationCommand command) {
        DisableCalibrationExecutor executor = new DisableCalibrationExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSPlayCommand command) {
        PlayExecutor executor = new PlayExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDefinePlaylistCommand command) {
        DefinePlaylistExecutor executor = new DefinePlaylistExecutor(command);
        return new CCSCommandResponse(executor);
    }

    //TODO: Remove me
    CCS getCCS() {
        return ccs;
    }

    FocalPlane getFocalPlane() {
        return focalPlane;
    }

    public Shutter getShutter() {
        return shutter;
    }

    public FilterChanger getFilterChanger() {
        return fcs;
    }

    MCMConfig getConfig() {
        return config;
    }

    class InitImageExecutor extends CCSExecutor {

        private final CCSInitImageCommand command;

        public InitImageExecutor(CCSInitImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (command.getDeltaT() <= 0 || command.getDeltaT() > 15) {
                throw new CCSPreconditionsNotMet("Invalid deltaT: " + command.getDeltaT());
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            Duration takeImagesExpected = Duration.ofMillis((long) (command.getDeltaT() * 1000));
            takeImageReadinessState.setState(TakeImageReadinessState.GETTING_READY);
            raftsClearSchedule = ccs.schedule(takeImagesExpected.minus(FocalPlane.CLEAR_TIME), () -> {
                try {
                    focalPlane.clear(1);
                } catch (ExecutionException x) {
                    LOG.log(Level.SEVERE, "Error while performing scheduled clear", x);
                }
            });
            shutterPrepSchedule = ccs.schedule(takeImagesExpected.minus(Shutter.PREP_TIME), () -> {
                shutter.prepare();
            });
        }

    }

    class TakeImagesExecutor extends CCSExecutor {

        private final CCSTakeImagesCommand command;
        private Map<String, String> parsedKeyValueData;
        private LocationSet locations;

        public TakeImagesExecutor(CCSTakeImagesCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (command.getNumImages() <= 0 || command.getNumImages() > config.getMaxImagesPerSequence()) {
                throw new CCSPreconditionsNotMet("Invalid number of images");
            }
            double minExposeTime = command.isShutter() ? config.getMinExposeTime() : 0;
            if (command.getExpTime() < minExposeTime || command.getExpTime() > config.getMaxExposeTime()) {
                throw new CCSPreconditionsNotMet("Invalid exposure time");
            }
            // Parse the keyValueData and check for legallity
            try {
                parsedKeyValueData = KeyValueParser.parse(command.getKeyValueMap());
                for (String key : config.getRequiredKeys()) {
                    if (!parsedKeyValueData.containsKey(key)) {
                        throw new CCSPreconditionsNotMet("Required key missing: " + key);
                    }
                }
                for (String key : parsedKeyValueData.keySet()) {
                    if (!config.getAllowedKeys().contains(key)) {
                        throw new CCSPreconditionsNotMet("Key not allowed: " + key);
                    }
                }
            } catch (ParseException x) {
                throw new CCSPreconditionsNotMet(x.getMessage());
            }

            if (!command.getSensors().trim().isEmpty()) {
                try {
                    locations = LocationSet.of(command.getSensors());
                } catch (RuntimeException x) {
                    throw new CCSPreconditionsNotMet("Invalid sensors argument" + command.getSensors());
                }
            } else {
                locations = new LocationSet();
            }

            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            // Worse case estimate
            return Duration.ofMillis((long) (command.getExpTime() * 1000)).plus(Shutter.MOVE_TIME).plus(FocalPlane.READOUT_TIME).multipliedBy(command.getNumImages());
        }

        @Override
        @SuppressWarnings("SleepWhileInLoop")
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException, CCSPreconditionsNotMet {

            List<ImageName> imageNames = imageNameService.getImageNames(command.getNumImages());
            Duration exposeTime = Duration.ofMillis((long) (command.getExpTime() * 1000));

            if (command.getNumImages() > 1) {
                shutter.startImageSequence();
            }

            try {

                for (int i = 0; i < command.getNumImages(); i++) {
                    Future waitUntilReady = ccs.waitForStatus(TakeImageReadinessState.READY);
                    if (takeImageReadinessState.isInState(TakeImageReadinessState.NOT_READY)) {
                        focalPlane.clear(1);
                        shutter.prepare();
                    } else if (takeImageReadinessState.isInState(TakeImageReadinessState.GETTING_READY)) {
                        boolean raftClearEnd = raftsClearSchedule.cancel(false);
                        boolean shutterPrepEnd = shutterPrepSchedule.cancel(false);
                        if (raftClearEnd) {
                            focalPlane.clear(1);
                        }
                        if (shutterPrepEnd) {
                            shutter.prepare();
                        }
                    }
                    waitUntilReady.get(20, TimeUnit.SECONDS);

                    ImageName imageName = imageNames.get(i);
                    // If we are directly triggering from focal-plane this event will not be generated.
                    // But it has info which is only available to the MCM, like the parsed key value data, the requested exposure time,
                    // and the number of images requested. 
                    ccs.fireEvent(new CCSImageNameEvent(parsedKeyValueData, command.getNumImages(), imageName, i,
                            integrationStartTime, command.getExpTime()));
                    LOG.log(Level.INFO, "Sent image name event for {0}", imageName);

                    if (command.isShutter()) {
                        focalPlane.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
                        shutter.expose(exposeTime);
                        //TODO: Check that shutter expose does not return until shutter has started opening
                        Future waitShutterClosed = ccs.waitForStatus(Shutter.ShutterState.CLOSED);
                        waitShutterClosed.get(exposeTime.plus(Shutter.MOVE_TIME).plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                        focalPlane.endIntegration(true, exposeTime);
                        // For the last exposure we only wait until the readout starts
                        // For other exposures we must wait until readout is complete
                        Future waitUntilDone = ccs.waitForStatus(i + 1 < command.getNumImages() ? FocalPlane.RaftsState.QUIESCENT : FocalPlane.RaftsState.READING_OUT);
                        waitUntilDone.get(FocalPlane.READOUT_TIME.plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                    } else {
                        focalPlane.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
                        Thread.sleep(exposeTime.toMillis());
                        // Is this correct, should we also set DARKTIME?
                        focalPlane.endIntegration(true, Duration.ZERO);
                        Future waitUntilDone = ccs.waitForStatus(i + 1 < command.getNumImages() ? FocalPlane.RaftsState.QUIESCENT : FocalPlane.RaftsState.READING_OUT);
                        waitUntilDone.get(FocalPlane.READOUT_TIME.plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                    }
                }

            } finally {
                shutter.endImageSequence();
            }
        }

    }

    @SkipEncoding
    public static class CCSEndOfImageTelemetryEvent extends CCSEvent {

        private final double darkTime;
        private final double exposureTime;
        private final long imageTag;
        private final CCSTimeStamp dateObs;
        private final CCSTimeStamp dateEnd;

        public CCSEndOfImageTelemetryEvent(double darkTime, double exposureTime, long imageTag, CCSTimeStamp dateObs, CCSTimeStamp dateEnd) {
            this.darkTime = darkTime;
            this.exposureTime = exposureTime;
            this.imageTag = imageTag;
            this.dateObs = dateObs;
            this.dateEnd = dateEnd;
        }

        public double getDarkTime() {
            return darkTime;
        }

        public double getExposureTime() {
            return exposureTime;
        }

        public long getImageTag() {
            return imageTag;
        }

        public CCSTimeStamp getDateObs() {
            return dateObs;
        }

        public CCSTimeStamp getDateEnd() {
            return dateEnd;
        }

    }

    @SkipEncoding
    public static class CCSImageNameEvent extends CCSEvent {

        private final int imagesInSequence;
        private final ImageName imageName;
        private final int sequenceNumber;
        private final CCSTimeStamp integrationStartTime;
        private final double exposureTime;
        private final Map<String, String> keyValueData;

        public CCSImageNameEvent(Map<String, String> keyValueData, int imagesInSequence, ImageName imageName, int sequenceNumber, CCSTimeStamp integrationStartTime, double exposureTime) {
            this.keyValueData = keyValueData;
            this.imagesInSequence = imagesInSequence;
            this.imageName = imageName;
            this.sequenceNumber = sequenceNumber;
            this.integrationStartTime = integrationStartTime;
            this.exposureTime = exposureTime;
        }

        public String getImageType() {
            return keyValueData.get("imageType");
        }

        public String getGroupId() {
            return keyValueData.get("groupId");
        }

        public int getImagesInSequence() {
            return imagesInSequence;
        }

        public ImageName getImageName() {
            return imageName;
        }

        public int getSequenceNumber() {
            return sequenceNumber;
        }

        public CCSTimeStamp getIntegrationStartTime() {
            return integrationStartTime;
        }

        public double getExposureTime() {
            return exposureTime;
        }

        public Map<String, String> getKeyValueData() {
            return keyValueData;
        }

        @Override
        public String toString() {
            return "CCSImageNameEvent{" + "imagesInSequence=" + imagesInSequence + ", imageName=" + imageName + ", sequenceNumber=" + sequenceNumber + ", integrationStartTime=" + integrationStartTime + ", exposureTime=" + exposureTime + ", keyValueData=" + keyValueData + '}';
        }

    }

    class SetFilterExecutor extends CCSExecutor {

        private final CCSSetFilterCommand command;

        public SetFilterExecutor(CCSSetFilterCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (fcs == null) {
                throw new CCSPreconditionsNotMet("setFilter command not supported");
            }

            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            if (!fcs.filterIsAvailable(command.getFilterName())) {
                throw new CCSPreconditionsNotMet("Invalid filter: " + command.getFilterName());
            }
            return fcs.getEstimatedTimeForChange(command.getFilterName());
        }

        @Override
        protected void execute() throws ExecutionException {
            fcs.setFilter(command.getFilterName());
        }
    }

    @SkipEncoding
    public static class CCSSetFilterEvent extends CCSEvent {

        private final String filterName;
        private final boolean start;
        private final double position;
        private final int slot;
        private final String filterType;

        public CCSSetFilterEvent(String filterName, String filterType) {
            this.filterName = filterName;
            this.filterType = filterType;
            this.slot = -1;
            this.position = -1;
            this.start = true;
        }

        public CCSSetFilterEvent(String filterName, String filterType, int slot, double position) {
            this.filterName = filterName;
            this.filterType = filterType;
            this.slot = slot;
            this.position = position;
            this.start = false;
        }

        public String getFilterName() {
            return filterName;
        }

        public boolean isStart() {
            return start;
        }

        public double getPosition() {
            return position;
        }

        public int getSlot() {
            return slot;
        }

        public String getFilterType() {
            return filterType;
        }

        @Override
        public String toString() {
            return "CCSSetFilterEvent{" + "filterName=" + filterName + ", start=" + start + ", position=" + position + ", slot=" + slot + ", filterType=" + filterType + '}';
        }

    }

    @SkipEncoding
    public static class CCSSettingsAppliedEvent extends CCSEvent {

        private final String settings;
        private final double timestamp;

        public CCSSettingsAppliedEvent(String settings, double timestamp) {

            this.settings = settings;
            this.timestamp = timestamp;
        }

        public String getSettings() {
            return settings;
        }

        public double getTimeStamp() {
            return timestamp;
        }

        @Override
        public String toString() {
            return "SettingsApplied{" + "settings=" + settings + ", timestamp=" + timestamp + '}';
        }

    }

    @SkipEncoding
    public static class CCSAtSettingsAppliedEvent extends CCSEvent {

        private final String settings;
        private final double timestamp;
        private final int version;
        private final String WREBSettingsVersion;
        private final String BonnShutterSettingsVersion;

        public CCSAtSettingsAppliedEvent(String settings, double timestamp, int version, String WREBSettingsVersion, String BonnShutterSettingsVersion) {

            this.settings = settings;
            this.timestamp = timestamp;
            this.version = version;
            this.WREBSettingsVersion = WREBSettingsVersion;
            this.BonnShutterSettingsVersion = BonnShutterSettingsVersion;
        }

        public String getSettings() {
            return settings;
        }

        public double getTimestamp() {
            return timestamp;
        }

        public int getVersion() {
            return version;
        }

        public String getWREBSettingsVersion() {
            return WREBSettingsVersion;
        }

        public String getBonnShutterSettingsVersion() {
            return BonnShutterSettingsVersion;
        }

        @Override
        public String toString() {
            return "CCSAtSettingsAppliedEvent{" + "settings=" + settings + ", timestamp=" + timestamp + ", version=" + version + ", WREBSettingsVersion=" + WREBSettingsVersion + ", BonnShutterSettingsVersion=" + BonnShutterSettingsVersion + '}';
        }

    }

    private class InitGuidersExecutor extends CCSExecutor {

        private final CCSInitGuidersCommand command;

        public InitGuidersExecutor(CCSInitGuidersCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            // FIXME: Currently this does not actually do anything
        }
    }

    private class EnableCalibrationExecutor extends CCSExecutor {

        private final CCSEnableCalibrationCommand command;

        public EnableCalibrationExecutor(CCSEnableCalibrationCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (!calibrationState.isInState(CalibrationState.DISABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            calibrationState.setState(CalibrationState.ENABLED);
        }
    }

    private class DisableCalibrationExecutor extends CCSExecutor {

        private final CCSDisableCalibrationCommand command;

        public DisableCalibrationExecutor(CCSDisableCalibrationCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            calibrationState.setState(CalibrationState.DISABLED);
        }
    }

    private class StartExecutor extends CCSExecutor {

        private final CCSStartCommand command;

        public StartExecutor(CCSStartCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws ExecutionException {
            if (fcs != null) {
                fcs.start("default");
            }
            if (focalPlane != null) {
                focalPlane.start("default");
            }
            if (shutter != null) {
                shutter.start("default");
            }
            isStarted = true;
            CCSTimeStamp now = CCSTimeStamp.currentTime();
            if (config.getCameraType() == Camera.AUXTEL) {
                ccs.fireEvent(new CCSAtSettingsAppliedEvent(command.getConfiguration(), now.getTAIDouble(), 1, "default(1)", "default(1)"));
            } else {
                ccs.fireEvent(new CCSSettingsAppliedEvent(command.getConfiguration(), now.getTAIDouble()));
            }
        }
    }

    private class ClearExecutor extends CCSExecutor {

        private final CCSClearCommand command;

        public ClearExecutor(CCSClearCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (command.getNClears() <= 0 || command.getNClears() > 15) {
                throw new CCSPreconditionsNotMet("Invalid nClears: " + command.getNClears());
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return FocalPlane.CLEAR_TIME.multipliedBy(command.getNClears());
        }

        @Override
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException {
            focalPlane.clear(command.getNClears());
            // TODO: Note, unlike initImages, the clear command remains active until the clears are complete (Correct?)
            Future waitUntilClear = ccs.waitForStatus(FocalPlane.RaftsState.QUIESCENT);
            waitUntilClear.get(FocalPlane.CLEAR_TIME.multipliedBy(command.getNClears()).plus(Duration.ofSeconds(1)).toMillis(), TimeUnit.MILLISECONDS);
        }

    }
    
    private class PlayExecutor extends CCSExecutor {

        private final CCSPlayCommand command;

        public PlayExecutor(CCSPlayCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (command.getPlaylist().length() == 0) {
                throw new CCSPreconditionsNotMet("Invalid playlist: " + command.getPlaylist());
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException {
            focalPlane.play(command.getPlaylist(), command.isRepeat());
        }

    }
    
    private class DefinePlaylistExecutor extends CCSExecutor {

        private final CCSDefinePlaylistCommand command;

        public DefinePlaylistExecutor(CCSDefinePlaylistCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (command.getPlaylist().length() == 0) {
                throw new CCSPreconditionsNotMet("Invalid playlist: " + command.getPlaylist());
            }
            if (command.getFolder().length() == 0) {
                throw new CCSPreconditionsNotMet("Invalid folder: " + command.getFolder());
            }
            if (command.getImages().length == 0) {
                throw new CCSPreconditionsNotMet("Invalid image list: " + command.getImages());
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException {
            focalPlane.definePlaylist(command.getPlaylist(), command.getFolder(), command.getImages());
        }

    }

    private class StartImageExecutor extends CCSExecutor {

        private final CCSStartImageCommand command;
        private Map<String, String> parsedKeyValueData;
        private LocationSet locations;

        public StartImageExecutor(CCSStartImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (command.getTimeout() < 1 | command.getTimeout() > 120) {
                throw new CCSPreconditionsNotMet("Invalid argument");
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }

            // Parse the keyValueData and check for legallity
            try {
                parsedKeyValueData = KeyValueParser.parse(command.getKeyValueMap());
                for (String key : config.getRequiredKeys()) {
                    if (!parsedKeyValueData.containsKey(key)) {
                        throw new CCSPreconditionsNotMet("Required key missing: " + key);
                    }
                }
                for (String key : parsedKeyValueData.keySet()) {
                    if (!config.getAllowedKeys().contains(key)) {
                        throw new CCSPreconditionsNotMet("Key not allowed: " + key);
                    }
                }
            } catch (ParseException x) {
                throw new CCSPreconditionsNotMet(x.getMessage());
            }

            if (!command.getSensors().trim().isEmpty()) {
                try {
                    locations = LocationSet.of(command.getSensors());
                } catch (RuntimeException x) {
                    throw new CCSPreconditionsNotMet("Invalid sensors argument" + command.getSensors());
                }
            } else {
                locations = new LocationSet();
            }

            return Duration.ofSeconds(1);
        }

        @Override
        protected void execute() throws Exception {
            Future waitUntilReady = ccs.waitForStatus(TakeImageReadinessState.READY);
            // FIXME: It is not necessary to always clear and prepare the shutter, especially
            // if we are not actually going to open the shutter.
            if (takeImageReadinessState.isInState(TakeImageReadinessState.NOT_READY)) {
                focalPlane.clear(1);
                shutter.prepare();
            }

            waitUntilReady.get(1, TimeUnit.SECONDS);

            ImageName imageName = imageNameService.getImageName();

            ccs.fireEvent(new CCSImageNameEvent(parsedKeyValueData, 1, imageName, 1, integrationStartTime, 0));
            if (command.isShutterOpen()) {
                focalPlane.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
                shutter.open();
                // FIXME: Wait for shutter to open? right now we return immediately
            } else {
                focalPlane.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
            }
            startIntegrationTime = Instant.now();
            calibrationState.setState(CalibrationState.INTEGRATING);
            startImageTimeout = ccs.schedule(Duration.ofMillis((long) (command.getTimeout() * 1000)), () -> {
                imageTimeout();
            });
        }
    }

    /**
     * Called if the timeout for a takeImages occurs
     */
    private void imageTimeout() {
        try {
            shutter.close();
            focalPlane.endIntegration(false, Duration.ZERO);
            calibrationState.setState(CalibrationState.ENABLED);
        } catch (ExecutionException x) {
            // TODO: Got into fault state
            LOG.log(Level.SEVERE, "Error during image timeout", x);
        }
    }

    private class EndImageExecutor extends CCSExecutor {

        private final CCSEndImageCommand command;

        public EndImageExecutor(CCSEndImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (!calibrationState.isInState(CalibrationState.INTEGRATING)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            if (startImageTimeout == null || startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("No exposure in progress");
            }
            return Shutter.MOVE_TIME;
        }

        @Override
        protected void execute() throws Exception {
            if (!startImageTimeout.cancel(false)) {
                throw new RuntimeException("Image exposure already timed out");
            }
            Future waitUntilClosed = ccs.waitForStatus(ShutterState.CLOSED);
            shutter.close();
            waitUntilClosed.get(Shutter.MOVE_TIME.plus(Duration.ofSeconds(1)).toMillis(), TimeUnit.MILLISECONDS);
            // TODO: What if shutter was never opened?
            Instant endIntegationTime = Instant.now();
            focalPlane.endIntegration(true, Duration.between(startIntegrationTime, endIntegationTime));
            calibrationState.setState(CalibrationState.ENABLED);
        }
    }

    private class DiscardRowsExecutor extends CCSExecutor {

        private final CCSDiscardRowsCommand command;

        public DiscardRowsExecutor(CCSDiscardRowsCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!isStarted) {
                throw new CCSPreconditionsNotMet("MCM start command has not been issued");
            }
            if (!calibrationState.isInState(CalibrationState.INTEGRATING)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            if (startImageTimeout == null || startImageTimeout.isDone()) {
                throw new CCSCommand.CCSPreconditionsNotMet("No exposure in progress");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            // FIXME: Nothing actually happens, should at least generate some events.
        }
    }
}
