package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.sim.MCMDirectLayer;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSCommandConverter;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSRevokeAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSimulateFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.sim.FilterChanger.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.OCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.PreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSImageNameEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSettingsAppliedEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig;
import org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSStateEventConverter;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSStateChangeToEventConverter;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.command.DisableCommand;
import org.lsst.sal.camera.command.EnableCommand;
import org.lsst.sal.camera.command.EnterControlCommand;
import org.lsst.sal.camera.command.ExitControlCommand;
import org.lsst.sal.camera.command.StandbyCommand;
import org.lsst.sal.camera.command.StartCommand;
import org.lsst.sal.camera.command.StartImageCommand;
import org.lsst.sal.camera.command.TakeImagesCommand;
import org.lsst.sal.camera.event.AppliedSettingsMatchStartEvent;
import org.lsst.sal.camera.event.AvailableFiltersEvent;
import org.lsst.sal.camera.event.EndOfImageTelemetryEvent;
import org.lsst.sal.camera.event.EndReadoutEvent;
import org.lsst.sal.camera.event.EndSetFilterEvent;
import org.lsst.sal.camera.event.SettingVersionsEvent;
import org.lsst.sal.camera.event.SettingsAppliedEvent;
import org.lsst.sal.camera.event.StartIntegrationEvent;
import org.lsst.sal.camera.event.StartReadoutEvent;
import org.lsst.sal.camera.event.StartSetFilterEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.CCSCommandStateEvent.CCSCommandState;
import org.lsst.sal.camera.states.SummaryStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent.SummaryState;
import org.lsst.sal.atcamera.event.ShutterMotionProfileEvent;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraStateChangeEvent;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.subsystem.common.actions.ComCamSharedVacState;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig.Camera;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.subsystem.focalplane.data.ImageMetaDataEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.DelimitedStringSplitJoin;
import org.lsst.ccs.subsystem.ocsbridge.util.FocalPlaneHardwareIdConverter;
import org.lsst.ccs.subsystem.ocsbridge.util.SummaryInfoConverter;

/**
 *
 * The Command Layer of the OCS Bridge. Contains executors for all commands
 * supported by the OCSBridge.
 *
 * @author The LSST CCS Team
 */
public class OCSBridge {

    private static final Logger LOG = Logger.getLogger(OCSBridge.class.getName());
    private final static DelimitedStringSplitJoin DELIMITED_STRING_SPLIT_JOIN = new DelimitedStringSplitJoin();

    private final CCS ccs;
    private final State lse209State;
    private final State offlineState;
    private OCSCommandExecutor ocsCommandExecutor;
    private CCSCommandExecutor ccsCommandExecutor;
    private final MCMLayer mcm;
    static final int DEFAULT_EVENT_PRIORITY = 1;
    private BlockingQueue<CCSImageNameEvent> imageNameTransfer = new ArrayBlockingQueue<>(1);
    private CCSImageNameEvent imageNameEvent;
    private EndOfImageTelemetryEvent endOfImageTelemetryEvent;
    private final OCSBridgeConfig config;

    private final State commandState;

    private final OCSCommandConverter ocsCcsCommandConverter = new OCSCommandConverter();
    private final OCSStateEventConverter ocsCcsStateEventConverter = new OCSStateEventConverter();
    private final OCSStateChangeToEventConverter ocsStateChangeToEventConverter = new OCSStateChangeToEventConverter();
    private CCSTimeStamp openTime;
    private final TelemetrySender sender;

    OCSBridge(OCSBridgeConfig config, CCS ccs, MCMLayer mcm) {

        this.config = config;
        this.ccs = ccs;
        this.mcm = mcm;
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        lse209State = new State(SummaryState.OFFLINE);
        ccs.getAggregateStatus().add(now, lse209State);
        offlineState = new State(OfflineState.OFFLINE_PUBLISH_ONLY);
        ccs.getAggregateStatus().add(now, offlineState);
        commandState = new State(CCSCommandState.IDLE);
        ccs.getAggregateStatus().add(now, commandState);
        ocsCommandExecutor = new OCSCommandExecutor(this);
        ccsCommandExecutor = new CCSCommandExecutor();
        sender = TelemetrySender.create(config.getDevice(), ocsCommandExecutor, ccs.getScheduler());
        
        SALHeartbeatGenerator shg = new SALHeartbeatGenerator(this);
        shg.start();

        mcm.addStateChangeListener((CCSTimeStamp when, Enum state, Enum oldState) -> {

            // There has to be a better way to do this but this is what I can implement for now - FA
            // Rafts related events
            if ((oldState == RaftsState.CLEARING || oldState == RaftsState.QUIESCENT) && state == RaftsState.INTEGRATING) {
                ccs.runInBackground(() -> {
                    try {
                        // We rendezvous the imageNameEvent from the MCM with the state change here
                        long start = System.currentTimeMillis(); // NOTE: this timestamp is only used for timing, not sent to SAL
                        imageNameEvent = imageNameTransfer.poll(2, TimeUnit.SECONDS);
                        long waitTime = System.currentTimeMillis() - start;
                        if (imageNameEvent == null) {
                            // TODO: Something better than just throw an exception
                            throw new RuntimeException("Image name could not be retrieved");
                        } else if (waitTime > 500) {
                            LOG.log(Level.WARNING, "Waited {0} for imageNameEvent", waitTime);
                        }

                        final ImageName imageName = imageNameEvent.getImageName();
                        StartIntegrationEvent sie = StartIntegrationEvent.builder()
                                .priority(DEFAULT_EVENT_PRIORITY)
                                .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                                .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                                .imagesInSequence(imageNameEvent.getImagesInSequence())
                                .imageIndex(imageNameEvent.getSequenceNumber())
                                .imageName(imageName.toString())
                                .imageNumber(imageName.getNumber())
                                .imageSource(imageName.getSource().getCode())
                                .imageController(imageName.getController().getCode())
                                .imageDate(imageName.getDateString())
                                .timestampAcquisitionStart(imageNameEvent.getIntegrationStartTime().getTAIDouble())
                                .exposureTime(imageNameEvent.getExposureTime())
                                .build();
                        ocsCommandExecutor.sendEvent(sie);
//                        ocsCommandExecutor.sendEvent(new ImageReadoutParametersEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageName().toString(), config.ccdNames(), config.ccdType(),
//                                config.overRows(), config.overCols(), config.readRows(), config.readCols(), config.readCols2(), config.preCols(), config.preRows(), config.postCols()));
                    } catch (RuntimeException | InterruptedException ex) {
                        LOG.log(Level.SEVERE, "Error sending StartRaftIntegrationEvent", ex);
                    }
                });
            }

            if (oldState == RaftsState.INTEGRATING && state == RaftsState.READING_OUT) {
                final ImageName imageName = imageNameEvent.getImageName();
                StartReadoutEvent sre = StartReadoutEvent.builder()
                        .priority(DEFAULT_EVENT_PRIORITY)
                        .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                        .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                        .imagesInSequence(imageNameEvent.getImagesInSequence())
                        .imageIndex(imageNameEvent.getSequenceNumber())
                        .imageName(imageName.toString())
                        .imageNumber(imageName.getNumber())
                        .imageSource(imageName.getSource().getCode())
                        .imageController(imageName.getController().getCode())
                        .imageDate(imageName.getDateString())
                        .timestampAcquisitionStart(imageNameEvent.getIntegrationStartTime().getTAIDouble())
                        .timestampStartOfReadout(when.getTAIDouble())
                        .exposureTime(imageNameEvent.getExposureTime())
                        .build();

                ocsCommandExecutor.sendEvent(sre);
            }

            if (oldState == RaftsState.READING_OUT) {
                final ImageName imageName = imageNameEvent.getImageName();
                EndReadoutEvent ere = EndReadoutEvent.builder()
                        .priority(DEFAULT_EVENT_PRIORITY)
                        .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                        .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                        .imagesInSequence(imageNameEvent.getImagesInSequence())
                        .imageIndex(imageNameEvent.getSequenceNumber())
                        .imageName(imageName.toString())
                        .imageNumber(imageName.getNumber())
                        .imageSource(imageName.getSource().getCode())
                        .imageController(imageName.getController().getCode())
                        .imageDate(imageName.getDateString())
                        .timestampAcquisitionStart(imageNameEvent.getIntegrationStartTime().getTAIDouble())
                        .timestampEndOfReadout(when.getTAIDouble())
                        .requestedExposureTime(imageNameEvent.getExposureTime())
                        .build();
                ocsCommandExecutor.sendEvent(ere);
                // FIXME: feeble attempt to make sure we get the right EndOfImageTelemetryEvent. We should do something more robust 
                final EndOfImageTelemetryEvent endOfImageTelemetryEventAtEndReadout = endOfImageTelemetryEvent;
                //For now we will always send the EndOfImageTelemetry event 300ms after the EndReadoutEvent
                ccs.schedule(Duration.ofMillis(300), () -> {
                    if (endOfImageTelemetryEventAtEndReadout != null) {
                        ocsCommandExecutor.sendEvent(endOfImageTelemetryEventAtEndReadout);
                    }
                });
            }
            CameraEvent converted = ocsStateChangeToEventConverter.convert(when, oldState, state, DEFAULT_EVENT_PRIORITY);
            if (converted != null) {
                ocsCommandExecutor.sendEvent(converted);
            }
            // Note, ATS shutter transitions directly from OPEN->CLOSED, without the intervening CLOSING state.
            if (state == ShutterState.OPEN) {
                //TODO: The time should be coming from the state change event. which
                // should be generated by the shutter system itself.
                openTime = when;
            }
            if ((oldState == ShutterState.CLOSING || oldState == ShutterState.OPEN) && state == ShutterState.CLOSED) {
                //TODO: The time should be coming from the state change event. which
                // should be generated by the shutter system itself.
                CCSTimeStamp closeTime = when;
                ocsCommandExecutor.sendEvent(new ShutterMotionProfileEvent(1, (closeTime.getTAIDouble() - openTime.getTAIDouble()) / 1000.));
            }
        });
        mcm.addEventListener((CCSEvent event) -> {
            if (event instanceof CCSAvailableFiltersEvent) {
                List<String> filters = ((CCSAvailableFiltersEvent) event).getAvailableFilters();
                ocsCommandExecutor.sendEvent(new AvailableFiltersEvent(DEFAULT_EVENT_PRIORITY, DELIMITED_STRING_SPLIT_JOIN.join(filters)));
            } else if (event instanceof CCSImageNameEvent) {
                LOG.log(Level.INFO, "CCSImageNameEvent event {0}", event);
                imageNameTransfer.add((CCSImageNameEvent) event);
            } else if (event instanceof CCSSetFilterEvent) {
                CCSSetFilterEvent setFilter = (CCSSetFilterEvent) event;
                if (setFilter.isStart()) {
                    ocsCommandExecutor.sendEvent(new StartSetFilterEvent(DEFAULT_EVENT_PRIORITY, setFilter.getFilterName(), setFilter.getFilterType()));
                } else {
                    ocsCommandExecutor.sendEvent(new EndSetFilterEvent(DEFAULT_EVENT_PRIORITY, setFilter.getFilterName(), setFilter.getFilterType(), setFilter.getSlot(), setFilter.getPosition()));
                }
            } else if (event instanceof CCSSettingsAppliedEvent) {
                String otherSettingsEvents = "FocalPlaneHardwareIdSettingsApplied"; // FIXME: should not be hardwired
                ocsCommandExecutor.sendEvent(new SettingsAppliedEvent(DEFAULT_EVENT_PRIORITY, ((CCSSettingsAppliedEvent) event).getSettings(), otherSettingsEvents));
            }
        });

        mcm.addStatusMessageListener((StatusMessage msg) -> {
            if (msg instanceof StatusConfigurationInfo) {
                StatusConfigurationInfo sci = (StatusConfigurationInfo) msg;
                final AgentInfo originAgentInfo = msg.getOriginAgentInfo();
                final String subsystemName = originAgentInfo == null ? "unknown" : originAgentInfo.getName();
                ConfigurationInfo info = sci.getConfigurationInfo();
                if (originAgentInfo != null && "FOCAL_PLANE".equals(originAgentInfo.getAgentProperty("agentCategory"))) {
                    SummaryInfoConverter converter = new SummaryInfoConverter(config.getDevice());
                    ocsCommandExecutor.sendEvent(converter.convert(info));
                    FocalPlaneHardwareIdConverter converter2 = new FocalPlaneHardwareIdConverter(config.getDevice());
                    ocsCommandExecutor.sendEvent(converter2.convert(info));
                } else {
                    LOG.log(Level.WARNING, "Got unhandled config from {0}", subsystemName);
                }
            } else if (msg instanceof StatusSubsystemData) {
                StatusSubsystemData ssd = (StatusSubsystemData) msg;
                final AgentInfo originAgentInfo = msg.getOriginAgentInfo();
                final String subsystemName = originAgentInfo == null ? "unknown" : originAgentInfo.getName();
                LOG.log(Level.FINE, "Received from {0} {1}", new Object[]{subsystemName, ssd.getDataKey()});
                LOG.log(Level.FINE, "Data {0}", new Object[]{ssd.getSubsystemData().getValue().toString()});
                if (ssd.getSubsystemData().getValue() instanceof org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent) {
                    org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent data = (org.lsst.ccs.subsystem.focalplane.data.ImageReadoutParametersEvent) ssd.getSubsystemData().getValue();
                    List<String> ccdNames = new ArrayList<>();
                    List<org.lsst.sal.camera.event.ImageReadoutParametersEvent.CcdType> ccdTypes = new ArrayList<>();
                    List<Integer> overCols = new ArrayList<>();
                    List<Integer> overRows = new ArrayList<>();
                    List<Integer> postCols = new ArrayList<>();
                    List<Integer> postRows = new ArrayList<>();
                    List<Integer> preCols = new ArrayList<>();
                    List<Integer> preRows = new ArrayList<>();
                    List<Integer> readCols = new ArrayList<>();
                    List<Integer> readCols2 = new ArrayList<>();
                    List<Integer> readRows = new ArrayList<>();
                    List<Integer> underCols = new ArrayList<>();
                    int index = 0;
                    for (Location location : data.getLocations()) {
                        int nSensors = location.type().getCCDCount();
                        for (int i = 0; i < nSensors; i++) {
                            ccdNames.add(location.getRaftName() + (config.getDevice() == OCSBridgeConfig.Device.AUXTEL ? "S00" : location.getSensorName(i)));
                            ccdTypes.add(org.lsst.sal.camera.event.ImageReadoutParametersEvent.CcdType.valueOf(data.getCcdType()[index].toUpperCase()));
                            overCols.add(data.getOverCols()[index]);
                            overRows.add(data.getOverRows()[index]);
                            postCols.add(data.getPostCols()[index]);
                            postRows.add(data.getPostRows()[index]);
                            preCols.add(data.getPreCols()[index]);
                            preRows.add(data.getPreRows()[index]);
                            readCols.add(data.getReadCols()[index]);
                            readCols2.add(data.getReadCols2()[index]);
                            readRows.add(data.getReadRows()[index]);
                            underCols.add(data.getUnderCols()[index]);
                        }
                        index++;
                    }
                    CameraEvent irpe;
                    if (config.getDevice() == OCSBridgeConfig.Device.AUXTEL) {
                        irpe = org.lsst.sal.atcamera.event.ImageReadoutParametersEvent.builder()
                                .imageName(data.getImageName().toString())
                                .ccdLocation(DELIMITED_STRING_SPLIT_JOIN.join(ccdNames))
                                .raftBay(ccdNames.stream().map(name -> name.substring(0, 3)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                .ccdSlot(ccdNames.stream().map(name -> name.substring(3, 6)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                .ccdType(org.lsst.sal.atcamera.event.ImageReadoutParametersEvent.CcdType.ITL)
                                .overRows(overRows.get(0))
                                .overCols(overCols.get(0))
                                .readRows(readRows.get(0))
                                .readCols(readCols.get(0))
                                .readCols2(readCols2.get(0))
                                .preCols(preCols.get(0))
                                .preRows(preRows.get(0))
                                .postCols(postCols.get(0))
                                .underCols(underCols.get(0))
                                .daqFolder(data.getDaqFolder())
                                .daqAnnotation(data.getAnnotation())
                                .build();
                    } else {
                        irpe = org.lsst.sal.camera.event.ImageReadoutParametersEvent.builder()
                                .imageName(data.getImageName().toString())
                                .ccdLocation(DELIMITED_STRING_SPLIT_JOIN.join(ccdNames))
                                .raftBay(ccdNames.stream().map(name -> name.substring(0, 3)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                .ccdSlot(ccdNames.stream().map(name -> name.substring(3, 6)).collect(DELIMITED_STRING_SPLIT_JOIN.joining()))
                                .ccdType(ccdTypes.stream().toArray(org.lsst.sal.camera.event.ImageReadoutParametersEvent.CcdType[]::new))
                                .overRows(overRows.stream().mapToInt(i -> i).toArray())
                                .overCols(overCols.stream().mapToInt(i -> i).toArray())
                                .readRows(readRows.stream().mapToInt(i -> i).toArray())
                                .readCols(readCols.stream().mapToInt(i -> i).toArray())
                                .readCols2(readCols2.stream().mapToInt(i -> i).toArray())
                                .preCols(preCols.stream().mapToInt(i -> i).toArray())
                                .preRows(preRows.stream().mapToInt(i -> i).toArray())
                                .postCols(postCols.stream().mapToInt(i -> i).toArray())
                                .underCols(underCols.stream().mapToInt(i -> i).toArray())
                                .daqFolder(data.getDaqFolder())
                                .daqAnnotation(data.getAnnotation())
                                .build();
                    }

                    ocsCommandExecutor.sendEvent(irpe);
                } else if (ssd.getSubsystemData().getValue() instanceof ImageMetaDataEvent) {
                    ImageMetaDataEvent data = (ImageMetaDataEvent) ssd.getSubsystemData().getValue();
                    // TODO: How do we know we are using the correct imageNameEvent?
                    endOfImageTelemetryEvent = EndOfImageTelemetryEvent.builder()
                            .priority(DEFAULT_EVENT_PRIORITY)
                            .additionalKeys(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().keySet()))
                            .additionalValues(DELIMITED_STRING_SPLIT_JOIN.join(imageNameEvent.getKeyValueData().values()))
                            .imagesInSequence(imageNameEvent.getImagesInSequence())
                            .imageIndex(imageNameEvent.getSequenceNumber())
                            .imageName(data.getImageName().toString())
                            .imageSource(data.getImageName().getSource().getCode())
                            .imageController(data.getImageName().getController().getCode())
                            .imageDate(data.getImageName().getDateString())
                            .imageNumber(data.getImageName().getNumber())
                            .timestampAcquisitionStart(data.getLastClearTime().getTAIDouble())
                            // FIXME: The next two lines are only correct when no shutter motion
                            .timestampDateEnd(data.getIntegrationEndTime().getTAIDouble())
                            .timestampDateObs(data.getIntegrationStartTime().getTAIDouble())
                            .darkTime(data.getDarkTime())
                            .imageTag(data.getDaqTag())
                            // For now we are using the requested exposure time for both. This should
                            // come from the shutter motion profile
                            .exposureTime(imageNameEvent.getExposureTime())
                            .measuredShutterOpenTime(imageNameEvent.getExposureTime())
                            .build();
                    //ocsCommandExecutor.sendEvent(event);
                } else if (ssd.getSubsystemData().getValue() instanceof List) {
                    sender.send(ssd);
                } else if (!(ssd.getSubsystemData().getValue() instanceof ComCamSharedVacState)) {
                    LOG.log(Level.WARNING, "Got unhandled telemetry from {0} of class {1}", new Object[]{subsystemName, ssd.getSubsystemData().getValue()});
                }
            }
        });

        ccs.addStateChangeListener((when, currentState, oldState) -> {
            int priority = DEFAULT_EVENT_PRIORITY;

            CameraStateChangeEvent converted = ocsCcsStateEventConverter.convert(when, currentState, priority);
            if (converted == null) {
                LOG.log(Level.WARNING, "Enum {0} of class {1} could not be converted to equivalent OCS state", new Object[]{currentState, currentState.getClass()});
            } else {
                ocsCommandExecutor.sendEvent(converted);
            }
        });
    }

    State<CCSCommandState> getCommandState() {
        return commandState;
    }

    public OCSBridgeConfig getConfig() {
        return config;
    }

    /**
     * Allow a user to provide an alternative implementation of the
     * OCSCommandExecutor. Used to override the default OCSCommandExecutor with
     * one that actually sends acknowledgments back to OCS.
     *
     * @param ocs
     */
    void setOCSCommandExecutor(OCSCommandExecutor ocs) {
        this.ocsCommandExecutor = ocs;
        this.sender.setSender(ocs);
        // Publish the current state
        CCSTimeStamp now = CCSTimeStamp.currentTime();
        ccs.getAggregateStatus().getStates().stream()
                .map((s) -> ocsCcsStateEventConverter.convert(now, s.getState(), DEFAULT_EVENT_PRIORITY))
                .filter(Objects::nonNull)
                .forEach((converted) -> ocsCommandExecutor.sendEvent(converted));
    }

    public OCSCommandExecutor getOcsCommandExecutor() {
        return ocsCommandExecutor;
    }

    void setCCSCommandExecutor(CCSCommandExecutor ccs) {
        this.ccsCommandExecutor = ccs;
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        OCSBridge ocs = createOCSBridge();
        ToyOCSGUI gui = new ToyOCSGUI(new GUIDirectLayer(ocs), ocs.getConfig().getDevice());
        gui.setVisible(true);

    }

    /**
     * Create an OCSBridge with default configuration
     *
     * @return
     */
    static OCSBridge createOCSBridge() {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig();
        MCMConfig mcmConfig = MCMConfig.createDefaultConfig();
        ImageNameService ins = ImageNameService.testInstance("jdbc:h2:mem:test;MODE=MYSQL", Controller.OCS, Source.MainCamera, "UTC", Duration.ofHours(12), Instant.parse("2022-01-01T00:00:00.00Z"));
        return createOCSBridge(ocsConfig, mcmConfig, ins);
    }

    static OCSBridge createOCSBridge(Camera camera) {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig();
        MCMConfig mcmConfig = new MCMConfig(camera);
        ImageNameService ins = ImageNameService.testInstance("jdbc:h2:mem:test;MODE=MYSQL", Controller.OCS, camera == Camera.AUXTEL ? Source.AuxTel : Source.MainCamera, "UTC", Duration.ofHours(12), Instant.parse("2022-01-01T00:00:00.00Z"));
        return createOCSBridge(ocsConfig, mcmConfig, ins);
    }

    static OCSBridge createOCSBridge(OCSBridgeConfig config, MCMConfig mcmConfig, ImageNameService imageNamingService) {
        CCS ccs = new CCS();
        MCMLayer mcm = new MCMDirectLayer(new MCM(ccs, mcmConfig, imageNamingService));
        return new OCSBridge(config, ccs, mcm);
    }

    State getLse209State() {
        return lse209State;
    }

    State getOfflineState() {
        return offlineState;
    }

    /**
     * Super class for all OCSExecutors which forward commands to MCM
     */
    class ForwardToMCMExecutor extends OCSCommandExecutor.OCSExecutor {

        private final CCSCommand ccsCommand;
        private CCSCommandResponse response;
        private final SummaryState initialState;

        ForwardToMCMExecutor(CameraCommand command) {
            this(command, SummaryStateEvent.SummaryState.ENABLED);
        }

        ForwardToMCMExecutor(CameraCommand command, SummaryState initialState) {
            super(command);
            this.ccsCommand = ocsCcsCommandConverter.convert(command);
            this.initialState = initialState;
        }

        @Override
        Duration testPreconditions() throws OCSCommandExecutor.PreconditionsNotMet {
            if (!lse209State.isInState(initialState)) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command not accepted in: " + lse209State);
            }
            CCSAckOrNack can = forwardCommand();
            if (can.isNack()) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command rejected: " + can.getReason());
            } else {
                return can.getDuration();
            }
        }

        /**
         * This is somewhat illogically executed from testPreconditions to give
         * the MCM the chance to reject the command. Override this if there is
         * anything that needs to be done before the command executes.
         *
         * @return
         */
        protected CCSAckOrNack forwardCommand() {
            response = mcm.execute(ccsCommand);
            return response.waitForAckOrNack();
        }

        @Override
        void execute() throws Exception {
            response.waitForCompletion();
        }
    }

    // 
    void execute(CameraCommand cmd) {
        if (cmd instanceof DisableCommand) {
            execute((DisableCommand) cmd);
        } else if (cmd instanceof EnableCommand) {
            execute((EnableCommand) cmd);
        } else if (cmd instanceof ExitControlCommand) {
            execute((ExitControlCommand) cmd);
        } else if (cmd instanceof EnterControlCommand) {
            execute((EnterControlCommand) cmd);
            // The next three commands override the execute method of the parent class 
            // and have to be treated as special cases and cannot be generalised to
            // be dealt with via ForwardToMCMExecutor followed by execute - we 
            // need to generalise these differently - coming soon - Farrukh 25/6/2018
        } else if (cmd instanceof TakeImagesCommand) {
            execute((TakeImagesCommand) cmd);
        } else if (cmd instanceof StartCommand) {
            execute((StartCommand) cmd);
        } else if (cmd instanceof StartImageCommand) {
            execute((StartImageCommand) cmd);
        } else if (cmd instanceof StandbyCommand) {
            execute((StandbyCommand) cmd);
        } // rest of camera commands are generalizeable in the following manner - Farrukh 25/6/2018
        else {
            ForwardToMCMExecutor fcm = new ForwardToMCMExecutor(cmd);
            ocsCommandExecutor.executeCommand(fcm);
        }
        // were throwing a run time exception before hopefully gets thrown via ocsCommandExecutor
        /*  
        else {
            throw new RuntimeException("Unrecognized command " + cmd);
        }
         */
    }

    void execute(TakeImagesCommand command) {
        OCSExecutor takeImages = new TakeImagesExecutor(command);
        ocsCommandExecutor.executeCommand(takeImages);
    }

    void execute(StartImageCommand command) {
        OCSExecutor startImage = new StartImageExecutor(command);
        ocsCommandExecutor.executeCommand(startImage);
    }

    void execute(EnterControlCommand command) {
        OCSExecutor takeControl = new EnterControlExecutor(command);
        ocsCommandExecutor.executeCommand(takeControl);
    }

    void execute(ExitControlCommand command) {
        OCSExecutor exit = new ExitExecutor(command);
        ocsCommandExecutor.executeCommand(exit);
    }

    void execute(StartCommand command) {
        OCSExecutor start = new StartExecutor(command);
        ocsCommandExecutor.executeCommand(start);
    }

    void execute(StandbyCommand command) {
        OCSExecutor standby = new StandbyExecutor(command);
        ocsCommandExecutor.executeCommand(standby);
    }

    void execute(EnableCommand command) {
        OCSExecutor enable = new EnableExecutor(command);
        ocsCommandExecutor.executeCommand(enable);
    }

    void execute(DisableCommand command) {
        OCSExecutor disable = new DisableExecutor(command);
        ocsCommandExecutor.executeCommand(disable);
    }

    void execute(CCSCommand command) {
        if (command instanceof CCSSetAvailableCommand) {
            execute((CCSSetAvailableCommand) command);
        } else if (command instanceof CCSRevokeAvailableCommand) {
            execute((CCSRevokeAvailableCommand) command);
        } else if (command instanceof CCSSimulateFaultCommand) {
            execute((CCSSimulateFaultCommand) command);
        } else if (command instanceof CCSClearFaultCommand) {
            execute((CCSClearFaultCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    void execute(CCSSetAvailableCommand command) {
        CCSExecutor setAvailable = new SetAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(setAvailable));
    }

    void execute(CCSRevokeAvailableCommand command) {
        CCSExecutor revokeAvailable = new RevokeAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(revokeAvailable));
    }

    void execute(CCSSimulateFaultCommand command) {
        CCSExecutor simulateFault = new SimulateFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(simulateFault));
    }

    void execute(CCSClearFaultCommand command) {
        CCSExecutor clearFault = new ClearFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(clearFault));
    }

    // Utility only used for testing   
    public Future<Void> waitForState(Enum state) {
        return ccs.waitForStatus(state);
    }

    CCS getCCS() {
        return ccs;
    }

    class TakeImagesExecutor extends ForwardToMCMExecutor {

        public TakeImagesExecutor(TakeImagesCommand command) {
            super(command);
        }

        @Override
        protected CCSAckOrNack forwardCommand() {
            imageNameTransfer.clear();
            return super.forwardCommand();
        }
    }

    class StartImageExecutor extends ForwardToMCMExecutor {

        public StartImageExecutor(StartImageCommand command) {
            super(command);

        }

        @Override
        protected CCSAckOrNack forwardCommand() {
            imageNameTransfer.clear();
            return super.forwardCommand();
        }
    }

    class EnterControlExecutor extends OCSExecutor {

        public EnterControlExecutor(EnterControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_AVAILABLE)) {
                throw new PreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: The argument should be a comma-delimited list of supported settings. Clearly should not be hard-wired here.
            ocsCommandExecutor.sendEvent(new SettingVersionsEvent(DEFAULT_EVENT_PRIORITY, "1", "Normal", "ccs:///configuration-service"));
            lse209State.setState(SummaryState.STANDBY);
        }
    }

    class ExitExecutor extends OCSExecutor {

        public ExitExecutor(ExitControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.STANDBY)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(SummaryState.OFFLINE);
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }
    }

    /**
     * Start command is a special case, since it is a lifecycle command, but
     * must also issue a start command to the MCM (to set configuration).
     */
    class StartExecutor extends ForwardToMCMExecutor {

        public StartExecutor(StartCommand command) {
            super(command, SummaryState.STANDBY);
        }

        @Override
        void execute() throws Exception {
            super.execute();
            ocsCommandExecutor.sendEvent(new AppliedSettingsMatchStartEvent(DEFAULT_EVENT_PRIORITY, true));

            lse209State.setState(SummaryState.DISABLED);
        }
    }

    class StandbyExecutor extends OCSExecutor {

        public StandbyExecutor(StandbyCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standy command if things are happening?
            //TODO: or wait until things finish and return then?
            //TODO: The argument should be a comma-delimited list of supported settings. Clearly should not be hard-wired here.
            ocsCommandExecutor.sendEvent(new SettingVersionsEvent(DEFAULT_EVENT_PRIORITY, "Normal", "", "" /*FIXME*/));
            lse209State.setState(SummaryState.STANDBY);
        }
    }

    class EnableExecutor extends OCSExecutor {

        public EnableExecutor(EnableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(SummaryState.ENABLED);
        }
    }

    class DisableExecutor extends OCSExecutor {

        public DisableExecutor(DisableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.ENABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            // Fixme: Can we reject the disable command if we are busy?
            // What about if we are not idle?
            // Note logic here is incorrect according to Paul Lotz, we must always accept
            // the disable command.
//            if (startImageTimeout != null && !startImageTimeout.isDone()) {
//                throw new PreconditionsNotMet("Exposure in progress");
//            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standby command if things are happening?
            //TODO: or wait until things finish and return then?
            lse209State.setState(SummaryState.DISABLED);
        }
    }

    class SetAvailableExecutor extends CCSExecutor {

        private SetAvailableExecutor(CCSSetAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_PUBLISH_ONLY)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.OFFLINE_AVAILABLE);
        }

    }

    class RevokeAvailableExecutor extends CCSExecutor {

        private RevokeAvailableExecutor(CCSRevokeAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_AVAILABLE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }

    }

    class SimulateFaultExecutor extends CCSExecutor {

        private SimulateFaultExecutor(CCSSimulateFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            //TODO: Should we also attempt to stop the subsystems?
            lse209State.setState(SummaryState.FAULT);
        }

    }

    class ClearFaultExecutor extends CCSExecutor {

        private ClearFaultExecutor(CCSClearFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.FAULT)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            lse209State.setState(SummaryState.OFFLINE);
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }
    }
}
