package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.util.List;
import java.util.concurrent.ExecutionException;
import org.lsst.ccs.bus.annotations.SkipEncoding;

/**
 * The filter changer used by the MCM
 *
 * @author tonyj
 */
public class FilterChanger {

    public enum FilterState {

        ROTATING, UNLOADED, LOADED, UNLOADING, LOADING
    }

    private List<String> installedFilters;
    private final CCS ccs;
    private FilterChangerInterface filterChanger;
    private final MCMConfig config;

    FilterChanger(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        this.filterChanger = new FilterChangerSimulation(ccs);
    }

    public void start(String configName) throws ExecutionException {
        filterChanger.start(configName);
        // filterChanger is locked at this point, so installed filters cannot change
        installedFilters = filterChanger.getInstalledFilters();
        ccs.fireEvent(new CCSAvailableFiltersEvent(installedFilters));
    }

    boolean filterIsAvailable(String filter) {
        return installedFilters.contains(filter);
    }

    List<String> getAvailableFilters() {
        return installedFilters;
    }

    void setFilter(String filter) throws ExecutionException {
        filterChanger.setFilter(filter);
    }

    Duration getEstimatedTimeForChange(String filterName) {
        return filterChanger.getEstimatedDurationForFilterChange(filterName);
    }

    void registerMCMSubsystem(MCMSubsystem mcm) {
        filterChanger = new FilterChangerComCamSubsystemLayer(mcm, ccs, config);
    }

    @SkipEncoding
    public static class CCSAvailableFiltersEvent extends CCSEvent {

        private final List<String> availableFilters;

        private CCSAvailableFiltersEvent(List<String> availableFilters) {
            this.availableFilters = availableFilters;
        }

        public List<String> getAvailableFilters() {
            return availableFilters;
        }

        @Override
        public String toString() {
            return "CCSAvailableFiltersEvent{" + "availableFilters=" + availableFilters + '}';
        }

    }
}
