package org.lsst.ccs.subsystem.ocsbridge;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Base class for all CCSCommands. This class is abstract and is extended by
 * specific implementations for each supported command. Note that this class
 * only encapsulates the command and its arguments, it does not include any code
 * for actually executing the command.
 *
 * @author tonyj
 */
public abstract class CCSCommand {

    /**
     * Gets the command name.
     *
     * @return The command name.
     */
    abstract String getCommand();

    /**
     * Gets the command arguments.
     *
     * @return The command arguments.
     */
    abstract List<Object> getArguments();

    @Override
    public String toString() {
        return "CCSCommand: " + getCommand() +" "+ getArguments().stream().map(Object::toString).collect(Collectors.joining(" "));
    }

    public static class CCSInitImageCommand extends CCSCommand {

        private final double deltaT;

        public CCSInitImageCommand() {
            this.deltaT = 0;
        }

        public CCSInitImageCommand(double deltaT) {
            this.deltaT = deltaT;
        }

        public double getDeltaT() {
            return deltaT;
        }

        @Override
        String getCommand() {
            return "initImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(deltaT);
        }

    }

    public static class CCSTakeImagesCommand extends CCSCommand {

        private final int numImages;
        private final double expTime;
        private final boolean shutter;
        private final String sensors;
        private final String obsNote;
        private final String keyValueMap;

        public CCSTakeImagesCommand() {
            this.numImages = 1;
            this.expTime = 1;
            this.shutter = false;
            this.sensors = "";
            this.obsNote = "";
            this.keyValueMap = "";
        }

        public CCSTakeImagesCommand(double expTime, int numImages, boolean shutter, String sensors, String keyValueMap, String obsNote) {
            this.numImages = numImages;
            this.expTime = expTime;
            this.shutter = shutter;
            this.sensors = sensors;
            this.keyValueMap = keyValueMap;
            this.obsNote = obsNote;
        }

        public int getNumImages() {
            return numImages;
        }

        public double getExpTime() {
            return expTime;
        }

        public boolean isShutter() {
            return shutter;
        }

        public String getSensors() {
            return sensors;
        }

        public String getObsNote() {
            return obsNote;
        }

        public String getKeyValueMap() {
            return keyValueMap;
        }

        @Override
        String getCommand() {
            return "takeImages";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(expTime, numImages, shutter, sensors, keyValueMap, obsNote);
        }

        @Override
        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append("CCSTakeImagesCommand{numImages=").append(numImages);
            sb.append(", expTime=").append(expTime);
            sb.append(", shutter=").append(shutter);
            sb.append(", sensors=").append(sensors);
            sb.append(", obsNote=").append(obsNote);
            sb.append(", keyValueMap=").append(keyValueMap);
            sb.append('}');
            return sb.toString();
        }
    }

    public static class CCSSetFilterCommand extends CCSCommand {

        private final String name;

        public CCSSetFilterCommand() {
            this.name = "Empty Constructor Default";
        }

        public CCSSetFilterCommand(String filterName) {
            this.name = filterName;
        }

        public String getFilterName() {
            return name;
        }

        @Override
        String getCommand() {
            return "setFilter";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(name);
        }
    }

    public static class CCSInitGuidersCommand extends CCSCommand {

        private final String roiSpec;

        /**
         *
         */
        public CCSInitGuidersCommand() {
            this.roiSpec = "will be overwritten";
        }

        public CCSInitGuidersCommand(String roiSpec) {
            this.roiSpec = roiSpec;
        }

        public String getRoiSpec() {
            return roiSpec;
        }

        @Override
        String getCommand() {
            return "initGuiders";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(roiSpec);
        }
    }

    public static class CCSStartImageCommand extends CCSCommand {

        private final boolean shutter;
        private final double timeout;
        private final String sensors;
        private final String keyValueMap;
        private final String obsNote;

        public CCSStartImageCommand() {
            this.shutter = false;
            this.sensors = "";
            this.keyValueMap = "";
            this.obsNote = "";
            this.timeout = 1.;
        }

        public CCSStartImageCommand(boolean shutter, String sensors, String keyValueMap, String obsNote, double timeout) {
            this.shutter = shutter;
            this.sensors = sensors;
            this.keyValueMap = keyValueMap;
            this.obsNote = obsNote;
            this.timeout = timeout;
        }

        public boolean isShutterOpen() {
            return shutter;
        }

        public String getSensors() {
            return sensors;
        }

        public String getKeyValueMap() {
            return keyValueMap;
        }

        public String getObsNote() {
            return obsNote;
        }

        public double getTimeout() {
            return timeout;
        }

        @Override
        String getCommand() {
            return "startImage";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(shutter, sensors, keyValueMap, obsNote, timeout);
        }

        @Override
        public String toString() {
            return "CCSStartImageCommand{" + "shutter=" + shutter + ", timeout=" + timeout + ", sensors=" + sensors + ", keyValueMap=" + keyValueMap + ", annotation=" + obsNote + '}';
        }
    }

    public static class CCSStartCommand extends CCSCommand {

        private final String settingsToApply;

        public CCSStartCommand() {

            this.settingsToApply = "will be oerwritten";
        }

        public CCSStartCommand(String configuration) {
            this.settingsToApply = configuration;
        }

        public String getConfiguration() {
            return settingsToApply;
        }

        @Override
        public String toString() {
            return "CCSStartCommand{" + "configuration=" + settingsToApply + '}';
        }

        @Override
        String getCommand() {
            return "start";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(settingsToApply);
        }
    }

    static abstract class NoArgCCSCommand extends CCSCommand {

        private String commandName;

        NoArgCCSCommand(String commandName) {
            this.commandName = commandName;
        }

        @Override
        String getCommand() {
            return commandName;
        }

        @Override
        List<Object> getArguments() {
            return Collections.emptyList();
        }
    }

    public static class CCSEnableCalibrationCommand extends NoArgCCSCommand {

        public CCSEnableCalibrationCommand() {
            super("enableCalibration");
        }

    }

    public static class CCSDisableCalibrationCommand extends NoArgCCSCommand {

        public CCSDisableCalibrationCommand() {
            super("disableCalibration");
        }

    }

    public static class CCSEndImageCommand extends NoArgCCSCommand {

        public CCSEndImageCommand() {
            super("endImage");
        }

    }

    public static class CCSSetAvailableCommand extends NoArgCCSCommand {

        public CCSSetAvailableCommand() {
            super("setAvailable");
        }
    }

    public static class CCSRevokeAvailableCommand extends NoArgCCSCommand {

        public CCSRevokeAvailableCommand() {
            super("revokeAvailable");
        }
    }

    public static class CCSSimulateFaultCommand extends NoArgCCSCommand {

        public CCSSimulateFaultCommand() {
            super("simulateFault");
        }
    }

    public static class CCSClearFaultCommand extends NoArgCCSCommand {

        public CCSClearFaultCommand() {
            super("clearFault");
        }
    }

    public static class CCSDiscardRowsCommand extends CCSCommand {

        private final int nRows;

        public CCSDiscardRowsCommand() {

            this.nRows = 0;
        }

        public CCSDiscardRowsCommand(int nRows) {
            this.nRows = nRows;
        }

        public int getnRows() {
            return nRows;
        }

        @Override
        String getCommand() {
            return "discardRows";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(nRows);
        }
    }

    public static class CCSClearCommand extends CCSCommand {

        private final int nClears;

        public CCSClearCommand() {
            this.nClears = 0;
        }

        public CCSClearCommand(int nClears) {
            this.nClears = nClears;
        }

        public int getNClears() {
            return nClears;
        }

        @Override
        String getCommand() {
            return "clear";
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(nClears);
        }

    }

    public static class CCSPlayCommand extends CCSCommand {

        private final String playlist;
        private final boolean repeat;

        public CCSPlayCommand() {
            this.playlist = "";
            this.repeat = false;
        }

        public CCSPlayCommand(String playlist, boolean repeat) {
            this.playlist = playlist;
            this.repeat = repeat;
        }

        public String getPlaylist() {
            return playlist;
        }

        public boolean isRepeat() {
            return repeat;
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(playlist, repeat);
        }

        @Override
        String getCommand() {
            return "play";
        }
    }

    public static class CCSDefinePlaylistCommand extends CCSCommand {

        private final String playlist;
        private final String folder;
        private final String[] images;

        public CCSDefinePlaylistCommand() {
            this.playlist = "";
            this.folder = "";
            this.images = new String[0];
        }

        public CCSDefinePlaylistCommand(String playlist, String folder, String... images) {
            this.playlist = playlist;
            this.folder = folder;
            this.images = images;
        }

        public String getPlaylist() {
            return playlist;
        }

        public String getFolder() {
            return folder;
        }

        public String[] getImages() {
            return images;
        }

        @Override
        List<Object> getArguments() {
            return Arrays.asList(playlist, folder, images);
        }

        @Override
        String getCommand() {
            return "definePlaylist";
        }

    }

    public static class CCSCommandResponse {

        private final CCSExecutor executor;

        public CCSCommandResponse(CCSExecutor executor) {
            this.executor = executor;
        }

        public CCSAckOrNack waitForAckOrNack() {
            try {
                Duration duration = executor.testPreconditions();
                return new CCSAckOrNack(duration);
            } catch (CCSPreconditionsNotMet ex) {
                return new CCSAckOrNack(ex.getReason());
            }
        }

        public void waitForCompletion() throws Exception {
            executor.execute();
        }

    }

    public static class CCSAckOrNack {

        private final boolean ack;
        private final Duration duration;
        private final String reason;

        private CCSAckOrNack(Duration duration) {
            this.ack = true;
            this.reason = null;
            this.duration = duration;
        }

        private CCSAckOrNack(String reason) {
            this.ack = false;
            this.reason = reason;
            this.duration = null;
        }

        public boolean isNack() {
            return !ack;
        }

        public String getReason() {
            return reason;
        }

        public Duration getDuration() {
            return duration;
        }

    }

    public static class CCSPreconditionsNotMet extends Exception {

        private static final long serialVersionUID = 7463370916615596884L;

        private final String reason;

        public CCSPreconditionsNotMet(String reason) {
            super("Reason : " + reason);
            this.reason = reason;
        }

        private String getReason() {
            return reason;
        }
    }
}
