package org.lsst.ccs.subsystem.ocsbridge.util;

import java.lang.reflect.Constructor;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.subsystem.ocsbridge.sim.FilterChanger.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.utilities.structs.Pair;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.event.EndLoadFilterEvent;
import org.lsst.sal.camera.event.EndRotateCarouselEvent;
import org.lsst.sal.camera.event.EndShutterCloseEvent;
import org.lsst.sal.camera.event.EndShutterOpenEvent;
import org.lsst.sal.camera.event.EndUnloadFilterEvent;
import org.lsst.sal.camera.event.StartLoadFilterEvent;
import org.lsst.sal.camera.event.StartRotateCarouselEvent;
import org.lsst.sal.camera.event.StartShutterCloseEvent;
import org.lsst.sal.camera.event.StartShutterOpenEvent;
import org.lsst.sal.camera.event.StartUnloadFilterEvent;

/**
 * This class exists to generate specific OCS events on stat changes, as mandated by LSE-71.
 * This usefulness of these generated events in questionable, since they contain no additional information
 * that was not in the state change, and in fact discard the information about when the state change actually occurred.
 * @author Farrukh Azfar, June 27 2018.
 */
public class OCSStateChangeToEventConverter {

    private final Map< Pair<Class<? extends Enum>, Class<? extends Enum>>, Class<? extends CameraEvent>> oldStateNewStateMap;

    public OCSStateChangeToEventConverter() {
        this.oldStateNewStateMap = new HashMap<>();

        // Note: ATS shutter does not use shutter OPENING/CLOSING states, but transitions directly CLSOED->OPEN->CLOSED
        oldStateNewStateMap.put(new Pair(ShutterState.CLOSED, ShutterState.OPENING), StartShutterOpenEvent.class);
        oldStateNewStateMap.put(new Pair(ShutterState.OPENING, ShutterState.OPEN), EndShutterOpenEvent.class);
        oldStateNewStateMap.put(new Pair(ShutterState.CLOSED, ShutterState.OPEN), EndShutterOpenEvent.class);
        oldStateNewStateMap.put(new Pair(ShutterState.OPEN, ShutterState.CLOSING), StartShutterCloseEvent.class);
        oldStateNewStateMap.put(new Pair(ShutterState.OPEN, ShutterState.CLOSED), EndShutterCloseEvent.class);
        oldStateNewStateMap.put(new Pair(ShutterState.CLOSING, ShutterState.CLOSED), EndShutterCloseEvent.class);
        oldStateNewStateMap.put(new Pair(FilterState.UNLOADED, FilterState.ROTATING), StartRotateCarouselEvent.class);
        oldStateNewStateMap.put(new Pair(FilterState.ROTATING, FilterState.UNLOADED), EndRotateCarouselEvent.class);
        oldStateNewStateMap.put(new Pair(FilterState.UNLOADED, FilterState.LOADING), StartLoadFilterEvent.class);
        oldStateNewStateMap.put(new Pair(FilterState.LOADING, FilterState.LOADED), EndLoadFilterEvent.class);
        oldStateNewStateMap.put(new Pair(FilterState.LOADED, FilterState.UNLOADING), StartUnloadFilterEvent.class);
        oldStateNewStateMap.put(new Pair(FilterState.UNLOADING, FilterState.UNLOADED), EndUnloadFilterEvent.class);
        // Note: Comcam filter changer does not using ROTATING or UNLOADING states
        oldStateNewStateMap.put(new Pair(FilterState.LOADED, FilterState.LOADING), StartLoadFilterEvent.class);
    }

    /**
     * Convert a CCS state change (represented by Enums) into the equivalent SAL
     * event.
     *
     * @param oldState The original state
     * @param newState The new state
     * @param eventPriority The priority for the created state change event
     * @return The SAL state change event, or <code>null</code> if no equivalent
     * event exists
     */
    public CameraEvent convert(Enum oldState, Enum newState, int eventPriority) {

        if (oldState == null || newState == null) {
            return null;
        }
        Pair statePair = new Pair(oldState, newState);
        Class<? extends CameraEvent> aa = oldStateNewStateMap.get(statePair);
        if (aa == null) {
            return null;
        }
        try {
            Constructor<? extends CameraEvent> bbc = oldStateNewStateMap.get(statePair).getConstructor(int.class);
            return bbc.newInstance(eventPriority);
        } catch (ReflectiveOperationException x) {
            throw new RuntimeException("Error converting state change : " + statePair, x);
        }

    }
}
