package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.ocsbridge.sim.FocalPlane.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * An interface for talking to a real focal-plane subsystem.
 *
 * @author tonyj
 */
class FocalPlaneSubsystemLayer extends ControlledSubsystem implements FocalPlaneInterface {

    private static final Logger LOG = Logger.getLogger(FocalPlaneSubsystemLayer.class.getName());
    private final static Map<Enum<FocalPlaneState>, Enum<RaftsState>> FOCALPLANE_TO_RAFTS_STATE = new HashMap<>();

    static {
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.NEEDS_CLEAR, RaftsState.NEEDS_CLEAR);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.CLEARING, RaftsState.CLEARING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.INTEGRATING, RaftsState.INTEGRATING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.READING_OUT, RaftsState.READING_OUT);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.QUIESCENT, RaftsState.QUIESCENT);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.ROW_SHIFT, RaftsState.INTEGRATING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.IMAGE_WAIT, RaftsState.READING_OUT);
    }

    FocalPlaneSubsystemLayer(MCMSubsystem mcm, CCS ccs, MCMConfig config) {
        super(mcm, config.getFocalPlaneSubsystemName(), ccs, config);
    }

    @Override
    public void endIntegration(boolean readout, Duration exposure) throws ExecutionException {
        Map<String, Serializable> meta = new HashMap<>();
        meta.put("ExposureTime", exposure.toMillis() / 1000.);
        commandSender.sendCommand("setHeaderKeywords", meta);
        commandSender.sendCommand("endIntegration");
    }

    @Override
    public void startIntegration(ImageName imageName, Map<String, String> parsedKeyValueData, LocationSet locations, String annotation) throws ExecutionException {
        String imageType = parsedKeyValueData.get("imageType");
        String testType = parsedKeyValueData.get("testType");
        String groupId = parsedKeyValueData.get("groupId");
        Map<String, Serializable> meta = new HashMap<>();
        meta.put("GroupId", groupId);
        meta.put("ImageType", imageType);
        if (testType != null) {
            meta.put("TestType", testType);
        }

        commandSender.sendCommand("setHeaderKeywords", meta);
        commandSender.sendCommand("startNamedIntegration", imageName, annotation, locations);
    }

    @Override
    public void clear(int nClears) throws ExecutionException {
        commandSender.sendCommand("clear", nClears);
    }

    @Override
    protected void onStateChange(StatusStateChangeNotification statusChange) {
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateFocalPlaneStateToRaftsState(value);
        });
    }

    @Override
    protected void onConnect(AgentInfo agent, StateBundle initialState) {
        LOG.info("Focal plane connected");
        FocalPlaneState state = initialState.getState(FocalPlaneState.class);
        translateFocalPlaneStateToRaftsState(state);
    }
    
    @Override
    protected void onDisconnect(AgentInfo agent) {
        LOG.info("Focal plane disconnected");
        // TOOD: Deal with initial state                        
//        StateBundle result = commandSender.sendCommand("getState", StateBundle.class);
//        FocalPlaneState state = result.getState(FocalPlaneState.class);
//        translateFocalPlaneStateToRaftsState(state);
    }
    /**
     * Translate focal-plane states to the RaftsState messages the MCM is expecting
     * @param value 
     */
    private void translateFocalPlaneStateToRaftsState(Enum value) {
        LOG.log(Level.INFO, "Got focal-plane state {0} ", value);
        Enum<RaftsState> converted = FOCALPLANE_TO_RAFTS_STATE.get(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(new State(converted));
        }
    }
}
