package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;

/**
 *
 * @author tonyj
 */
public class FilterChangerSimulation implements FilterChangerInterface {

    private enum InstalledFilter {
        NONE(0), U(10), G(9), R(1), I(9), X(100);

        private final short serial;

        InstalledFilter(int serial) {
            this.serial = (short) serial;
        }

        public short getSerial() {
            return serial;
        }

        @Override
        public String toString() {
            if (this == NONE) {
                return super.toString();
            } else {
                return super.toString().toLowerCase() + "-" + serial;
            }
        }
    }

    static final Duration LOAD_TIME = Duration.ofMillis(15000);
    static final Duration ROTATION_TIME_PER_DEGREE = Duration.ofMillis(100);
    static final Duration UNLOAD_TIME = Duration.ofMillis(15000);

    private final Map<String, String> availableFilters = new LinkedHashMap<>();
    private InstalledFilter currentFilter = InstalledFilter.NONE;
    private int currentRotationPosition = 0;
    private final CCS ccs;
    private final State filterState;

    FilterChangerSimulation(CCS ccs) {
        this.ccs = ccs;
        for (InstalledFilter f : InstalledFilter.values()) {
            availableFilters.put(f.toString(), f.name().toLowerCase());
        }
        filterState = new State(FilterChanger.FilterState.UNLOADED);
        ccs.getAggregateStatus().add(filterState);
    }

    @Override
    public Map<String, String> getAvailableFilters() {
        return Collections.unmodifiableMap(availableFilters);
    }

    @Override
    public List<String> getInstalledFilters() {
        return new ArrayList(availableFilters.keySet());
    }

    @Override
    public void start(String configName) {
        /// No Op
    }

    @Override
    public String getCurrentFilter() {
        return currentFilter.toString();
    }

    @Override
    public void setFilter(String filterName) {
        try {
            int position = getInstalledFilters().indexOf(filterName);
            if (position < 0) {
                throw new IllegalArgumentException("Invalid filter: " + filterName);
            }
            InstalledFilter requestedFilter = InstalledFilter.values()[position];
            ccs.fireEvent(new MCM.CCSSetFilterEvent(filterName, requestedFilter.name().toLowerCase()));

            int targetRotation = -1;
            if (requestedFilter.equals(currentFilter)) {
                // No-op?
            } else if (requestedFilter == InstalledFilter.NONE) {
                filterState.setState(FilterChanger.FilterState.UNLOADING);
                Future<Void> waitForUnloaded = ccs.waitForStatus(FilterChanger.FilterState.UNLOADED);
                ccs.schedule(UNLOAD_TIME, () -> {
                    filterState.setState(FilterChanger.FilterState.UNLOADED);
                    currentFilter = InstalledFilter.NONE;
                });
                waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            } else {
                if (currentFilter != InstalledFilter.NONE) {
                    filterState.setState(FilterChanger.FilterState.UNLOADING);
                    Future<Void> waitForUnloaded = ccs.waitForStatus(FilterChanger.FilterState.UNLOADED);
                    ccs.schedule(UNLOAD_TIME, () -> {
                        filterState.setState(FilterChanger.FilterState.UNLOADED);
                        currentFilter = InstalledFilter.NONE;
                    });
                    waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
                }
                targetRotation = (position - 1) * 360 / 5;
                if (currentRotationPosition != targetRotation) {
                    // FIXME: Always rotates in same direction, probably not realistic
                    int degreesToRotate = Math.abs(currentRotationPosition - targetRotation) % 360;
                    filterState.setState(FilterChanger.FilterState.ROTATING);
                    Future<Void> waitForRotation = ccs.waitForStatus(FilterChanger.FilterState.UNLOADED);
                    Duration rotationTime = ROTATION_TIME_PER_DEGREE.multipliedBy(degreesToRotate);
                    ccs.schedule(rotationTime, () -> {
                        filterState.setState(FilterChanger.FilterState.UNLOADED);
                    });
                    waitForRotation.get(rotationTime.toMillis() * 2, TimeUnit.MILLISECONDS);
                    currentRotationPosition = targetRotation;
                }
                filterState.setState(FilterChanger.FilterState.LOADING);
                Future<Void> waitForLoaded = ccs.waitForStatus(FilterChanger.FilterState.LOADED);
                ccs.schedule(LOAD_TIME, () -> {
                    filterState.setState(FilterChanger.FilterState.LOADED);
                    currentFilter = requestedFilter;
                });
                waitForLoaded.get(LOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
            }
            ccs.fireEvent(new MCM.CCSSetFilterEvent(filterName, requestedFilter.name().toLowerCase(), position, targetRotation));
        } catch (InterruptedException | ExecutionException | TimeoutException ex) {
            throw new RuntimeException("Error while changing filter", ex);
        }
    }

    @Override
    public Duration getEstimatedDurationForFilterChange(String filterName) {
        return ROTATION_TIME_PER_DEGREE.multipliedBy(360).plus(LOAD_TIME).plus(UNLOAD_TIME);
    }
}
