package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.subsystem.ocsbridge.util.State.StateChangeListener;
import org.lsst.sal.camera.states.CCSCommandStateEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.SummaryStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent.SummaryState;

/**
 * Top level "module" for running the OCSBridge as a CCS Subsystem
 *
 * @author tonyj
 */
public class OCSSubsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsystem;
    private OCSBridge ocsBridge;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private OCSBridgeConfig config;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;

    OCSSubsystem() {
    }

    @Override
    public void init() {
        agentStateService.registerState(SummaryState.class, "SAL Summary State", subsystem);
        agentStateService.registerState(OfflineState.class, "SAL Offline State", subsystem);
        agentStateService.registerState(CCSCommandStateEvent.CCSCommandState.class, "SAL Command State", subsystem);
    }

    @Override
    public void postStart() {
        CCS ccs = new CCS();
        MCMLayer mcmLayer = new MCMCCSLayer(subsystem, ccs, config);
        ocsBridge = new OCSBridge(config, ccs, mcmLayer);
        // Make the LSE209 state available as a CCS state
        final State lse209State = ocsBridge.getLse209State();
        final State offlineState = ocsBridge.getOfflineState();
        final State commandState = ocsBridge.getCommandState();
        SoftwareVersionsGenerator softwareVersionsGenerator = new SoftwareVersionsGenerator(ocsBridge, subsystem);
        agentStateService.updateAgentState(lse209State.getState(), offlineState.getState(), commandState.getState());
        StateChangeListener l = (StateChangeListener) (Enum state, Enum oldState) -> {
            agentStateService.updateAgentState(state);
            if (state == SummaryStateEvent.SummaryState.STANDBY) {
                softwareVersionsGenerator.run();
            }
        };
        lse209State.addStateChangeListener(l);
        offlineState.addStateChangeListener(l);
        commandState.addStateChangeListener(l);
        ocsBridge.setCCSCommandExecutor(new CCSCommandExecutor() {
            @Override
            void sendAck(CCSCommand.CCSAckOrNack can) {
                // Note: We add 100mS to the estimate to keep ShellCommandConsole from timing
                // out, especially on zero length commands.
                subsystem.sendAck(can.getDuration().plus(Duration.ofMillis(100)));
            }

            @Override
            void sendNack(CCSCommand.CCSAckOrNack can) {
                subsystem.sendNack(can.getReason());
            }
        });

        OCSBridgeSALLayer ocsInterface = new OCSBridgeSALLayer(ocsBridge);
        ocsBridge.setOCSCommandExecutor(ocsInterface);
        ocsInterface.start();
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setAvailable() throws Exception {
        CCSCommand.CCSSetAvailableCommand setAvailable = new CCSCommand.CCSSetAvailableCommand();
        ocsBridge.execute(setAvailable);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void revokeAvailable() throws Exception {
        CCSCommand.CCSRevokeAvailableCommand revokeAvailable = new CCSCommand.CCSRevokeAvailableCommand();
        ocsBridge.execute(revokeAvailable);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void simulateFault() throws Exception {
        CCSCommand.CCSSimulateFaultCommand simulateFault = new CCSCommand.CCSSimulateFaultCommand();
        ocsBridge.execute(simulateFault);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clearFault() throws Exception {
        CCSCommand.CCSClearFaultCommand clearFault = new CCSCommand.CCSClearFaultCommand();
        ocsBridge.execute(clearFault);
    }
}
