package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ScheduledFuture;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.subsystem.ocsbridge.util.State.StateChangeListener;
import org.lsst.ccs.utilities.location.LocationSet;

/**

* <ol>
 * <li>If raftsSubsystem is null (the default) it simulates all raft operations within
 * this class</li>
 * <li>If raftsSubsystem is specified, it simulates the states within
 * this class, but delegates the operations to the specified subsystem</li> 
 * </ol>
 * This is temporary until the subsystem implements the states itself.
 *
 * @author tonyj
 */
public class Rafts implements ImageTaking {

    /**
     * Time to readout the science rafts
     */
    static final Duration READOUT_TIME = Duration.ofMillis(2000);
    /**
     * Time to clear the sensors
     */
    static final Duration CLEAR_TIME = Duration.ofMillis(70);
    /**
     * Idle time before a clear is required
     */
    static final Duration QUIESCENT_BEFORE_CLEAR = Duration.ofMillis(4000);

    public enum RaftsState {
        NEEDS_CLEAR, CLEARING, INTEGRATING, READING_OUT, QUIESCENT
    }

    private final State raftsState;
    private final CCS ccs;
    private RaftsSubsystemLayer raftsSubsystem;

    Rafts(CCS ccs) {
        this.ccs = ccs;
        raftsState = new State(RaftsState.NEEDS_CLEAR);
        ccs.getAggregateStatus().add(raftsState);
        // Whenever we enter quiescent state, we start a timer to indicate when a clear is needed
        // If we exit ready state we cancel the timer.
        raftsState.addStateChangeListener(new StateChangeListener<RaftsState>() {
            private ScheduledFuture<?> clearFuture;

            @Override
            public void stateChanged(RaftsState currentState, RaftsState oldState) {
                if (currentState == RaftsState.QUIESCENT) {
                    clearFuture = ccs.schedule(QUIESCENT_BEFORE_CLEAR, () -> {
                       raftsState.setState(RaftsState.NEEDS_CLEAR);
                    });
                } else {
                    if (clearFuture != null) {
                        clearFuture.cancel(false);
                        clearFuture = null;
                    }
                }
            }

        });
    }

    @Override
    public void clear(int nClears) {
        raftsState.checkState(RaftsState.QUIESCENT, RaftsState.NEEDS_CLEAR);
        raftsState.setState(RaftsState.CLEARING);
        if (raftsSubsystem == null) {
            ccs.schedule(CLEAR_TIME.multipliedBy(nClears), () -> {
                raftsState.setState(RaftsState.QUIESCENT);
            });
        } else {
            ccs.schedule(Duration.ZERO, () -> {
                raftsSubsystem.clear(nClears);
                raftsState.setState(RaftsState.QUIESCENT);
            });
        }
    }

    @Override
    public void startIntegration(ImageName imageName, Map<String, String> parsedKeyValueData, LocationSet locations, String annotation) {
        raftsState.checkState(RaftsState.QUIESCENT);
        if (raftsSubsystem != null) {
            raftsSubsystem.startIntegration(imageName, parsedKeyValueData.get("imageType"), parsedKeyValueData.get("groupId"));
        }
        raftsState.setState(RaftsState.INTEGRATING);    }

    @Override
    public void endIntegration(boolean readout, Duration exposure) {
        raftsState.checkState(RaftsState.INTEGRATING);
        if (readout) {
            raftsState.setState(RaftsState.READING_OUT);
            if (raftsSubsystem != null) {
                ccs.schedule(Duration.ZERO, () -> {
                    raftsSubsystem.acquireImage(exposure);
                    raftsState.setState(RaftsState.QUIESCENT);
                });
            } else {
                ccs.schedule(READOUT_TIME, () -> {
                    raftsState.setState(RaftsState.QUIESCENT);
                });
            }
        } else {
            raftsState.setState(RaftsState.NEEDS_CLEAR);
        }
    }

    /**
     * Called when the MCM receives a start command.
     * @param configName 
     */
    @Override
   public  void start(String configName) {        
        if (raftsSubsystem != null) {
            raftsSubsystem.start(configName);
        } else {
            // TODO: For the moment we always send the ATS configuration info
//            try (InputStream in = Rafts.class.getResourceAsStream("config-info.ser");
//                    ObjectInputStream ois = new ObjectInputStream(in)) {
//                StatusConfigurationInfo msg = (StatusConfigurationInfo) ois.readObject();
//                ConfigurationInfo info = msg.getConfigurationInfo();
//                StateBundle state = msg.getState();
//                ccs.fireEvent(new StatusConfigurationInfo(info,state));
//            } catch (IOException | ClassNotFoundException x) {
//                throw new RuntimeException("Unable to read simulation configuration event", x);
//            }
//            ccs.schduleAtFixedRate(10, TimeUnit.SECONDS, () -> sendTrending());
        }
    }

    private void sendTrending() {
        try (InputStream in = Rafts.class.getResourceAsStream("ats-wreb-trending.ser");
                ObjectInputStream ois = new ObjectInputStream(in)) {
            StatusSubsystemData msg = (StatusSubsystemData) ois.readObject();
            KeyValueData data = msg.getSubsystemData();
            ccs.fireEvent(new StatusSubsystemData(data));
        } catch (IOException | ClassNotFoundException x) {
            throw new RuntimeException("Unable to read simulation ats-wreb-trending event", x);
        } 
    }

    void setRaftsSubsystem(RaftsSubsystemLayer raftsSubsystem) {
        this.raftsSubsystem = raftsSubsystem;
    }
}
