package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.text.ParseException;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.AggregateStatus;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig.Camera;
import org.lsst.ccs.subsystem.ocsbridge.sim.Rafts.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.KeyValueParser;
import org.lsst.ccs.utilities.location.LocationSet;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A simple implementation  of the MCM. Designed to be used either directly within
 * the OCSSimulation, or as a standalone process.
 *
 * @author tonyj
 */
public class MCM {

    private final MCMConfig config;
    private final ImageNameService imageNameService;
    private static final Logger LOG = Logger.getLogger(MCM.class.getName());

    public enum TakeImageReadinessState {

        NOT_READY, GETTING_READY, READY
    };

    public enum CalibrationState {
        DISABLED, ENABLED, INTEGRATING
    }

    private final CCS ccs;
    private final State takeImageReadinessState;
    private final State calibrationState;
    private final Shutter shutter;
    private final ImageTaking imageTaking;
    private final Filter fcs;

    private Instant startIntegrationTime;
    private ScheduledFuture<?> startImageTimeout;
    private ScheduledFuture<?> shutterPrepSchedule;
    private ScheduledFuture<?> raftsClearSchedule;
    private long integrationStartTime;

    public MCM(CCS ccs, MCMConfig config, ImageNameService imageNameService) {
        this.ccs = ccs;
        this.config = config;
        this.imageNameService = imageNameService;
        takeImageReadinessState = new State(TakeImageReadinessState.NOT_READY);
        ccs.getAggregateStatus().add(takeImageReadinessState);
        calibrationState = new State(CalibrationState.DISABLED);
        ccs.getAggregateStatus().add(calibrationState);
        shutter = new Shutter(ccs);
        imageTaking = config.hasFocalPlaneSubsystem() ? new FocalPlane(ccs) : new Rafts(ccs);
        if (config.hasFilterChanger()) {
            fcs = new Filter(ccs);
        } else {
            fcs = null;
        }

        // We are ready to take an image only if the rafts have been cleared, and the shutter
        // has been prepared.
        ccs.addStateChangeListener((state, oldState) -> {
            AggregateStatus as = ccs.getAggregateStatus();
            if (as.hasState(Rafts.RaftsState.QUIESCENT, Shutter.ShutterReadinessState.READY)) {
                takeImageReadinessState.setState(TakeImageReadinessState.READY);
            } else if (!as.hasState(TakeImageReadinessState.GETTING_READY)) {
                takeImageReadinessState.setState(TakeImageReadinessState.NOT_READY);
            }
            if (oldState == RaftsState.CLEARING || oldState == RaftsState.READING_OUT) {
                //TODO: Should be TAI time.
                integrationStartTime = System.currentTimeMillis();
            }
        });
    }

    CCSCommandResponse execute(CCSCommand command) {
        if (command instanceof CCSInitImageCommand) {
            return execute((CCSInitImageCommand) command);
        } else if (command instanceof CCSTakeImagesCommand) {
            return execute((CCSTakeImagesCommand) command);
        } else if (command instanceof CCSSetFilterCommand) {
            return execute((CCSSetFilterCommand) command);
        } else if (command instanceof CCSInitGuidersCommand) {
            return execute((CCSInitGuidersCommand) command);
        } else if (command instanceof CCSStartImageCommand) {
            return execute((CCSStartImageCommand) command);
        } else if (command instanceof CCSEndImageCommand) {
            return execute((CCSEndImageCommand) command);
        } else if (command instanceof CCSClearCommand) {
            return execute((CCSClearCommand) command);
        } else if (command instanceof CCSDiscardRowsCommand) {
            return execute((CCSDiscardRowsCommand) command);
        } else if (command instanceof CCSStartCommand) {
            return execute((CCSStartCommand) command);
        } else if (command instanceof CCSEnableCalibrationCommand) {
            return execute((CCSEnableCalibrationCommand) command);
        } else if (command instanceof CCSDisableCalibrationCommand) {
            return execute((CCSDisableCalibrationCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    CCSCommandResponse execute(CCSInitImageCommand command) {
        InitImageExecutor executor = new InitImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSTakeImagesCommand command) {
        TakeImagesExecutor executor = new TakeImagesExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSSetFilterCommand command) {
        SetFilterExecutor executor = new SetFilterExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSInitGuidersCommand command) {
        InitGuidersExecutor executor = new InitGuidersExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSStartImageCommand command) {
        StartImageExecutor executor = new StartImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSEndImageCommand command) {
        EndImageExecutor executor = new EndImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSClearCommand command) {
        ClearExecutor executor = new ClearExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDiscardRowsCommand command) {
        DiscardRowsExecutor executor = new DiscardRowsExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSStartCommand command) {
        StartExecutor executor = new StartExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSEnableCalibrationCommand command) {
        EnableCalibrationExecutor executor = new EnableCalibrationExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDisableCalibrationCommand command) {
        DisableCalibrationExecutor executor = new DisableCalibrationExecutor(command);
        return new CCSCommandResponse(executor);
    }

    //TODO: Remove me
    CCS getCCS() {
        return ccs;
    }

    ImageTaking getImageTaking() {
        return imageTaking;
    }

    public Shutter getShutter() {
        return shutter;
    }

    MCMConfig getConfig() {
        return config;
    }

    class InitImageExecutor extends CCSExecutor {

        private final CCSInitImageCommand command;

        public InitImageExecutor(CCSInitImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (command.getDeltaT() <= 0 || command.getDeltaT() > 15) {
                throw new CCSPreconditionsNotMet("Invalid deltaT: " + command.getDeltaT());
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            Duration takeImagesExpected = Duration.ofMillis((long) (command.getDeltaT() * 1000));
            takeImageReadinessState.setState(TakeImageReadinessState.GETTING_READY);
            raftsClearSchedule = ccs.schedule(takeImagesExpected.minus(Rafts.CLEAR_TIME), () -> {
                imageTaking.clear(1);
            });
            shutterPrepSchedule = ccs.schedule(takeImagesExpected.minus(Shutter.PREP_TIME), () -> {
                shutter.prepare();
            });
        }

    }

    class TakeImagesExecutor extends CCSExecutor {

        private final CCSTakeImagesCommand command;
        private Map<String, String> parsedKeyValueData;
        private LocationSet locations;

        public TakeImagesExecutor(CCSTakeImagesCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {

            if (command.getNumImages() <= 0 || command.getNumImages() > config.getMaxImagesPerSequence()) {
                throw new CCSPreconditionsNotMet("Invalid number of images");
            }
            double minExposeTime = command.isShutter() ? config.getMinExposeTime() : 0;
            if (command.getExpTime() < minExposeTime|| command.getExpTime() > config.getMaxExposeTime()) {
                throw new CCSPreconditionsNotMet("Invalid exposure time");
            }    
            // Parse the keyValueData and check for legallity
            try {
                parsedKeyValueData = KeyValueParser.parse(command.getKeyValueMap());
                for (String key : config.getRequiredKeys()) {
                    if (!parsedKeyValueData.containsKey(key)) {
                       throw new CCSPreconditionsNotMet("Required key missing: " + key);
                    }
                }
                for (String key : parsedKeyValueData.keySet()) {
                    if (!config.getAllowedKeys().contains(key)) {
                       throw new CCSPreconditionsNotMet("Key not allowed: " + key);                        
                    }
                }
            } catch (ParseException x) {
                throw new CCSPreconditionsNotMet(x.getMessage());
            }

            if (!command.getSensors().trim().isEmpty()) {
                try {
                    locations = LocationSet.of(command.getSensors());
                } catch (RuntimeException x) {
                    throw new CCSPreconditionsNotMet("Invalid sensors argument" + command.getSensors());                        
                }
            } else {
                locations = new LocationSet();
            }
            
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            // Worse case estimate
            return Duration.ofMillis((long) (command.getExpTime() * 1000)).plus(Shutter.MOVE_TIME).plus(Rafts.READOUT_TIME).multipliedBy(command.getNumImages());
        }

        @Override
        @SuppressWarnings("SleepWhileInLoop")
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException, CCSPreconditionsNotMet {
            
            List<ImageName> imageNames = imageNameService.getImageNames(command.getNumImages());
            Duration exposeTime = Duration.ofMillis((long) (command.getExpTime() * 1000));
            
            if (command.getNumImages() > 1) {
                shutter.startImageSequence();
            }

            try {

                for (int i = 0; i < command.getNumImages(); i++) {
                    Future waitUntilReady = ccs.waitForStatus(TakeImageReadinessState.READY);
                    if (takeImageReadinessState.isInState(TakeImageReadinessState.NOT_READY)) {
                        imageTaking.clear(1);
                        shutter.prepare();
                    } else if (takeImageReadinessState.isInState(TakeImageReadinessState.GETTING_READY)) {
                        boolean raftClearEnd = raftsClearSchedule.cancel(false);
                        boolean shutterPrepEnd = shutterPrepSchedule.cancel(false);
                        if (raftClearEnd) {
                            imageTaking.clear(1);
                        }
                        if (shutterPrepEnd) {
                            shutter.prepare();
                        }
                    }
                    waitUntilReady.get(20, TimeUnit.SECONDS);

                    ImageName imageName = imageNames.get(i);
                    ccs.fireEvent(new CCSImageNameEvent(parsedKeyValueData, command.getNumImages(), imageName, i, 
                            integrationStartTime, command.getExpTime()));
                    LOG.log(Level.INFO, "Sent image name event for {0}", imageName);

                    if (command.isShutter()) {
                        imageTaking.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
                        shutter.expose(exposeTime);
                        //TODO: Check that shutter expose does not return until shutter has started opening
                        Future waitShutterClosed = ccs.waitForStatus(Shutter.ShutterState.CLOSED);
                        waitShutterClosed.get(exposeTime.plus(Shutter.MOVE_TIME).plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                        imageTaking.endIntegration(true, exposeTime);
                        // For the last exposure we only wait until the readout starts
                        // For other exposures we must wait until readout is complete
                        Future waitUntilDone = ccs.waitForStatus(i + 1 < command.getNumImages() ? Rafts.RaftsState.QUIESCENT : Rafts.RaftsState.READING_OUT);
                        waitUntilDone.get(Rafts.READOUT_TIME.plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                    } else {
                        imageTaking.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
                        Thread.sleep(exposeTime.toMillis());
                        // Is this correct, should we also set DARKTIME?
                        imageTaking.endIntegration(true, Duration.ZERO);
                        Future waitUntilDone = ccs.waitForStatus(i + 1 < command.getNumImages() ? Rafts.RaftsState.QUIESCENT : Rafts.RaftsState.READING_OUT);
                        waitUntilDone.get(Rafts.READOUT_TIME.plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                    }
                }

            } finally {
                shutter.endImageSequence();
            }
        }

    }

    public static class CCSEndOfImageTelemetryEvent extends CCSEvent {
        private final double darkTime;
        private final double exposureTime;
        private final long imageTag;
        private final CCSTimeStamp dateObs;
        private final CCSTimeStamp dateEnd;

        public CCSEndOfImageTelemetryEvent(double darkTime, double exposureTime, long imageTag, CCSTimeStamp dateObs, CCSTimeStamp dateEnd) {
            this.darkTime = darkTime;
            this.exposureTime = exposureTime;
            this.imageTag = imageTag;
            this.dateObs = dateObs;
            this.dateEnd = dateEnd;
        }

        public double getDarkTime() {
            return darkTime;
        }

        public double getExposureTime() {
            return exposureTime;
        }

        public long getImageTag() {
            return imageTag;
        }

        public CCSTimeStamp getDateObs() {
            return dateObs;
        }

        public CCSTimeStamp getDateEnd() {
            return dateEnd;
        }
        
    }
    
    public static class CCSImageNameEvent extends CCSEvent {

        private final int imagesInSequence;
        private final ImageName imageName;
        private final int sequenceNumber;
        private final long integrationStartTime;
        private final double exposureTime;
        private final Map<String, String> keyValueData;

        public CCSImageNameEvent(Map<String, String> keyValueData, int imagesInSequence, ImageName imageName, int sequenceNumber, long integrationStartTime, double exposureTime) {
            this.keyValueData = keyValueData;
            this.imagesInSequence = imagesInSequence;
            this.imageName = imageName;
            this.sequenceNumber = sequenceNumber;
            this.integrationStartTime = integrationStartTime;
            this.exposureTime = exposureTime;
        }

        public String getImageType() {
            return keyValueData.get("imageType");
        }

        public String getGroupId() {
            return keyValueData.get("groupId");
        }

        public int getImagesInSequence() {
            return imagesInSequence;
        }

        public ImageName getImageName() {
            return imageName;
        }

        public int getSequenceNumber() {
            return sequenceNumber;
        }

        public long getIntegrationStartTime() {
            return integrationStartTime;
        }

        public double getExposureTime() {
            return exposureTime;
        }

        public Map<String, String> getKeyValueData() {
            return keyValueData;
        }

        @Override
        public String toString() {
            return "CCSImageNameEvent{" + "imagesInSequence=" + imagesInSequence + ", imageName=" + imageName + ", sequenceNumber=" + sequenceNumber + ", integrationStartTime=" + integrationStartTime + ", exposureTime=" + exposureTime + ", keyValueData=" + keyValueData + '}';
        }

    }
    
    class SetFilterExecutor extends CCSExecutor {

        private final CCSSetFilterCommand command;

        public SetFilterExecutor(CCSSetFilterCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (fcs == null) {
                throw new CCSPreconditionsNotMet("setFilter command not supported");
            }

            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            if (!fcs.filterIsAvailable(command.getFilterName())) {
                throw new CCSPreconditionsNotMet("Invalid filter: " + command.getFilterName());
            }
            // Worse case
            return Filter.ROTATION_TIME_PER_DEGREE.multipliedBy(360).plus(Filter.LOAD_TIME).plus(Filter.UNLOAD_TIME);
        }

        @Override
        protected void execute() throws Exception {
            ccs.fireEvent(new CCSSetFilterEvent(command.getFilterName(), true));
            fcs.setFilter(command.getFilterName());
            ccs.fireEvent(new CCSSetFilterEvent(command.getFilterName(), false));
        }
    }

    public static class CCSSetFilterEvent extends CCSEvent {

        private final String filterName;
        private final boolean start;

        public CCSSetFilterEvent(String filterName, boolean start) {
            this.filterName = filterName;
            this.start = start;
        }

        public String getFilterName() {
            return filterName;
        }

        public boolean isStart() {
            return start;
        }

        @Override
        public String toString() {
            return "CCSSetFilterEvent{" + "filterName=" + filterName + ", start=" + start + '}';
        }
    }

    public static class CCSSettingsAppliedEvent extends CCSEvent {

        private final String settings;
        private final double timestamp;

        public CCSSettingsAppliedEvent(String settings, double timestamp) {

            this.settings = settings;
            this.timestamp = timestamp;
        }

        public String getSettings() {
            return settings;
        }

        public double getTimeStamp() {
            return timestamp;
        }

        @Override
        public String toString() {
            return "SettingsApplied{" + "settings=" + settings + ", timestamp=" + timestamp + '}';
        }

    }

    public static class CCSAtSettingsAppliedEvent extends CCSEvent {

        private final String settings;
        private final double timestamp;
        private final int version;
        private final String WREBSettingsVersion;
        private final String BonnShutterSettingsVersion;

        public CCSAtSettingsAppliedEvent(String settings, double timestamp, int version, String WREBSettingsVersion, String BonnShutterSettingsVersion) {

            this.settings = settings;
            this.timestamp = timestamp;
            this.version = version;
            this.WREBSettingsVersion = WREBSettingsVersion;
            this.BonnShutterSettingsVersion = BonnShutterSettingsVersion;
        }

        public String getSettings() {
            return settings;
        }

        public double getTimestamp() {
            return timestamp;
        }

        public int getVersion() {
            return version;
        }

        public String getWREBSettingsVersion() {
            return WREBSettingsVersion;
        }

        public String getBonnShutterSettingsVersion() {
            return BonnShutterSettingsVersion;
        }

        @Override
        public String toString() {
            return "CCSAtSettingsAppliedEvent{" + "settings=" + settings + ", timestamp=" + timestamp + ", version=" + version + ", WREBSettingsVersion=" + WREBSettingsVersion + ", BonnShutterSettingsVersion=" + BonnShutterSettingsVersion + '}';
        }

    }

    private class InitGuidersExecutor extends CCSExecutor {

        private final CCSInitGuidersCommand command;

        public InitGuidersExecutor(CCSInitGuidersCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            // FIXME: Currently this does not actually do anything
        }
    }

    private class EnableCalibrationExecutor extends CCSExecutor {

        private final CCSEnableCalibrationCommand command;

        public EnableCalibrationExecutor(CCSEnableCalibrationCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.DISABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            calibrationState.setState(CalibrationState.ENABLED);
        }
    }

    private class DisableCalibrationExecutor extends CCSExecutor {

        private final CCSDisableCalibrationCommand command;

        public DisableCalibrationExecutor(CCSDisableCalibrationCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            calibrationState.setState(CalibrationState.DISABLED);
        }
    }

    private class StartExecutor extends CCSExecutor {

        private final CCSStartCommand command;

        public StartExecutor(CCSStartCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            if (fcs != null) {
                ccs.fireEvent(new CCSAvailableFiltersEvent(fcs.getAvailableFilters()));
            }
            if (imageTaking != null) {
                imageTaking.start("default");
            }
            if (shutter != null) {
                shutter.start("default");
            }
            CCSTimeStamp now = CCSTimeStamp.currentTime();
            if (config.getCameraType() == Camera.AUXTEL) {
                ccs.fireEvent(new CCSAtSettingsAppliedEvent(command.getConfiguration(), now.getTAIDouble(), 1, "default(1)", "default(1)"));
            } else {
                ccs.fireEvent(new CCSSettingsAppliedEvent(command.getConfiguration(), now.getTAIDouble()));
            }  
        }
    }

    private class ClearExecutor extends CCSExecutor {

        private final CCSClearCommand command;

        public ClearExecutor(CCSClearCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (command.getNClears() <= 0 || command.getNClears() > 15) {
                throw new CCSPreconditionsNotMet("Invalid nClears: " + command.getNClears());
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Rafts.CLEAR_TIME.multipliedBy(command.getNClears());
        }

        @Override
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException {
            imageTaking.clear(command.getNClears());
            // TODO: Note, unlike initImages, the clear command remains active until the clears are complete (Correct?)
            Future waitUntilClear = ccs.waitForStatus(Rafts.RaftsState.QUIESCENT);
            waitUntilClear.get(Rafts.CLEAR_TIME.multipliedBy(command.getNClears()).plus(Duration.ofSeconds(1)).toMillis(), TimeUnit.MILLISECONDS);
        }

    }

    private class StartImageExecutor extends CCSExecutor {

        private final CCSStartImageCommand command;
        private Map<String,String> parsedKeyValueData;
        private LocationSet locations;

        public StartImageExecutor(CCSStartImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {

            if (command.getTimeout() < 1 | command.getTimeout() > 120) {
                throw new CCSPreconditionsNotMet("Invalid argument");
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            
            // Parse the keyValueData and check for legallity
            try {
                parsedKeyValueData = KeyValueParser.parse(command.getKeyValueMap());
                for (String key : config.getRequiredKeys()) {
                    if (!parsedKeyValueData.containsKey(key)) {
                       throw new CCSPreconditionsNotMet("Required key missing: " + key);
                    }
                }
                for (String key : parsedKeyValueData.keySet()) {
                    if (!config.getAllowedKeys().contains(key)) {
                       throw new CCSPreconditionsNotMet("Key not allowed: " + key);                        
                    }
                }
            } catch (ParseException x) {
                throw new CCSPreconditionsNotMet(x.getMessage());
            }

            if (!command.getSensors().trim().isEmpty()) {
                try {
                    locations = LocationSet.of(command.getSensors());
                } catch (RuntimeException x) {
                    throw new CCSPreconditionsNotMet("Invalid sensors argument" + command.getSensors());                        
                }
            } else {
                locations = new LocationSet();
            }


            return Duration.ofSeconds(1);
        }

        @Override
        protected void execute() throws Exception {
            Future waitUntilReady = ccs.waitForStatus(TakeImageReadinessState.READY);
            // FIXME: It is not necessary to always clear and prepare the shutter, especially
            // if we are not actually going to open the shutter.
            if (takeImageReadinessState.isInState(TakeImageReadinessState.NOT_READY)) {
                imageTaking.clear(1);
                shutter.prepare();
            }

            waitUntilReady.get(1, TimeUnit.SECONDS);

            ImageName imageName = imageNameService.getImageName();

            ccs.fireEvent(new CCSImageNameEvent(parsedKeyValueData, 1, imageName, 1, integrationStartTime, 0));
            if (command.isShutterOpen()) {
                imageTaking.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
                shutter.open();
                // FIXME: Wait for shutter to open? right now we return immediately
            } else {
                imageTaking.startIntegration(imageName, parsedKeyValueData, locations, command.getObsNote());
            }
            startIntegrationTime = Instant.now();
            calibrationState.setState(CalibrationState.INTEGRATING);
            startImageTimeout = ccs.schedule(Duration.ofMillis((long) (command.getTimeout() * 1000)), () -> {
                imageTimeout();
            });
        }
    }

    /**
     * Called if the timeout for a takeImages occurs
     */
    private void imageTimeout() {
        // FIXME: Is this a NOOP if the shutter is already closed?
        shutter.close();
        imageTaking.endIntegration(false, Duration.ZERO);
        calibrationState.setState(CalibrationState.ENABLED);
    }

    private class EndImageExecutor extends CCSExecutor {

        private final CCSEndImageCommand command;

        public EndImageExecutor(CCSEndImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.INTEGRATING)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            if (startImageTimeout == null || startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("No exposure in progress");
            }
            return Shutter.MOVE_TIME;
        }

        @Override
        protected void execute() throws Exception {
            if (!startImageTimeout.cancel(false)) {
                throw new RuntimeException("Image exposure already timed out");
            }
            Future waitUntilClosed = ccs.waitForStatus(ShutterState.CLOSED);
            shutter.close();
            waitUntilClosed.get(Shutter.MOVE_TIME.plus(Duration.ofSeconds(1)).toMillis(), TimeUnit.MILLISECONDS);
            // TODO: What if shutter was never opened?
            Instant endIntegationTime = Instant.now();
            imageTaking.endIntegration(true, Duration.between(startIntegrationTime, endIntegationTime));
            calibrationState.setState(CalibrationState.ENABLED);
        }
    }

    private class DiscardRowsExecutor extends CCSExecutor {

        private final CCSDiscardRowsCommand command;

        public DiscardRowsExecutor(CCSDiscardRowsCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.INTEGRATING)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            if (startImageTimeout == null || startImageTimeout.isDone()) {
                throw new CCSCommand.CCSPreconditionsNotMet("No exposure in progress");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            // FIXME: Nothing actually happens, should at least generate some events.
        }
    }
}
