package org.lsst.ccs.subsystem.ocsbridge.util;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.subsystem.focalplane.data.SequencerType;
import org.lsst.sal.cccamera.event.FocalPlaneSummaryInfoEvent;
import org.lsst.sal.cccamera.event.FocalPlaneSummaryInfoEvent.CcdType;

/**
 * Generates a special event built for the header service from a focal-plane
 * configuration info object.
 *
 * @author tonyj
 */
public class SummaryInfoConverter {

    Pattern sensorPattern = Pattern.compile("(R..)/(Reb.)/(S..)/(\\w+)");
    Pattern rebPattern = Pattern.compile("(R..)/(Reb.)(?:_hardware)?/(\\w+)");
    Pattern raftPattern = Pattern.compile("(R..)/(.+)");

    public FocalPlaneSummaryInfoEvent convert(ConfigurationInfo data) {
        Set<String> ccdLocations = new TreeSet<>();
        Map<String, String> ccdNames = new HashMap<>();
        Map<String, String> manSerNums = new HashMap<>();
        Map<String, String> rebNames = new HashMap<>();
        Map<String, String> rebSerialNums = new HashMap<>();
        Map<String, String> raftNames = new HashMap<>();
        Map<String, Double> setTemp = new HashMap<>();
        Map<SequencerType, String> seqMap = new HashMap<>();
        // FIXME: Not currently available in config
        Map<SequencerType, String> seqCheckSum = Collections.EMPTY_MAP; 

        // First loop over hardware Ids
        Map<String, String> hardwareMap = data.getCurrentValuesForCategory("HardwareId");
        for (Map.Entry<String, String> entry : hardwareMap.entrySet()) {
            //System.out.println(entry.getKey() + "=" + entry.getValue());
            Matcher sensorMatcher = sensorPattern.matcher(entry.getKey());
            Matcher rebMatcher = rebPattern.matcher(entry.getKey());
            Matcher raftMatcher = raftPattern.matcher(entry.getKey());

            if (sensorMatcher.matches()) {
                final String ccdName = sensorMatcher.group(1) + sensorMatcher.group(3);
                ccdLocations.add(ccdName);
                String type = sensorMatcher.group(4);
                if ("name".equals(type)) {
                    ccdNames.put(ccdName, entry.getValue());
                } else if ("manSerNum".equals(type)) {
                    manSerNums.put(ccdName, entry.getValue());
                }
            } else if (rebMatcher.matches()) {
                final String rebName = rebMatcher.group(1) + rebMatcher.group(2);
                String type = rebMatcher.group(3);
                if ("name".equals(type)) {
                    rebNames.put(rebName, entry.getValue());
                }
            } else if (raftMatcher.matches()) {
                final String raftName = raftMatcher.group(1);
                String type = raftMatcher.group(2);
                if ("name".equals(type)) {
                    raftNames.put(raftName, entry.getValue());
                }
            }
        }
        Map<String, String> raftsMap = data.getCurrentValuesForCategory("Rafts");
        for (Map.Entry<String, String> entry : raftsMap.entrySet()) {
            //System.out.println(entry.getKey() + "=" + entry.getValue());
            Matcher rebMatcher = rebPattern.matcher(entry.getKey());
            if (rebMatcher.matches()) {
                final String rebName = rebMatcher.group(1) + rebMatcher.group(2);
                String type = rebMatcher.group(3);
                if ("serialNum".equals(type)) {
                    rebSerialNums.put(rebName, entry.getValue());
                }
            }
        }

        Map<String, String> tempControlMap = data.getCurrentValuesForCategory("RaftTempControl");
        for (Map.Entry<String, String> entry : tempControlMap.entrySet()) {
            Matcher raftMatcher = raftPattern.matcher(entry.getKey());
            if (raftMatcher.matches()) {
                final String raftName = raftMatcher.group(1);
                String type = raftMatcher.group(2);
                if ("TempControl/setTemp".equals(type)) {
                    setTemp.put(raftName, Double.parseDouble(entry.getValue()));
                }
            }
        }

        Map<String, String> sequencerMap = data.getCurrentValuesForCategory("Sequencer");
        List<ConfigurationParameterInfo> allParameterInfo = data.getAllParameterInfo();
        for (ConfigurationParameterInfo pi : allParameterInfo) {
            if ("sequencerConfig/sequencer".equals(pi.getPathName())) {
                seqMap.putAll((Map<SequencerType, String>) pi.getConfiguredValueObject());
            }
        }

        return FocalPlaneSummaryInfoEvent.builder()
                .ccdLocation(ccdLocations.stream().collect(join()))
                .raftBay(ccdLocations.stream().map(SummaryInfoConverter::raftBayForCCDLocation).collect(join()))
                .ccdSlot(ccdLocations.stream().map(SummaryInfoConverter::ccdSlotForCCDLocation).collect(join()))
                .raftLSSTName(ccdLocations.stream().map(SummaryInfoConverter::raftBayForCCDLocation).map(name -> raftNames.get(name)).collect(join()))
                .ccdLSSTName(ccdLocations.stream().map(k -> ccdNames.get(k)).collect(join()))
                .ccdManufacturer(ccdLocations.stream().map(k -> ccdNames.get(k).substring(0, 3)).collect(join()))
                .ccdType(ccdLocations.stream().map(k -> ccdNames.get(k).substring(0, 3)).map(k -> CcdType.valueOf(k)).toArray(CcdType[]::new))
                .ccdManSerNum(ccdLocations.stream().map(k -> get(manSerNums, k)).collect(join()))
                .rebSerialNumber(ccdLocations.stream().map(SummaryInfoConverter::rebSlotForCCDLocation).map(name -> rebSerialNums.get(name)).collect(join()))
                .rebLSSTName(ccdLocations.stream().map(SummaryInfoConverter::rebSlotForCCDLocation).map(name -> get(rebNames, name)).collect(join()))
                .ccdTempSetPoint(ccdLocations.stream().map(SummaryInfoConverter::raftBayForCCDLocation).map(name -> get(setTemp, name, -999.0)).mapToDouble(d -> d).toArray())
                .sequencerKey(ccdLocations.stream().map(k -> ccdNames.get(k).substring(0, 3)).map(k -> SequencerType.valueOf(k)).map(k -> get(seqMap, k)).collect(join()))
                .sequencerChecksum(ccdLocations.stream().map(k -> ccdNames.get(k).substring(0, 3)).map(k -> SequencerType.valueOf(k)).map(k -> get(seqCheckSum, k)).collect(join()))
                .build();
    }
    private static String raftBayForCCDLocation(String ccdLocation) {
        return ccdLocation.substring(0, 3);
    }
    
    private static String ccdSlotForCCDLocation(String ccdLocation) {
        return ccdLocation.substring(3, 6);
    }
    
    private static String rebSlotForCCDLocation(String ccdLocation) {
        return ccdLocation.substring(0, 3) + "Reb" + ccdLocation.substring(5, 6);
    }
    
    private static <K> String get(Map<K, String> map, K name) {
        return get(map, name, "Unknown");
    }
    
    private static <K, V> V get(Map<K, V> map, K key, V defaultValue) {
        return map.getOrDefault(key, defaultValue);
    }
    
    private static Collector<CharSequence, ?, String> join() {
        return Collectors.joining(":");
    }
}
