package org.lsst.ccs.subsystem.ocsbridge.sim;

import java.io.IOException;
import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.services.AgentLockService;
import org.lsst.ccs.services.UnauthorizedLevelException;
import org.lsst.ccs.services.UnauthorizedLockException;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Rafts.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * An interface for talking to a real focal-plane subsystem.
 *
 * @author tonyj
 */
class FocalPlaneSubsystemLayer {

    private static final Logger LOG = Logger.getLogger(FocalPlaneSubsystemLayer.class.getName());
    private final Subsystem mcm;
    private final CCS ccs;
    private final MCMConfig config;
    private ImageName imageName;
    private String imageType;
    private String groupId;
    private final static Map<Enum<FocalPlaneState>, Enum<RaftsState>> FOCALPLANE_TO_RAFTS_STATE = new HashMap<>();

    static {
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.NEEDS_CLEAR, RaftsState.NEEDS_CLEAR);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.CLEARING, RaftsState.CLEARING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.INTEGRATING, RaftsState.INTEGRATING);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.READING_OUT, RaftsState.READING_OUT);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.QUIESCENT, RaftsState.QUIESCENT);
        FOCALPLANE_TO_RAFTS_STATE.put(FocalPlaneState.ROW_SHIFT, RaftsState.INTEGRATING);
    }

    FocalPlaneSubsystemLayer(Subsystem mcm, CCS ccs, MCMConfig config) {
        this.mcm = mcm;
        this.ccs = ccs;
        this.config = config;

        Predicate<BusMessage<? extends Serializable, ?>> stateChangeFilter
                = BusMessageFilterFactory.messageOrigin(config.getFocalPlaneSubsystemName()).and(BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class));
        mcm.getMessagingAccess().addStatusMessageListener((msg) -> handleStatusChangeMessage(msg), stateChangeFilter);
        
        mcm.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connected(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (agent.getName().equals(config.getFocalPlaneSubsystemName())) {
                        StateBundle result = (StateBundle) sendCommand("getState");
                        FocalPlaneState state = result.getState(FocalPlaneState.class);
                        translateFocalPlaneStateToRaftsState(state);
                    }
                }
            }
        });


    }

    void endIntegration(boolean readout, Duration exposure) {
        sendCommand("endIntegration");
        // TODO: This should not be needed, but lets try it
        sendCommand("waitForImages", Duration.ofSeconds(60));
    }
    
    void startIntegration(ImageName imageName, Map<String, String> parsedKeyValueData, LocationSet locations, String annotation) {
        // TODO: Do something with the parsedKeyValueData
        sendCommand("startNamedIntegration", imageName, annotation, locations);
    }

    void start(String configName) {
        try {
            AgentLockService agentService = mcm.getAgentService(AgentLockService.class);
            agentService.setLevelForAgent(config.getFocalPlaneSubsystemName(), 10);
            sendCommand("publishConfigurationInfo");
        } catch (IOException | UnauthorizedLevelException | UnauthorizedLockException x) {
            throw new RuntimeException("Failed to lock subsystem " + config.getFocalPlaneSubsystemName(), x);
        }
    }

    void clear(int nClears) {
        sendCommand("clear", nClears);
    }

    private Object sendCommand(String command, Object... args) {
        CommandRequest request = new CommandRequest(config.getFocalPlaneSubsystemName(), command, args);
        ConcurrentMessagingUtils cmu = new ConcurrentMessagingUtils(mcm.getMessagingAccess());
        try {
            return cmu.sendSynchronousCommand(request);
        } catch (Exception ex) {
            // TODO: Fix me, we need to do something better here
            LOG.log(Level.SEVERE, "Error executing command", ex);
            return null;
        }
    }

    private void handleStatusChangeMessage(StatusMessage msg) {
        // called when the focal-plane status changes.
        // We need to translate them into the RaftsState messages the MCM is expecting
        StatusStateChangeNotification statusChange = (StatusStateChangeNotification) msg;
        StateBundle newStates = statusChange.getNewState();
        StateBundle oldStates = statusChange.getOldState();
        StateBundle changedStates = newStates.diffState(oldStates);
        changedStates.getDecodedStates().entrySet().stream().map((changedState) -> changedState.getValue()).forEachOrdered((value) -> {
            translateFocalPlaneStateToRaftsState(value);
        });
    }

    private void translateFocalPlaneStateToRaftsState(Enum value) {
        LOG.log(Level.INFO, "Got focal-plane state {0} ", value);
        Enum<RaftsState> converted = FOCALPLANE_TO_RAFTS_STATE.get(value);
        if (converted != null) {
            ccs.getAggregateStatus().add(new State(converted));
        }
    }
}
