package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.concurrent.ScheduledFuture;

/**
 * Very simple shutter simulation
 *
 * @author tonyj
 */
public class Shutter {

    /**
     * Time needed to get the shutter ready
     */
    static final Duration PREP_TIME = Duration.ofMillis(150);
    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME = Duration.ofMillis(4000);

    /**
     * Time shutter can remain ready after prepare or move
     */
    static final Duration READY_TIME_FOR_SEVERAL_IMAGES = Duration.ofMillis(10000);

    /**
     * Time needed to move a shutter blade
     */
    static final Duration MOVE_TIME = Duration.ofMillis(980);

    public boolean imageSequence = false;

    public enum ShutterReadinessState {

        NOT_READY, READY, GETTING_READY
    };

    public enum ShutterState {

        CLOSED, OPENING, OPEN, CLOSING
    };

    private final State shutterReadinessState;
    private final State shutterState;
    private ScheduledFuture<?> notReadyFuture;
    private ShutterSubsystemLayer shutterSubsystem;

    private final CCS ccs;

    Shutter(CCS ccs) {
        this.ccs = ccs;
        shutterReadinessState = new State(ShutterReadinessState.NOT_READY);
        ccs.getAggregateStatus().add(shutterReadinessState);
        shutterState = new State(ShutterState.CLOSED);
        ccs.getAggregateStatus().add(shutterState);
        // When the shutter is closed, we only keep the motors powered up for a limited time
        shutterState.addStateChangeListener((state, oldState) -> {
            if (state == ShutterState.CLOSED) {
                scheduleNotReady();
            } else {
                cancelNotReady();
            }
        });
    }

    public void startImageSequence() {

        imageSequence = true;
    }

    public void endImageSequence() {

        imageSequence = false;
    }

    private void scheduleNotReady() {

        Duration ready_time;
        if (imageSequence == true) {
            ready_time = READY_TIME_FOR_SEVERAL_IMAGES;
        } else {
            ready_time = READY_TIME;

        }

        notReadyFuture = ccs.schedule(ready_time, () -> {
            shutterReadinessState.setState(Shutter.ShutterReadinessState.NOT_READY);
        });
    }

    private void cancelNotReady() {
        if (notReadyFuture != null) {
            notReadyFuture.cancel(false);
        }
    }

    void prepare() {
        cancelNotReady();
        shutterReadinessState.setState(ShutterReadinessState.GETTING_READY);
        ccs.schedule(PREP_TIME, () -> {
            shutterReadinessState.setState(ShutterReadinessState.READY);
            scheduleNotReady();
        });
    }

    void expose(Duration exposureTime) {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutterState.checkState(ShutterState.CLOSED);
        if (shutterSubsystem == null) {
            shutterState.setState(ShutterState.OPENING);
            Duration time = MOVE_TIME;
            // TODO: This does not correctly handle the case when both blades move at once
            ccs.schedule(time, () -> {
                shutterState.setState(ShutterState.OPEN);
            });
            time = time.plus(exposureTime);
            ccs.schedule(time, () -> {
                shutterState.setState(ShutterState.CLOSING);
            });
            time = time.plus(MOVE_TIME);
            ccs.schedule(time, () -> {
                shutterState.setState(ShutterState.CLOSED);
            });
        } else {
            // TODO: Better event handling
            shutterState.setState(ShutterState.OPEN);
            shutterSubsystem.expose(exposureTime);
            shutterState.setState(ShutterState.CLOSED);            
        }
    }

    void start(String configName) {
        if (shutterSubsystem != null) {
            shutterSubsystem.start(configName);
        }
    }

    void open() {
        shutterReadinessState.checkState(ShutterReadinessState.READY);
        shutterState.checkState(ShutterState.CLOSED);
        shutterState.setState(ShutterState.OPENING);
        ccs.schedule(MOVE_TIME, () -> {
            shutterState.setState(ShutterState.OPEN);
        });
    }

    void close() {
        if (!shutterState.isInState(ShutterState.CLOSED)) {
            shutterState.checkState(ShutterState.OPEN);
            shutterState.setState(ShutterState.CLOSING);
            ccs.schedule(MOVE_TIME, () -> {
                shutterState.setState(ShutterState.CLOSED);
            });
        }
    }

    public void setShutterSubsystem(ShutterSubsystemLayer shutterSubsystem) {
        this.shutterSubsystem = shutterSubsystem;
    }

}
