package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;

/**
 * Run the simulated MCM as a CCS subsystem, able to send/receive commands using
 * CCS buses.
 *
 * @author tonyj
 */
public class MCMSubsystem implements HasLifecycle {

    private final CCS ccs = new CCS();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private MCMConfig config;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;
    private MCM mcm;

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsystem;

    @Override
    public void postStart() {

        mcm = new MCM(ccs, config, imageNameService);

        if (config.getRaftsSubsystemName() != null && !config.getRaftsSubsystemName().trim().isEmpty()) {
            mcm.getRafts().setRaftsSubsystem(new RaftsSubsystemLayer(subsystem, ccs, config));
        }
        if (config.getShutterSubsystemName() != null && !config.getShutterSubsystemName().trim().isEmpty()) {
            mcm.getShutter().setShutterSubsystem(new ShutterSubsystemLayer(subsystem , ccs, config));
        }
        ccs.addStateChangeListener((state, oldState) -> {
            // publish whenever state change occurs
            subsystem.updateAgentState(state);
        });

        // publish initial states
        List<Enum> initialStates = new ArrayList<>();
        ccs.getAggregateStatus().getStates().forEach((state) -> {
            initialStates.add(state.getState());
        });
        subsystem.updateAgentState(initialStates.toArray(new Enum[initialStates.size()]));

        ccs.addStatusMessageListener((StatusMessage msg) -> {
            // This should be called when a simulated subsystem generates trending/config
            // data.
            subsystem.getMessagingAccess().sendStatusMessage(msg);
        });
        
        ccs.addEventListener((CCSEvent event) -> {
            subsystem.publishSubsystemDataOnStatusBus(new KeyValueData("CCSEvent",event));
        });
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void initImage(@Argument(description="Time in seconds") double delta) throws Exception {
        CCSInitImageCommand initImage = new CCSInitImageCommand(delta);
        executeAndHandleResponse(initImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void startImage(boolean shutter, boolean wfs, boolean science, boolean guide, double timeout, String imageType, String groupId) throws Exception {
        CCSStartImageCommand startImage = new CCSStartImageCommand(shutter, wfs, guide, science, timeout, imageType, groupId);
        executeAndHandleResponse(startImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void takeImages(double expTime, int numImages, boolean shutter, boolean science, boolean wfs, boolean guide, String imageType, String groupId) throws Exception {
        CCSTakeImagesCommand takeImages = new CCSTakeImagesCommand(expTime, numImages, shutter, science, wfs, guide, imageType, groupId);
        executeAndHandleResponse(takeImages);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setFilter(String name) throws Exception {
        CCSSetFilterCommand setFilter = new CCSSetFilterCommand(name);
        executeAndHandleResponse(setFilter);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void initGuiders(String roiSpec) throws Exception {
        CCSInitGuidersCommand initGuiders = new CCSInitGuidersCommand(roiSpec);
        executeAndHandleResponse(initGuiders);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void endImage() throws Exception {
        CCSEndImageCommand endImage = new CCSEndImageCommand();
        executeAndHandleResponse(endImage);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clear(int nClears) throws Exception {
        CCSClearCommand clear = new CCSClearCommand(nClears);
        executeAndHandleResponse(clear);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void discardRows(int nRows) throws Exception {
        CCSDiscardRowsCommand discardRows = new CCSDiscardRowsCommand(nRows);
        executeAndHandleResponse(discardRows);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void start(String configuration) throws Exception {
        CCSStartCommand start = new CCSStartCommand(configuration);
        executeAndHandleResponse(start);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void enableCalibration() throws Exception {
        CCSEnableCalibrationCommand enableCalibration = new CCSEnableCalibrationCommand();
        executeAndHandleResponse(enableCalibration);
    }

    @Command(type = CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void disableCalibration() throws Exception {
        CCSDisableCalibrationCommand disableCalibration = new CCSDisableCalibrationCommand();
        executeAndHandleResponse(disableCalibration);
    }

    private void executeAndHandleResponse(CCSCommand command) throws Exception {
        CCSCommandResponse response = mcm.execute(command);
        CCSAckOrNack can = response.waitForAckOrNack();
        if (can.isNack()) {
            subsystem.sendNack(can.getReason());
        } else {
            // Note: We add 100mS to the estimate to keep ShellCommandConsole from timing
            // out, especially on zero length commands.
            subsystem.sendAck(can.getDuration().plus(Duration.ofMillis(1000)));
            response.waitForCompletion();
        }
    }
}
