package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.sim.MCMDirectLayer;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSCommandConverter;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSRevokeAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSimulateFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.OCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.PreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSAtSettingsAppliedEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSImageNameEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSettingsAppliedEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig;
import org.lsst.ccs.subsystem.ocsbridge.sim.Rafts.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSStateEventConverter;
import org.lsst.ccs.subsystem.ocsbridge.util.OCSStateChangeToEventConverter;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.command.DisableCommand;
import org.lsst.sal.camera.command.EnableCommand;
import org.lsst.sal.camera.command.EnterControlCommand;
import org.lsst.sal.camera.command.ExitControlCommand;
import org.lsst.sal.camera.command.StandbyCommand;
import org.lsst.sal.camera.command.StartCommand;
import org.lsst.sal.camera.command.StartImageCommand;
import org.lsst.sal.camera.command.TakeImagesCommand;
import org.lsst.sal.camera.event.AppliedSettingsMatchStartEvent;
import org.lsst.sal.camera.event.AvailableFiltersEvent;
import org.lsst.sal.camera.event.EndOfImageTelemetryEvent;
import org.lsst.sal.camera.event.EndReadoutEvent;
import org.lsst.sal.camera.event.EndSetFilterEvent;
import org.lsst.sal.camera.event.SettingVersionsEvent;
import org.lsst.sal.camera.event.SettingsAppliedEvent;
import org.lsst.sal.camera.event.StartIntegrationEvent;
import org.lsst.sal.camera.event.StartReadoutEvent;
import org.lsst.sal.camera.event.StartSetFilterEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.CCSCommandStateEvent.CCSCommandState;
import org.lsst.sal.camera.states.SummaryStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent.SummaryState;
import org.lsst.sal.atcamera.event.ShutterMotionProfileEvent;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraStateChangeEvent;
import org.lsst.sal.camera.event.ImageReadoutParametersEvent;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.subsystem.ocsbridge.util.BigUglyConverter;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.Source;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig.Camera;
import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.sal.atcamera.event.WrebSettingsAppliedEvent;
import org.lsst.sal.atcamera.telemetry.VacuumTelemetry;
import org.lsst.sal.atcamera.telemetry.WrebPowerTelemetry;
import org.lsst.sal.atcamera.telemetry.WrebTelemetry;

/**
 *
 * The Command Layer of the OCS Bridge. Contains executors for all commands
 * supported by the OCSBridge.
 *
 * @author The LSST CCS Team
 */
public class OCSBridge {

    private static final Logger LOG = Logger.getLogger(OCSBridge.class.getName());
    private final CCS ccs;
    private final State lse209State;
    private final State offlineState;
    private OCSCommandExecutor ocsCommandExecutor;
    private CCSCommandExecutor ccsCommandExecutor;
    private final MCMLayer mcm;
    private static final int DEFAULT_EVENT_PRIORITY = 1;
    private BlockingQueue<CCSImageNameEvent> imageNameTransfer = new ArrayBlockingQueue<>(1);
    private CCSImageNameEvent imageNameEvent;
    private final OCSBridgeConfig config;

    private final State commandState;
    private final BigUglyConverter buConverter = new BigUglyConverter();

    private final OCSCommandConverter ocsCcsCommandConverter = new OCSCommandConverter();
    private final OCSStateEventConverter ocsCcsStateEventConverter = new OCSStateEventConverter();
    private final OCSStateChangeToEventConverter ocsStateChangeToEventConverter = new OCSStateChangeToEventConverter();
    private long openTime;

    OCSBridge(OCSBridgeConfig config, CCS ccs, MCMLayer mcm) {

        this.config = config;
        this.ccs = ccs;
        this.mcm = mcm;
        lse209State = new State(SummaryState.OFFLINE);
        ccs.getAggregateStatus().add(lse209State);
        offlineState = new State(OfflineState.OFFLINE_PUBLISH_ONLY);
        ccs.getAggregateStatus().add(offlineState);
        commandState = new State(CCSCommandState.IDLE);
        ccs.getAggregateStatus().add(commandState);
        ocsCommandExecutor = new OCSCommandExecutor(this);
        ccsCommandExecutor = new CCSCommandExecutor();

        mcm.addStateChangeListener((Enum state, Enum oldState) -> {

            // FIXME: This is not really the right timestamp to use
            CCSTimeStamp timeStamp = CCSTimeStamp.currentTime();
            // There has to be a better way to do this but this is what I can implement for now - FA
            // Rafts related events
            if ((oldState == RaftsState.CLEARING || oldState == RaftsState.QUIESCENT) && state == RaftsState.INTEGRATING) {
                ccs.runInBackground(() -> {
                    try {
                        imageNameEvent = imageNameTransfer.poll(1, TimeUnit.SECONDS);
                        if (imageNameEvent == null) {
                            throw new RuntimeException("Image name could not be retreived");
                        }
                        final ImageName imageName = imageNameEvent.getImageName();
                        ocsCommandExecutor.sendEvent(new StartIntegrationEvent(DEFAULT_EVENT_PRIORITY,
                                imageNameEvent.getImageType(), imageNameEvent.getGroupId(), imageNameEvent.getImagesInSequence(), imageName.toString(), imageNameEvent.getSequenceNumber(), 
                                imageName.getSource().getCode(), imageName.getController().getCode(), imageName.getDateString(), imageName.getNumber(),
                                imageNameEvent.getIntegrationStartTime()/1000.0, imageNameEvent.getExposureTime()));
                        ocsCommandExecutor.sendEvent(new ImageReadoutParametersEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageName().toString(), config.ccdNames(), config.ccdType(),
                                config.overRows(), config.overCols(), config.readRows(), config.readCols(), config.readCols2(), config.preCols(), config.preRows(), config.postCols()));
                    } catch (RuntimeException | InterruptedException ex) {
                        LOG.log(Level.SEVERE, "Error sending StartRaftIntegrationEvent", ex);
                    }
                });
            }

            if (oldState == RaftsState.INTEGRATING && state == RaftsState.READING_OUT) {
               final ImageName imageName = imageNameEvent.getImageName(); 
               ocsCommandExecutor.sendEvent(new StartReadoutEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageType(), imageNameEvent.getGroupId(), 
                       imageNameEvent.getImagesInSequence(), imageName.toString(), imageNameEvent.getSequenceNumber(), 
                       imageName.getSource().getCode(), imageName.getController().getCode(), imageName.getDateString(), imageName.getNumber(),
                       imageNameEvent.getIntegrationStartTime()/1000.0, imageNameEvent.getExposureTime(), timeStamp.getTAIDouble()));
            }

            if (oldState == RaftsState.READING_OUT) {
                final ImageName imageName = imageNameEvent.getImageName();
                ocsCommandExecutor.sendEvent(new EndReadoutEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageType(), imageNameEvent.getGroupId(),
                        imageNameEvent.getImagesInSequence(), imageName.toString(), imageNameEvent.getSequenceNumber(), 
                        imageName.getSource().getCode(), imageName.getController().getCode(), imageName.getDateString(), imageName.getNumber(),
                        imageNameEvent.getIntegrationStartTime()/1000.0, imageNameEvent.getExposureTime(), timeStamp.getTAIDouble()));
                // For now we will always send the EndOfImageTelemetry event 300ms after the EndRoutoutEvent
                ccs.schedule(Duration.ofMillis(300), new Runnable() {
                    // imageNameEvet may change before 300ms is up.
                    private CCSImageNameEvent event = imageNameEvent;
                    final ImageName imageName = event.getImageName();

                    @Override
                    public void run() {
                        ocsCommandExecutor.sendEvent(new EndOfImageTelemetryEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageType(), imageNameEvent.getGroupId(),
                                imageNameEvent.getImagesInSequence(), imageName.toString(), event.getSequenceNumber(), 
                                imageName.getSource().getCode(), imageName.getController().getCode(), imageName.getDateString(), imageName.getNumber(),
                                event.getIntegrationStartTime()/1000.0, event.getExposureTime(),"", 0,0,0,0 /* FIXME */));
                    }
                });
            }
            CameraEvent converted = ocsStateChangeToEventConverter.convert(oldState, state, DEFAULT_EVENT_PRIORITY);
            if (converted != null) {
                ocsCommandExecutor.sendEvent(converted);
            }
            // Note, ATS shutter transitions directly from OPEN->CLOSED, without the intervening CLOSING state.
            if (state == ShutterState.OPEN) { 
                //TODO: The time should be coming from the state change event. which
                // should be generated by the shutter system itself.
                openTime = System.currentTimeMillis();
            }
            if ((oldState == ShutterState.CLOSING || oldState ==ShutterState.OPEN) && state == ShutterState.CLOSED) {
                //TODO: The time should be coming from the state change event. which
                // should be generated by the shutter system itself.
                long closeTime = System.currentTimeMillis();
                ocsCommandExecutor.sendEvent(new ShutterMotionProfileEvent(1, (closeTime-openTime)/1000.));
            } 
        });
        mcm.addEventListener((CCSEvent event) -> {
            if (event instanceof CCSAvailableFiltersEvent) {
                List<String> filters = ((CCSAvailableFiltersEvent) event).getAvailableFilters();
                ocsCommandExecutor.sendEvent(new AvailableFiltersEvent(DEFAULT_EVENT_PRIORITY, String.join(":", filters)));
            } else if (event instanceof CCSImageNameEvent) {
                imageNameTransfer.add((CCSImageNameEvent) event);
            } else if (event instanceof CCSSetFilterEvent) {
                CCSSetFilterEvent setFilter = (CCSSetFilterEvent) event;
                if (setFilter.isStart()) {
                    ocsCommandExecutor.sendEvent(new StartSetFilterEvent(DEFAULT_EVENT_PRIORITY, setFilter.getFilterName()));
                } else {
                    ocsCommandExecutor.sendEvent(new EndSetFilterEvent(DEFAULT_EVENT_PRIORITY, setFilter.getFilterName()));
                }
            } else if (event instanceof CCSSettingsAppliedEvent) {
                ocsCommandExecutor.sendEvent(new SettingsAppliedEvent(DEFAULT_EVENT_PRIORITY, ((CCSSettingsAppliedEvent) event).getSettings()));
            } else if (event instanceof CCSAtSettingsAppliedEvent) {
                ocsCommandExecutor.sendEvent(new org.lsst.sal.atcamera.event.SettingsAppliedEvent(DEFAULT_EVENT_PRIORITY, ((CCSAtSettingsAppliedEvent) event).getSettings(), ((CCSAtSettingsAppliedEvent) event).getVersion(), ((CCSAtSettingsAppliedEvent) event).getWREBSettingsVersion(), ((CCSAtSettingsAppliedEvent) event).getBonnShutterSettingsVersion()));
            }
        });

        mcm.addStatusMessageListener((StatusMessage msg) -> {
            if (msg instanceof StatusConfigurationInfo) {
                StatusConfigurationInfo sci = (StatusConfigurationInfo) msg;
                final AgentInfo originAgentInfo = msg.getOriginAgentInfo();
                final String subsystemName = originAgentInfo == null ? "unknown" : originAgentInfo.getName();
                ConfigurationInfo info = sci.getConfigurationInfo();
                if (info.getConfigurationName().startsWith("ats-wreb[")) {
                    WrebSettingsAppliedEvent wrebSettings = buConverter.convertWREBSettings(sci.getConfigurationInfo());
                    ocsCommandExecutor.sendEvent(wrebSettings);
                } else {
                    LOG.log(Level.WARNING, "Got unhandled config from {0}", subsystemName);
                }
            } else if (msg instanceof StatusSubsystemData) {
                StatusSubsystemData ssd = (StatusSubsystemData) msg;
                final AgentInfo originAgentInfo = msg.getOriginAgentInfo();
                final String subsystemName = originAgentInfo == null ? "unknown" : originAgentInfo.getName();
                String values = ssd.getSubsystemData().getValue().toString();
                if (values.contains("WREB.Temp1")) {
                    WrebTelemetry wrebTelemetry = buConverter.convertWREBTelemetry(ssd);
                    ocsCommandExecutor.sendTelemetry(wrebTelemetry);
                } else if (values.contains("Digital_V")) {
                    WrebPowerTelemetry wrebPowerTelemetry = buConverter.converWREBPowerTelemetry(ssd);
                    ocsCommandExecutor.sendTelemetry(wrebPowerTelemetry);
                } else if (values.contains("TempCCD")) {
                    VacuumTelemetry vacuumTelemetry = buConverter.convertVacuumTelemetry(ssd);
                    ocsCommandExecutor.sendTelemetry(vacuumTelemetry); 
                } else {
                    LOG.log(Level.WARNING, "Got unhandled telemetry from {0}", subsystemName);
                }
            }
        });

        ccs.addStateChangeListener((currentState, oldState) -> {
            int priority = DEFAULT_EVENT_PRIORITY;

            CameraStateChangeEvent converted = ocsCcsStateEventConverter.convert(currentState, priority);
            if (converted == null) {
                LOG.log(Level.WARNING, "Enum {0} of class {1} could not be converted to equivalent OCS state", new Object[]{currentState, currentState.getClass()});
            } else {
                ocsCommandExecutor.sendEvent(converted);
            }
        });
    }

    State<CCSCommandState> getCommandState() {
        return commandState;
    }

    public OCSBridgeConfig getConfig() {
        return config;
    }

    /**
     * Allow a user to provide an alternative implementation of the
     * OCSCommandExecutor. Used to override the default OCSCommandExecutor with
     * one that actually sends acknowledgments back to OCS.
     *
     * @param ocs
     */
    void setOCSCommandExecutor(OCSCommandExecutor ocs) {
        this.ocsCommandExecutor = ocs;
    }

    void setCCSCommandExecutor(CCSCommandExecutor ccs) {
        this.ccsCommandExecutor = ccs;
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        OCSBridge ocs = createOCSBridge();
        ToyOCSGUI gui = new ToyOCSGUI(new GUIDirectLayer(ocs), ocs.getConfig().getDevice());
        gui.setVisible(true);

    }

    /**
     * Create an OCSBridge with default configuration
     *
     * @return
     */
    static OCSBridge createOCSBridge() {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig();
        MCMConfig mcmConfig = MCMConfig.createDefaultConfig();
        ImageNameService ins = ImageNameService.testInstance("jdbc:h2:mem:test;MODE=MYSQL", Controller.OCS, Source.MainCamera, "UTC", Duration.ofHours(12), Instant.parse("2022-01-01T00:00:00.00Z"));
        return createOCSBridge(ocsConfig, mcmConfig, ins);
    }

    static OCSBridge createOCSBridge(Camera camera) {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig();
        MCMConfig mcmConfig = new MCMConfig(camera);
        ImageNameService ins = ImageNameService.testInstance("jdbc:h2:mem:test;MODE=MYSQL", Controller.OCS, camera == Camera.AUXTEL ? Source.AuxTel : Source.MainCamera, "UTC", Duration.ofHours(12), Instant.parse("2022-01-01T00:00:00.00Z"));
        return createOCSBridge(ocsConfig, mcmConfig, ins);
    }

    static OCSBridge createOCSBridge(OCSBridgeConfig config, MCMConfig mcmConfig, ImageNameService imageNamingService) {
        CCS ccs = new CCS();
        MCMLayer mcm = new MCMDirectLayer(new MCM(ccs, mcmConfig, imageNamingService));
        return new OCSBridge(config, ccs, mcm);
    }

    State getLse209State() {
        return lse209State;
    }

    State getOfflineState() {
        return offlineState;
    }

    /**
     * Super class for all OCSExecutors which forward commands to MCM
     */
    // removed abstract from class definitions 
    class ForwardToMCMExecutor extends OCSCommandExecutor.OCSExecutor {

        private final CCSCommand ccsCommand;
        private CCSCommandResponse response;
        private final SummaryState initialState;

        ForwardToMCMExecutor(CameraCommand command) {

            this(command, SummaryStateEvent.SummaryState.ENABLED);
        }

        ForwardToMCMExecutor(CameraCommand command, SummaryState initialState) {
            super(command);
            this.ccsCommand = ocsCcsCommandConverter.convert(command);
            this.initialState = initialState;

        }

        @Override
        Duration testPreconditions() throws OCSCommandExecutor.PreconditionsNotMet {
            if (!lse209State.isInState(initialState)) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command not accepted in: " + lse209State);
            }
            response = mcm.execute(ccsCommand);
            CCSAckOrNack can = response.waitForAckOrNack();
            if (can.isNack()) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command rejected: " + can.getReason());
            } else {
                return can.getDuration();
            }
        }

        @Override
        void execute() throws Exception {
            response.waitForCompletion();
        }
    }

    // 
    void execute(CameraCommand cmd) {
        if (cmd instanceof DisableCommand) {
            execute((DisableCommand) cmd);
        } else if (cmd instanceof EnableCommand) {
            execute((EnableCommand) cmd);
        } else if (cmd instanceof ExitControlCommand) {
            execute((ExitControlCommand) cmd);
        } else if (cmd instanceof EnterControlCommand) {
            execute((EnterControlCommand) cmd);
            // The next three commands override the execute method of the parent class 
            // and have to be treated as special cases and cannot be generalised to
            // be dealt with via ForwardToMCMExecutor followed by execute - we 
            // need to generalise these differently - coming soon - Farrukh 25/6/2018
        } else if (cmd instanceof TakeImagesCommand) {
            execute((TakeImagesCommand) cmd);
        } else if (cmd instanceof StartCommand) {
            execute((StartCommand) cmd);
        } else if (cmd instanceof StartImageCommand) {
            execute((StartImageCommand) cmd);
        } else if (cmd instanceof StandbyCommand) {
            execute((StandbyCommand) cmd);
        } // rest of camera commands are generalizeable in the following manner - Farrukh 25/6/2018
        else {
            ForwardToMCMExecutor fcm = new ForwardToMCMExecutor(cmd);
            ocsCommandExecutor.executeCommand(fcm);
        }
        // were throwing a run time exception before hopefully gets thrown via ocsCommandExecutor
        /*  
        else {
            throw new RuntimeException("Unrecognized command " + cmd);
        }
         */
    }

    void execute(TakeImagesCommand command) {
        OCSExecutor takeImages = new TakeImagesExecutor(command);
        ocsCommandExecutor.executeCommand(takeImages);
    }

    void execute(StartImageCommand command) {
        OCSExecutor startImage = new StartImageExecutor(command);
        ocsCommandExecutor.executeCommand(startImage);
    }

    void execute(EnterControlCommand command) {
        OCSExecutor takeControl = new EnterControlExecutor(command);
        ocsCommandExecutor.executeCommand(takeControl);
    }

    void execute(ExitControlCommand command) {
        OCSExecutor exit = new ExitExecutor(command);
        ocsCommandExecutor.executeCommand(exit);
    }

    void execute(StartCommand command) {
        OCSExecutor start = new StartExecutor(command);
        ocsCommandExecutor.executeCommand(start);
    }

    void execute(StandbyCommand command) {
        OCSExecutor standby = new StandbyExecutor(command);
        ocsCommandExecutor.executeCommand(standby);
    }

    void execute(EnableCommand command) {
        OCSExecutor enable = new EnableExecutor(command);
        ocsCommandExecutor.executeCommand(enable);
    }

    void execute(DisableCommand command) {
        OCSExecutor disable = new DisableExecutor(command);
        ocsCommandExecutor.executeCommand(disable);
    }

    void execute(CCSCommand command) {
        if (command instanceof CCSSetAvailableCommand) {
            execute((CCSSetAvailableCommand) command);
        } else if (command instanceof CCSRevokeAvailableCommand) {
            execute((CCSRevokeAvailableCommand) command);
        } else if (command instanceof CCSSimulateFaultCommand) {
            execute((CCSSimulateFaultCommand) command);
        } else if (command instanceof CCSClearFaultCommand) {
            execute((CCSClearFaultCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    void execute(CCSSetAvailableCommand command) {
        CCSExecutor setAvailable = new SetAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(setAvailable));
    }

    void execute(CCSRevokeAvailableCommand command) {
        CCSExecutor revokeAvailable = new RevokeAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(revokeAvailable));
    }

    void execute(CCSSimulateFaultCommand command) {
        CCSExecutor simulateFault = new SimulateFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(simulateFault));
    }

    void execute(CCSClearFaultCommand command) {
        CCSExecutor clearFault = new ClearFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(clearFault));
    }

    // Utility only used for testing   
    public Future<Void> waitForState(Enum state) {
        return ccs.waitForStatus(state);
    }

    CCS getCCS() {
        return ccs;
    }

    class TakeImagesExecutor extends ForwardToMCMExecutor {

        public TakeImagesExecutor(TakeImagesCommand command) {
            super(command);
        }

        @Override
        void execute() throws Exception {
            imageNameTransfer.clear();
            super.execute();
        }
    }

    class StartImageExecutor extends ForwardToMCMExecutor {

        public StartImageExecutor(StartImageCommand command) {
            super(command);

        }

        @Override
        void execute() throws Exception {
            imageNameTransfer.clear();
            super.execute();
        }
    }

    class EnterControlExecutor extends OCSExecutor {

        public EnterControlExecutor(EnterControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_AVAILABLE)) {
                throw new PreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: The argument should be a comma-delimited list of supported settings. Clearly should not be hard-wired here.
            ocsCommandExecutor.sendEvent(new SettingVersionsEvent(DEFAULT_EVENT_PRIORITY, "Normal","", "" /* FIXME */));
            lse209State.setState(SummaryState.STANDBY);
        }
    }

    class ExitExecutor extends OCSExecutor {

        public ExitExecutor(ExitControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.STANDBY)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(SummaryState.OFFLINE);
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }
    }

    /**
     * Start command is a special case, since it is a lifecycle command, but
     * must also issue a start command to the MCM (to set configuration).
     */
    class StartExecutor extends ForwardToMCMExecutor {

        public StartExecutor(StartCommand command) {
            super(command, SummaryState.STANDBY);
        }

        @Override
        void execute() throws Exception {
            super.execute();
            ocsCommandExecutor.sendEvent(new AppliedSettingsMatchStartEvent(DEFAULT_EVENT_PRIORITY, true));

            lse209State.setState(SummaryState.DISABLED);
        }
    }

    class StandbyExecutor extends OCSExecutor {

        public StandbyExecutor(StandbyCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standy command if things are happening?
            //TODO: or wait until things finish and return then?
            //TODO: The argument should be a comma-delimited list of supported settings. Clearly should not be hard-wired here.
            ocsCommandExecutor.sendEvent(new SettingVersionsEvent(DEFAULT_EVENT_PRIORITY, "Normal", "", "" /*FIXME*/));
            lse209State.setState(SummaryState.STANDBY);
        }
    }

    class EnableExecutor extends OCSExecutor {

        public EnableExecutor(EnableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(SummaryState.ENABLED);
        }
    }

    class DisableExecutor extends OCSExecutor {

        public DisableExecutor(DisableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.ENABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            // Fixme: Can we reject the disable command if we are busy?
            // What about if we are not idle?
            // Note logic here is incorrect according to Paul Lotz, we must always accept
            // the disable command.
//            if (startImageTimeout != null && !startImageTimeout.isDone()) {
//                throw new PreconditionsNotMet("Exposure in progress");
//            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standby command if things are happening?
            //TODO: or wait until things finish and return then?
            lse209State.setState(SummaryState.DISABLED);
        }
    }

    class SetAvailableExecutor extends CCSExecutor {

        private SetAvailableExecutor(CCSSetAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_PUBLISH_ONLY)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.OFFLINE_AVAILABLE);
        }

    }

    class RevokeAvailableExecutor extends CCSExecutor {

        private RevokeAvailableExecutor(CCSRevokeAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.OFFLINE_AVAILABLE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }

    }

    class SimulateFaultExecutor extends CCSExecutor {

        private SimulateFaultExecutor(CCSSimulateFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            //TODO: Should we also attempt to stop the subsystems?
            lse209State.setState(SummaryState.FAULT);
        }

    }

    class ClearFaultExecutor extends CCSExecutor {

        private ClearFaultExecutor(CCSClearFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(SummaryState.FAULT)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            lse209State.setState(SummaryState.OFFLINE);
            offlineState.setState(OfflineState.OFFLINE_PUBLISH_ONLY);
        }
    }
}
