package org.lsst.ccs.subsystem.ocsbridge.util;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Encapsulates a state, and generates state change events.
 * @author tonyj
 * @param <T> The enumeration representing the states.
 */
public class State<T extends Enum> {

    private T currentState;
    private final Class<T> enumClass;
    private final List<StateChangeListener<T>> listeners = new CopyOnWriteArrayList<>();
    private static final Logger logger = Logger.getLogger(State.class.getName());
    private final ConcurrentLinkedDeque<Runnable> workQueue = new ConcurrentLinkedDeque<>();

    /**
     * Create a new state
     * @param initialState Initial state
     */
    public State(T initialState) {
        this.enumClass = (Class<T>) initialState.getClass();
        currentState = initialState;
    }

    /**
     * Changes the current state. Generates a status change notification when
     * the status is changed.
     *
     * @param state The new state
     */
    public void setState(T state) {
        if (currentState != state) {
            T oldState = currentState;
            currentState = state;
            logger.log(Level.INFO, String.format("State Changed %s: %s->%s %d",currentState.getClass().getSimpleName(), oldState, currentState, this.hashCode()));

            // Instead of calling the listeners immediately we queue them up. This ensures that
            // if a listener triggers a state change the order of executution will be more determinate.
            // However the thread safety of this code has not really been full thought through
            boolean wasEmpty;
            synchronized (this) {
                // TODO: Just because the queue was not empty when we entered this code
                // does not mean that it will not be empty when we finish queuing the state changes.
                wasEmpty = workQueue.isEmpty();
                listeners.forEach((l) -> {
                    workQueue.add(() -> l.stateChanged(state, oldState));
                });
            }
            if (wasEmpty) {
                for (;;) {
                    Runnable runnable = workQueue.pollFirst();
                    if (runnable == null) break;
                    runnable.run();
                }
            }
        }
    }

    public T getState() {
        return currentState;
    }

    public Class<T> getEnumClass() {
        return enumClass;
    }

    public boolean isInState(T state) {
        return currentState == state;
    }
    
    public void addStateChangeListener(StateChangeListener<T> listener) {
        listeners.add(listener);
    }
    
    public void removeStateChangeListener(StateChangeListener<T> listener) {
        listeners.remove(listener);
    }

    /**
     * Check the state and generate an exception if the current state does not
     * match.
     *
     * @param states The expected state, may be a list of possible states
     * @throws InvalidStateException If the state does not match.
     */
    public void checkState(T... states) throws InvalidStateException {
        for (T state : states) {
            if (state == currentState) {
                return;
            }
        }
        throw new InvalidStateException(String.format("State: %s expected %s was %s", enumClass.getSimpleName(), Arrays.toString(states), currentState));
    }

    public static class InvalidStateException extends RuntimeException {

        private static final long serialVersionUID = -8878766853816068565L;

        public InvalidStateException(String reason) {
            super(reason);
        }
    }

    public static interface StateChangeListener<T extends Enum> {
        void stateChanged(T state, T oldState);
    }

    @Override
    public String toString() {
        return "State{" + enumClass.getSimpleName() +" = "+ currentState +'}';
    }
}
