package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCSEvent;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * A trivial Filter simulation.
 *
 * @author tonyj 
 */
public class Filter {

    public enum FilterState {

        ROTATING, UNLOADED, LOADED, UNLOADING, LOADING
    }

    private enum InstalledFilter {
        NONE(0), U(10), G(9), R(1), I(9), X(100);

        private final short serial;

        InstalledFilter(int serial) {
            this.serial = (short) serial;
        }

        public short getSerial() {
            return serial;
        }

        @Override
        public String toString() {
            if (this == NONE) {
                return super.toString();
            } else {
                return super.toString().toLowerCase() + "-" + serial;
            }
        }
    }

    static final Duration LOAD_TIME = Duration.ofMillis(15000);
    static final Duration ROTATION_TIME_PER_DEGREE = Duration.ofMillis(100);
    static final Duration UNLOAD_TIME = Duration.ofMillis(15000);

    private final List<String> availableFilters = new ArrayList<>();
    private InstalledFilter currentFilter = InstalledFilter.NONE;
    private int currentRotationPosition = 0;

    private final State filterState;
    private final CCS ccs;

    Filter(CCS ccs) {
        this.ccs = ccs;
        for (InstalledFilter f : InstalledFilter.values()) {
            availableFilters.add(f.toString());
        }
        filterState = new State(FilterState.UNLOADED);
        ccs.getAggregateStatus().add(filterState);
    }

    boolean filterIsAvailable(String filter) {
        return availableFilters.contains(filter);
    }

    List<String> getAvailableFilters() {
        return Collections.unmodifiableList(availableFilters);
    }

    void setFilter(String filter) throws InterruptedException, ExecutionException, TimeoutException {
        int position = availableFilters.indexOf(filter);
        if (position < 0) {
            throw new IllegalArgumentException("Invalid filter: " + filter);
        }
        InstalledFilter requestedFilter = InstalledFilter.values()[position];
        if (requestedFilter.equals(currentFilter)) {
            // No-op?
        } else if (requestedFilter == InstalledFilter.NONE) {
            filterState.setState(FilterState.UNLOADING);
            Future<Void> waitForUnloaded = ccs.waitForStatus(FilterState.UNLOADED);
            ccs.schedule(UNLOAD_TIME, () -> {
                filterState.setState(FilterState.UNLOADED);
                currentFilter = InstalledFilter.NONE;
            });
            waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
        } else {
            if (currentFilter != InstalledFilter.NONE) {
                filterState.setState(FilterState.UNLOADING);
                Future<Void> waitForUnloaded = ccs.waitForStatus(FilterState.UNLOADED);
                ccs.schedule(UNLOAD_TIME, () -> {
                    filterState.setState(FilterState.UNLOADED);
                    currentFilter = InstalledFilter.NONE;
                });
                waitForUnloaded.get(UNLOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);               
            }            
            int targetRotation = (position - 1) * 360 / 5;
            if (currentRotationPosition != targetRotation) {
                // FIXME: Always rotates in same direction, probably not realistic
                int degreesToRotate = Math.abs(currentRotationPosition - targetRotation) % 360;
                filterState.setState(FilterState.ROTATING);
                Future<Void> waitForRotation = ccs.waitForStatus(FilterState.UNLOADED);
                Duration rotationTime = ROTATION_TIME_PER_DEGREE.multipliedBy(degreesToRotate);
                ccs.schedule(rotationTime, () -> {
                    filterState.setState(FilterState.UNLOADED);
                });
                waitForRotation.get(rotationTime.toMillis() * 2, TimeUnit.MILLISECONDS);
                currentRotationPosition = targetRotation;
            }
            filterState.setState(FilterState.LOADING);
            Future<Void> waitForUnloaded = ccs.waitForStatus(FilterState.LOADED);
            ccs.schedule(LOAD_TIME, () -> {
                filterState.setState(FilterState.LOADED);
                currentFilter = requestedFilter;
            });
            waitForUnloaded.get(LOAD_TIME.toMillis() * 2, TimeUnit.MILLISECONDS);
        }
    }

    short getFilterId() {
        return currentFilter.getSerial();
    }

    public static class CCSAvailableFiltersEvent extends CCSEvent {

        private final List<String> availableFilters;

        // TODO: Should not need to be public
        public CCSAvailableFiltersEvent(List<String> availableFilters) {
            this.availableFilters = availableFilters;
        }

        public List<String> getAvailableFilters() {
            return availableFilters;
        }

        @Override
        public String toString() {
            return "CCSAvailableFiltersEvent{" + "availableFilters=" + availableFilters + '}';
        }

    }
}
