package org.lsst.ccs.subsystem.ocsbridge.sim;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.AggregateStatus;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.Rafts.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.ccs.subsystem.ocsbridge.util.Event;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * A simple simulation of the MCM. Designed to be used either directly within
 * the OCSSimulation, or as a standalone process.
 *
 * @author tonyj
 */
public class MCM {

    private final MCMConfig config;

    public enum TakeImageReadinessState {

        NOT_READY, GETTING_READY, READY
    };

    public enum CalibrationState {
        DISABLED, ENABLED, INTEGRATING
    }

    private final CCS ccs;
    private final State takeImageReadinessState;
    private final State calibrationState;
    private final Shutter shutter;
    private final Rafts rafts;
    private final Filter fcs;
    private final ImageNames imageNames = new ImageNames();

    private ScheduledFuture<?> startImageTimeout;
    private ScheduledFuture<?> shutterPrepSchedule;
    private ScheduledFuture<?> raftsClearSchedule;
    private long integrationStartTime;
    private final List<String> imagesAlreadyTaken = new ArrayList<>();
    private List<String> imagesToBeTaken = new ArrayList<>();

    public MCM(CCS ccs, MCMConfig config) {
        this.ccs = ccs;
        this.config = config;
        takeImageReadinessState = new State(TakeImageReadinessState.NOT_READY);
        ccs.getAggregateStatus().add(takeImageReadinessState);
        calibrationState = new State(CalibrationState.DISABLED);
        ccs.getAggregateStatus().add(calibrationState);
        shutter = new Shutter(ccs);
        rafts = new Rafts(ccs);
        if (config.hasFilterChanger()) {
           fcs = new Filter(ccs);
        } else {
            fcs = null;
        }

        // We are ready to take an image only if the rafts have been cleared, and the shutter
        // has been prepared.
        ccs.addStateChangeListener((state, oldState) -> {
            AggregateStatus as = ccs.getAggregateStatus();
            if (as.hasState(Rafts.RaftsState.QUIESCENT, Shutter.ShutterReadinessState.READY)) {
                takeImageReadinessState.setState(TakeImageReadinessState.READY);
            } else if (!as.hasState(TakeImageReadinessState.GETTING_READY)) {
                takeImageReadinessState.setState(TakeImageReadinessState.NOT_READY);
            }
            if (oldState == RaftsState.CLEARING || oldState == RaftsState.READING_OUT) {
                integrationStartTime = System.currentTimeMillis();
            }
        });
        if (fcs != null) {
            ccs.schduleAtFixedRate(10, TimeUnit.SECONDS, () -> {
                CCSFilterTelemetry telemetry = new CCSFilterTelemetry(fcs.getFilterId());
                ccs.fireEvent(telemetry);
            });
        }
    }

    CCSCommandResponse execute(CCSCommand command) {
        if (command instanceof CCSInitImageCommand) {
            return execute((CCSInitImageCommand) command);
        } else if (command instanceof CCSTakeImagesCommand) {
            return execute((CCSTakeImagesCommand) command);
        } else if (command instanceof CCSSetFilterCommand) {
            return execute((CCSSetFilterCommand) command);
        } else if (command instanceof CCSInitGuidersCommand) {
            return execute((CCSInitGuidersCommand) command);
        } else if (command instanceof CCSStartImageCommand) {
            return execute((CCSStartImageCommand) command);
        } else if (command instanceof CCSEndImageCommand) {
            return execute((CCSEndImageCommand) command);
        } else if (command instanceof CCSClearCommand) {
            return execute((CCSClearCommand) command);
        } else if (command instanceof CCSDiscardRowsCommand) {
            return execute((CCSDiscardRowsCommand) command);
        } else if (command instanceof CCSStartCommand) {
            return execute((CCSStartCommand) command);
        } else if (command instanceof CCSEnableCalibrationCommand) {
            return execute((CCSEnableCalibrationCommand) command);
        } else if (command instanceof CCSDisableCalibrationCommand) {
            return execute((CCSDisableCalibrationCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    CCSCommandResponse execute(CCSInitImageCommand command) {
        InitImageExecutor executor = new InitImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSTakeImagesCommand command) {
        TakeImagesExecutor executor = new TakeImagesExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSSetFilterCommand command) {
        SetFilterExecutor executor = new SetFilterExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSInitGuidersCommand command) {
        InitGuidersExecutor executor = new InitGuidersExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSStartImageCommand command) {
        StartImageExecutor executor = new StartImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSEndImageCommand command) {
        EndImageExecutor executor = new EndImageExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSClearCommand command) {
        ClearExecutor executor = new ClearExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDiscardRowsCommand command) {
        DiscardRowsExecutor executor = new DiscardRowsExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSStartCommand command) {
        StartExecutor executor = new StartExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSEnableCalibrationCommand command) {
        EnableCalibrationExecutor executor = new EnableCalibrationExecutor(command);
        return new CCSCommandResponse(executor);
    }

    CCSCommandResponse execute(CCSDisableCalibrationCommand command) {
        DisableCalibrationExecutor executor = new DisableCalibrationExecutor(command);
        return new CCSCommandResponse(executor);
    }

    //TODO: Remove me
    CCS getCCS() {
        return ccs;
    }

    Rafts getRafts() {
        return rafts;
    }
    
    MCMConfig getConfig() {
        return config;
    }

    class InitImageExecutor extends CCSExecutor {

        private final CCSInitImageCommand command;

        public InitImageExecutor(CCSInitImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (command.getDeltaT() <= 0 || command.getDeltaT() > 15) {
                throw new CCSPreconditionsNotMet("Invalid deltaT: " + command.getDeltaT());
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            Duration takeImagesExpected = Duration.ofMillis((long) (command.getDeltaT() * 1000));
            takeImageReadinessState.setState(TakeImageReadinessState.GETTING_READY);
            raftsClearSchedule = ccs.schedule(takeImagesExpected.minus(Rafts.CLEAR_TIME), () -> {
                rafts.clear(1);
            });
            shutterPrepSchedule = ccs.schedule(takeImagesExpected.minus(Shutter.PREP_TIME), () -> {
                shutter.prepare();
            });
        }

    }

    class TakeImagesExecutor extends CCSExecutor {

        private final CCSTakeImagesCommand command;

        public TakeImagesExecutor(CCSTakeImagesCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {

            // check to see if any of these images is repeated - 
            if (command.getNumImages() <= 0 || command.getNumImages() > config.getMaxImagesPerSequence() || 
                    command.getExpTime() < config.getMinExposeTime() || command.getExpTime() > config.getMaxExposeTime()) {
                throw new CCSPreconditionsNotMet("Invalid argument");
            }
            String imageSequenceName = command.getImageSequenceName();
            if (imageSequenceName == null || imageSequenceName.trim().length() == 0) {
                throw new CCSPreconditionsNotMet("Invalid empty imageSequenceName");
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }

            // CheckNames will throw CCSPreconditionsNotMet if imageSequenceName is invalid
            imagesToBeTaken = imageNames.checkNames(command.getImageSequenceName(), command.getNumImages());

            if (!Collections.disjoint(imagesToBeTaken, imagesAlreadyTaken)) {
                throw new CCSPreconditionsNotMet("Image Names reused ! " + imagesToBeTaken.retainAll(imagesAlreadyTaken));
            }
            // Worse case estimate
            return Duration.ofMillis((long) (command.getExpTime() * 1000)).plus(Shutter.MOVE_TIME).plus(Rafts.READOUT_TIME).multipliedBy(command.getNumImages());
        }

        @Override
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException, CCSPreconditionsNotMet {
            Duration exposeTime = Duration.ofMillis((long) (command.getExpTime() * 1000));
            if (command.getNumImages() > 1) {
                shutter.startImageSequence();
            }

            try {

                for (int i = 0; i < command.getNumImages(); i++) {
                    Future waitUntilReady = ccs.waitForStatus(TakeImageReadinessState.READY);
                    if (takeImageReadinessState.isInState(TakeImageReadinessState.NOT_READY)) {
                        rafts.clear(1);
                        shutter.prepare();
                    } else if (takeImageReadinessState.isInState(TakeImageReadinessState.GETTING_READY)) {
                        boolean raftClearEnd = raftsClearSchedule.cancel(false);
                        boolean shutterPrepEnd = shutterPrepSchedule.cancel(false);
                        if (raftClearEnd) {
                            rafts.clear(1);
                        }
                        if (shutterPrepEnd) {
                            shutter.prepare();
                        }
                    }

                    waitUntilReady.get(1, TimeUnit.SECONDS);

                    String imageName = imagesToBeTaken.get(i);
                    imagesAlreadyTaken.add(imageName);
                    ccs.fireEvent(new CCSImageNameEvent(command.getImageSequenceName(), command.getNumImages(), imageName, i, integrationStartTime, command.getExpTime()));

                    if (command.isShutter()) {
                        shutter.expose(exposeTime);
                        rafts.expose(exposeTime.plus(Shutter.MOVE_TIME), imageName);
                        // For the last exposure we only wait until the readout starts
                        // For other exposures we must wait until readout is complete
                        Future waitUntilDone = ccs.waitForStatus(i + 1 < command.getNumImages() ? Rafts.RaftsState.QUIESCENT : Rafts.RaftsState.READING_OUT);
                        waitUntilDone.get(exposeTime.plus(Shutter.MOVE_TIME).plus(Rafts.READOUT_TIME).plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                    } else {
                        rafts.expose(exposeTime, imageName);
                        Future waitUntilDone = ccs.waitForStatus(i + 1 < command.getNumImages() ? Rafts.RaftsState.QUIESCENT : Rafts.RaftsState.READING_OUT);
                        waitUntilDone.get(exposeTime.plus(Shutter.MOVE_TIME).plus(Rafts.READOUT_TIME).plusSeconds(10).toMillis(), TimeUnit.MILLISECONDS);
                    }
                }

            } finally {
                shutter.endImageSequence();
            }
        }

    }

    public static class CCSImageNameEvent extends Event {

        private final String imageSequenceName;
        private final int imagesInSequence;
        private final String imageName;
        private final int sequenceNumber;
        private final long integrationStartTime;
        private final double exposureTime;

        public CCSImageNameEvent(String imageSequenceName, int imagesInSequence, String imageName, int sequenceNumber, long integrationStartTime, double exposureTime) {
            this.imageSequenceName = imageSequenceName;
            this.imagesInSequence = imagesInSequence;
            this.imageName = imageName;
            this.sequenceNumber = sequenceNumber;
            this.integrationStartTime = integrationStartTime;
            this.exposureTime = exposureTime;
        }

        public String getImageSequenceName() {
            return imageSequenceName;
        }

        public int getImagesInSequence() {
            return imagesInSequence;
        }

        public String getImageName() {
            return imageName;
        }

        public int getSequenceNumber() {
            return sequenceNumber;
        }

        public long getIntegrationStartTime() {
            return integrationStartTime;
        }

        public double getExposureTime() {
            return exposureTime;
        }

        @Override
        public String toString() {
            return "CCSImageNameEvent{" + "imageSequenceName=" + imageSequenceName + ", imagesInSequence=" + imagesInSequence + ", imageName=" + imageName + ", sequenceNumber=" + sequenceNumber + ", integrationStartTime=" + integrationStartTime + ", exposureTime=" + exposureTime + '}';
        }

    }

    class SetFilterExecutor extends CCSExecutor {

        private final CCSSetFilterCommand command;

        public SetFilterExecutor(CCSSetFilterCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (fcs == null) {
                throw new CCSPreconditionsNotMet("setFilter command not supported");
            }
            
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            if (!fcs.filterIsAvailable(command.getFilterName())) {
                throw new CCSPreconditionsNotMet("Invalid filter: " + command.getFilterName());
            }
            // Worse case
            return Filter.ROTATION_TIME_PER_DEGREE.multipliedBy(360).plus(Filter.LOAD_TIME).plus(Filter.UNLOAD_TIME);
        }

        @Override
        protected void execute() throws Exception {
            ccs.fireEvent(new CCSSetFilterEvent(command.getFilterName(), true));
            fcs.setFilter(command.getFilterName());
            ccs.fireEvent(new CCSSetFilterEvent(command.getFilterName(), false));
        }
    }

    public static class CCSSetFilterEvent extends Event {

        private final String filterName;
        private final boolean start;

        public CCSSetFilterEvent(String filterName, boolean start) {
            this.filterName = filterName;
            this.start = start;
        }

        public String getFilterName() {
            return filterName;
        }

        public boolean isStart() {
            return start;
        }

        @Override
        public String toString() {
            return "CCSSetFilterEvent{" + "filterName=" + filterName + ", start=" + start + '}';
        }
    }

    public static class CCSSettingsAppliedEvent extends Event {

        private final String settings;
        private final double timestamp;

        public CCSSettingsAppliedEvent(String settings, double timestamp) {

            this.settings = settings;
            this.timestamp = timestamp;
        }

        public String getSettings() {
            return settings;
        }

        public double getTimeStamp() {
            return timestamp;
        }

        @Override
        public String toString() {
            return "SettingsApplied{" + "settings=" + settings + ", timestamp=" + timestamp + '}';
        }

    }

    public static class CCSFilterTelemetry extends Event {

        private final short filterId;

        public CCSFilterTelemetry(short filterId) {
            this.filterId = filterId;
        }

        public short getFilterId() {
            return filterId;
        }

        @Override
        public String toString() {
            return "CCSFilterTelemetry{" + "filterId=" + filterId + '}';
        }

    }

    private class InitGuidersExecutor extends CCSExecutor {

        private final CCSInitGuidersCommand command;

        public InitGuidersExecutor(CCSInitGuidersCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            // FIXME: Currently this does not actually do anything
        }
    }

    private class EnableCalibrationExecutor extends CCSExecutor {

        private final CCSEnableCalibrationCommand command;

        public EnableCalibrationExecutor(CCSEnableCalibrationCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.DISABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            calibrationState.setState(CalibrationState.ENABLED);
        }
    }

    private class DisableCalibrationExecutor extends CCSExecutor {

        private final CCSDisableCalibrationCommand command;

        public DisableCalibrationExecutor(CCSDisableCalibrationCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            calibrationState.setState(CalibrationState.DISABLED);
        }
    }

    private class StartExecutor extends CCSExecutor {

        private final CCSStartCommand command;

        public StartExecutor(CCSStartCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() {
            if (fcs != null) {
                ccs.fireEvent(new CCSAvailableFiltersEvent(fcs.getAvailableFilters()));
            }
            CCSTimeStamp now = CCSTimeStamp.currentTime();
            ccs.fireEvent(new CCSSettingsAppliedEvent(command.getConfiguration(), now.getTAIDouble()));
        }
    }

    private class ClearExecutor extends CCSExecutor {

        private final CCSClearCommand command;

        public ClearExecutor(CCSClearCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (command.getNClears() <= 0 || command.getNClears() > 15) {
                throw new CCSPreconditionsNotMet("Invalid nClears: " + command.getNClears());
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            return Rafts.CLEAR_TIME.multipliedBy(command.getNClears());
        }

        @Override
        protected void execute() throws InterruptedException, ExecutionException, TimeoutException {
            rafts.clear(command.getNClears());
            // TODO: Note, unlike initImages, the clear command remains active until the clears are complete (Correct?)
            Future waitUntilClear = ccs.waitForStatus(Rafts.RaftsState.QUIESCENT);
            waitUntilClear.get(Rafts.CLEAR_TIME.multipliedBy(command.getNClears()).plus(Duration.ofSeconds(1)).toMillis(), TimeUnit.MILLISECONDS);
        }

    }

    private class StartImageExecutor extends CCSExecutor {

        private final CCSStartImageCommand command;

        public StartImageExecutor(CCSStartImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {

            if (command.getTimeout() < 1 | command.getTimeout() > 120) {
                throw new CCSPreconditionsNotMet("Invalid argument");
            }
            if (startImageTimeout != null && !startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("Exposure in progress");
            }
            if (!calibrationState.isInState(CalibrationState.ENABLED)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }

            imagesToBeTaken = imageNames.checkNames(command.getImageSequenceName(), 1);

            if (!Collections.disjoint(imagesToBeTaken, imagesAlreadyTaken)) {
                throw new CCSPreconditionsNotMet("Image Names reused ! " + imagesToBeTaken.retainAll(imagesAlreadyTaken));
            }

            return Duration.ofSeconds(1);
        }

        @Override
        protected void execute() throws Exception {
            Future waitUntilReady = ccs.waitForStatus(TakeImageReadinessState.READY);
            // FIXME: It is not necessary to always clear and prepare the shutter, especially
            // if we are not actually going to open the shutter.
            if (takeImageReadinessState.isInState(TakeImageReadinessState.NOT_READY)) {
                rafts.clear(1);
                shutter.prepare();
            }

            waitUntilReady.get(1, TimeUnit.SECONDS);

            String imageName = imagesToBeTaken.get(0);

            if (imagesAlreadyTaken.contains(imageName)) {
                throw new CCSPreconditionsNotMet("Image requested already exists " + imageName);
            } else {
                imagesAlreadyTaken.add(imageName);
            }

            ccs.fireEvent(new CCSImageNameEvent(command.getImageSequenceName(), 1, imageName, 0, integrationStartTime, 0));
            if (command.isShutterOpen()) {
                shutter.open();
                rafts.startExposure();
                // FIXME: Wait for shutter to open? right now we return immediately
            } else {
                rafts.startExposure();
            }
            calibrationState.setState(CalibrationState.INTEGRATING);
            startImageTimeout = ccs.schedule(Duration.ofMillis((long) (command.getTimeout() * 1000)), () -> {
                imageTimeout();
            });
        }
    }

    /**
     * Called if the timeout for a takeImages occurs
     */
    private void imageTimeout() {
        // FIXME: Is this a NOOP if the shutter is already closed?
        shutter.close();
        rafts.endExposure(false);
        calibrationState.setState(CalibrationState.ENABLED);
    }

    private class EndImageExecutor extends CCSExecutor {

        private final CCSEndImageCommand command;

        public EndImageExecutor(CCSEndImageCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.INTEGRATING)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            if (startImageTimeout == null || startImageTimeout.isDone()) {
                throw new CCSPreconditionsNotMet("No exposure in progress");
            }
            return Shutter.MOVE_TIME;
        }

        @Override
        protected void execute() throws Exception {
            if (!startImageTimeout.cancel(false)) {
                throw new RuntimeException("Image exposure already timed out");
            }
            Future waitUntilClosed = ccs.waitForStatus(ShutterState.CLOSED);
            shutter.close();
            waitUntilClosed.get(Shutter.MOVE_TIME.plus(Duration.ofSeconds(1)).toMillis(), TimeUnit.MILLISECONDS);
            rafts.endExposure(true);
            calibrationState.setState(CalibrationState.ENABLED);
        }
    }

    private class DiscardRowsExecutor extends CCSExecutor {

        private final CCSDiscardRowsCommand command;

        public DiscardRowsExecutor(CCSDiscardRowsCommand command) {
            this.command = command;
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!calibrationState.isInState(CalibrationState.INTEGRATING)) {
                throw new CCSPreconditionsNotMet("Invalid calibration state " + calibrationState);
            }
            if (startImageTimeout == null || startImageTimeout.isDone()) {
                throw new CCSCommand.CCSPreconditionsNotMet("No exposure in progress");
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            // FIXME: Nothing actually happens, should at least generate some events.
        }
    }
}
