package org.lsst.ccs.subsystem.ocsbridge.sim;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Assign image names from imageSequenceNames based on the proposal for image name
 * handling linked below.
 * @author Farrukh Azfar
 * @see <a href="https://confluence.lsstcorp.org/x/C5piB">Image naming and visit definition conventions</a>
 */
public class ImageNames {

    private static final Pattern SEQUENCE_NAME_PATTERN = Pattern.compile("^(.*?)(\\d+)$");

    /**
     * Allocate image names based on the imageSequenceName and the number of images
     * requested.
     * @param imageBaseName The requested imageSequenceName
     * @param numImages The number of images requested.
     * @return <code>true</code> if the generation of image names succeeded, <code>false</code>
     * otherwise. If this method returns false then a future call to #getErrorMessage will return 
     * the reason.
     */
    List<String> checkNames(String imageBaseName, int numImages) throws CCSPreconditionsNotMet {

          List<String> imageNames = new ArrayList<>();

        
        
        Matcher matcher = SEQUENCE_NAME_PATTERN.matcher(imageBaseName);

        if (!matcher.matches()) {

           throw new CCSPreconditionsNotMet("Did not match prescribed notation");
        } else {
            String stem = matcher.group(1);
            String numberString = matcher.group(2);
            int numberOfDigits = numberString.length();
            int number = Integer.parseInt(numberString);
            String imageFormat = String.format("%%s%%0%dd", numberOfDigits);

            if (Integer.toString(number + numImages).length() > numberOfDigits) {
                
                throw new CCSPreconditionsNotMet("You've exceeded the size of the allocated number field");    
            }

            System.out.printf("stem: %s number as string: %s number size: %d number: %d image format: %s\n",
                    stem, numberString, numberOfDigits, number, imageFormat);

            for (int i = 0; i < numImages; i++) {
                int imageNumber = number + i;
                String imageName = String.format(imageFormat, stem, imageNumber);
                imageNames.add(imageName);
                System.out.printf("Image %d name: %s\n", i, imageName);
            }
         
            

        }
        return imageNames;
    }

   

}
