package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.sim.Filter;
import org.lsst.ccs.subsystem.ocsbridge.util.Event;
import org.lsst.ccs.subsystem.ocsbridge.util.AggregateStatus;
import java.awt.Component;
import java.awt.Font;
import java.awt.event.KeyListener;
import java.awt.event.MouseListener;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Formatter;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.logging.StreamHandler;
import javax.swing.AbstractButton;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.plaf.FontUIResource;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSimulateFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSRevokeAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearFaultCommand;
import org.lsst.sal.camera.command.ClearCommand;
import org.lsst.sal.camera.command.DisableCalibrationCommand;
import org.lsst.sal.camera.command.DisableCommand;
import org.lsst.sal.camera.command.DiscardRowsCommand;
import org.lsst.sal.camera.command.EnableCalibrationCommand;
import org.lsst.sal.camera.command.EnableCommand;
import org.lsst.sal.camera.command.EndImageCommand;
import org.lsst.sal.camera.command.EnterControlCommand;
import org.lsst.sal.camera.command.ExitControlCommand;
import org.lsst.sal.camera.command.InitGuidersCommand;
import org.lsst.sal.camera.command.InitImageCommand;
import org.lsst.sal.camera.command.SetFilterCommand;
import org.lsst.sal.camera.command.StandbyCommand;
import org.lsst.sal.camera.command.StartCommand;
import org.lsst.sal.camera.command.StartImageCommand;
import org.lsst.sal.camera.command.TakeImagesCommand;

/**
 *
 * @author tonyj
 */
public final class ToyOCSGUI extends javax.swing.JFrame {

    private static final long serialVersionUID = -7536384356408611374L;

    private GUILayer guiInterface;
    private Map<String, JComboBox> statusMap = new HashMap<>();

    /**
     * Creates new form ToyOCSGUI
     *
     * @param guiLayer
     */
    ToyOCSGUI(GUILayer guiLayer, OCSBridgeConfig.Device device) {
        this.guiInterface = guiLayer;
        initializeFontSize();
        initComponents();
        if (!guiLayer.supportsCCSCommands()) {
            remove(commandPanel2);
        }
        if (device == OCSBridgeConfig.Device.AUXTEL) {
            commandPanel1.remove(setFilterButton);
            commandPanel1.remove(filterComboBox);
            commandPanel1.remove(initGuidersButton);
            commandPanel1.remove(roiSpecTextField);
            commandPanel1.remove(roiLabel);
        }
        this.setTitle("OCS GUI for " + device);
        AggregateStatus aggregateStatus = guiLayer.getAggregateStatus();
        aggregateStatus.getStates().forEach((state) -> {
            createStateComboBox(state.getState());
        });
        guiLayer.addStateChangeListener((newState, oldState) -> {
            SwingUtilities.invokeLater(() -> {
                JComboBox combo = statusMap.get(newState.getClass().getSimpleName());
                if (combo == null) {
                    combo = createStateComboBox(newState);
                    statusPanel.revalidate();
                }
                combo.setSelectedItem(newState);

            });
        });
        DefaultComboBoxModel<String> availableFilters = new DefaultComboBoxModel<>(new String[]{"NONE"});
        filterComboBox.setModel(availableFilters);
        guiLayer.addEventListener((Event event) -> {
            if (event instanceof Filter.CCSAvailableFiltersEvent) {
                final List<String> filters = ((Filter.CCSAvailableFiltersEvent) event).getAvailableFilters();
                SwingUtilities.invokeLater(() -> {
                    availableFilters.removeAllElements();
                    filters.forEach((filter) -> {
                        availableFilters.addElement(filter);
                    });
                });
            }
        });

        Logger logger = Logger.getLogger(OCSBridge.class.getName()).getParent();
        TextAreaHandler handler = new TextAreaHandler();
        handler.setFormatter(new Formatter() {

            @Override
            public String format(LogRecord record) {

                return String.format("[%tc] %s\n", record.getMillis(), formatMessage(record));
            }

        });
        logger.addHandler(handler);
    }

    private void initializeFontSize() {
        String fontSizeParam = System.getProperty("ocs-gui.fontSize");
        if (fontSizeParam != null) {
            float multiplier = Integer.parseInt(fontSizeParam) / 100.0f;
            UIDefaults defaults = UIManager.getDefaults();
            int i = 0;
            for (Enumeration e = defaults.keys(); e.hasMoreElements(); i++) {
                Object key = e.nextElement();
                Object value = defaults.get(key);
                if (value instanceof Font) {
                    Font font = (Font) value;
                    int newSize = Math.round(font.getSize() * multiplier);
                    if (value instanceof FontUIResource) {
                        defaults.put(key, new FontUIResource(font.getName(), font.getStyle(), newSize));
                    } else {
                        defaults.put(key, new Font(font.getName(), font.getStyle(), newSize));
                    }
                }
            }
        }
    }

    private JComboBox createStateComboBox(Enum state) {
        String name = state.getClass().getSimpleName();
        Box box = Box.createHorizontalBox();
        box.add(new JLabel(name));
        box.add(Box.createHorizontalStrut(10));
        JComboBox combo = new JComboBox(state.getClass().getEnumConstants());
        combo.setEditable(false);
        combo.setSelectedItem(state);
        setReadonly(combo);
        box.add(combo);
        box.add(Box.createHorizontalGlue());
        statusPanel.add(box);
        statusMap.put(name, combo);
        return combo;
    }

    /**
     * Taken from http://stackoverflow.com/questions/23500183
     */
    private void setReadonly(JComboBox combo) {
        Component editorComponent = combo.getEditor().getEditorComponent();
        if (editorComponent instanceof JTextField) {
            ((JTextField) editorComponent).setEditable(false);
        }

        for (Component childComponent : combo.getComponents()) {
            if (childComponent instanceof AbstractButton) {
                childComponent.setEnabled(false);
                final MouseListener[] listeners = childComponent.getListeners(MouseListener.class);
                for (MouseListener listener : listeners) {
                    childComponent.removeMouseListener(listener);
                }
            }
        }

        final MouseListener[] mouseListeners = combo.getListeners(MouseListener.class);
        for (MouseListener listener : mouseListeners) {
            combo.removeMouseListener(listener);
        }

        final KeyListener[] keyListeners = combo.getListeners(KeyListener.class);
        for (KeyListener keyListener : keyListeners) {
            combo.removeKeyListener(keyListener);
        }

        combo.setFocusable(false);

        //box.getActionMap().clear(); //no effect
        //box.getInputMap().clear();
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        statusPanel = new javax.swing.JPanel();
        commandPanel2 = new javax.swing.JPanel();
        setAvailableButton = new javax.swing.JButton();
        revokeAvailableButton = new javax.swing.JButton();
        simulateFault = new javax.swing.JButton();
        clearFaultButton = new javax.swing.JButton();
        javax.swing.JPanel commandPanel = new javax.swing.JPanel();
        enterControlButton = new javax.swing.JButton();
        exitButton = new javax.swing.JButton();
        startButton = new javax.swing.JButton();
        standbyButton = new javax.swing.JButton();
        enableButton = new javax.swing.JButton();
        disableButton = new javax.swing.JButton();
        startTextField = new javax.swing.JTextField();
        commandPanel1 = new javax.swing.JPanel();
        initImageButton = new javax.swing.JButton();
        jLabel4 = new javax.swing.JLabel();
        deltaTSpinner = new javax.swing.JSpinner();
        takeImagesButton = new javax.swing.JButton();
        jLabel5 = new javax.swing.JLabel();
        nImagesSpinner = new javax.swing.JSpinner();
        jLabel6 = new javax.swing.JLabel();
        exposureSpinner = new javax.swing.JSpinner();
        openShutterCheckBox = new javax.swing.JCheckBox();
        setFilterButton = new javax.swing.JButton();
        filterComboBox = new javax.swing.JComboBox();
        scienceCheckBox = new javax.swing.JCheckBox();
        guiderCheckBox = new javax.swing.JCheckBox();
        visitNameTextField = new javax.swing.JTextField();
        initGuidersButton = new javax.swing.JButton();
        roiSpecTextField = new javax.swing.JTextField();
        roiLabel = new javax.swing.JLabel();
        wavefrontCheckBox = new javax.swing.JCheckBox();
        enableCalibrationButton = new javax.swing.JButton();
        disableCalibrationButton = new javax.swing.JButton();
        jScrollPane2 = new javax.swing.JScrollPane();
        logTextArea = new javax.swing.JTextArea();
        jPanel1 = new javax.swing.JPanel();
        clearButton = new javax.swing.JButton();
        nClearSpinner = new javax.swing.JSpinner();
        jLabel2 = new javax.swing.JLabel();
        startImageButton = new javax.swing.JButton();
        openShutterCheckBox1 = new javax.swing.JCheckBox();
        scienceCheckBox1 = new javax.swing.JCheckBox();
        wavefrontCheckBox1 = new javax.swing.JCheckBox();
        guiderCheckBox1 = new javax.swing.JCheckBox();
        visitNameTextField1 = new javax.swing.JTextField();
        jLabel7 = new javax.swing.JLabel();
        timeoutSpinner = new javax.swing.JSpinner();
        endImageButton = new javax.swing.JButton();
        discardRowsButton = new javax.swing.JButton();
        jLabel3 = new javax.swing.JLabel();
        discardRowsSpinner = new javax.swing.JSpinner();

        setDefaultCloseOperation(javax.swing.WindowConstants.EXIT_ON_CLOSE);

        statusPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Status"));
        statusPanel.setLayout(new javax.swing.BoxLayout(statusPanel, javax.swing.BoxLayout.PAGE_AXIS));

        commandPanel2.setBorder(javax.swing.BorderFactory.createTitledBorder("CCS Commands"));

        setAvailableButton.setText("setAvailable");
        setAvailableButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                setAvailableButtonActionPerformed(evt);
            }
        });

        revokeAvailableButton.setText("revokeAvailable");
        revokeAvailableButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                revokeAvailableButtonActionPerformed(evt);
            }
        });

        simulateFault.setText("simulateFault");
        simulateFault.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                simulateFaultActionPerformed(evt);
            }
        });

        clearFaultButton.setText("clearFault");
        clearFaultButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                clearFaultButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout commandPanel2Layout = new javax.swing.GroupLayout(commandPanel2);
        commandPanel2.setLayout(commandPanel2Layout);
        commandPanel2Layout.setHorizontalGroup(
            commandPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(commandPanel2Layout.createSequentialGroup()
                .addComponent(setAvailableButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(revokeAvailableButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(simulateFault)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(clearFaultButton)
                .addGap(0, 0, Short.MAX_VALUE))
        );
        commandPanel2Layout.setVerticalGroup(
            commandPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(commandPanel2Layout.createSequentialGroup()
                .addGroup(commandPanel2Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(setAvailableButton)
                    .addComponent(revokeAvailableButton)
                    .addComponent(simulateFault)
                    .addComponent(clearFaultButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        commandPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("OCS Lifecycle Commands"));

        enterControlButton.setText("enterControl");
        enterControlButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                enterControlButtonActionPerformed(evt);
            }
        });

        exitButton.setText("exitControl");
        exitButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                exitButtonActionPerformed(evt);
            }
        });

        startButton.setText("start");
        startButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                startButtonActionPerformed(evt);
            }
        });

        standbyButton.setText("standby");
        standbyButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                standbyButtonActionPerformed(evt);
            }
        });

        enableButton.setText("enable");
        enableButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                enableButtonActionPerformed(evt);
            }
        });

        disableButton.setText("disable");
        disableButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                disableButtonActionPerformed(evt);
            }
        });

        startTextField.setColumns(20);
        startTextField.setText("Normal");
        startTextField.setToolTipText("Configuration name");

        javax.swing.GroupLayout commandPanelLayout = new javax.swing.GroupLayout(commandPanel);
        commandPanel.setLayout(commandPanelLayout);
        commandPanelLayout.setHorizontalGroup(
            commandPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(commandPanelLayout.createSequentialGroup()
                .addGroup(commandPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(commandPanelLayout.createSequentialGroup()
                        .addComponent(enterControlButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(exitButton))
                    .addGroup(commandPanelLayout.createSequentialGroup()
                        .addComponent(startButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(startTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(standbyButton))
                    .addGroup(commandPanelLayout.createSequentialGroup()
                        .addComponent(enableButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(disableButton)))
                .addGap(0, 0, Short.MAX_VALUE))
        );
        commandPanelLayout.setVerticalGroup(
            commandPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(commandPanelLayout.createSequentialGroup()
                .addGroup(commandPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(enterControlButton)
                    .addComponent(exitButton))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(commandPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(startButton)
                    .addComponent(startTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(standbyButton))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(commandPanelLayout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(enableButton)
                    .addComponent(disableButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        commandPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("OCS Camera Commands"));

        initImageButton.setText("initImage");
        initImageButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                initImageButtonActionPerformed(evt);
            }
        });

        jLabel4.setLabelFor(deltaTSpinner);
        jLabel4.setText("deltaT");

        deltaTSpinner.setModel(new javax.swing.SpinnerNumberModel(Float.valueOf(1.0f), Float.valueOf(0.0f), Float.valueOf(15.0f), Float.valueOf(0.1f)));

        takeImagesButton.setText("takeImages");
        takeImagesButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                takeImagesButtonActionPerformed(evt);
            }
        });

        jLabel5.setLabelFor(nImagesSpinner);
        jLabel5.setText("nImages");

        nImagesSpinner.setModel(new javax.swing.SpinnerNumberModel(2, 0, 20, 1));

        jLabel6.setLabelFor(exposureSpinner);
        jLabel6.setText("exposure");

        exposureSpinner.setModel(new javax.swing.SpinnerNumberModel(Float.valueOf(15.0f), Float.valueOf(0.0f), Float.valueOf(30.0f), Float.valueOf(1.0f)));

        openShutterCheckBox.setSelected(true);
        openShutterCheckBox.setText("openShutter");
        openShutterCheckBox.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        setFilterButton.setText("setFilter");
        setFilterButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                setFilterButtonActionPerformed(evt);
            }
        });

        filterComboBox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));

        scienceCheckBox.setSelected(true);
        scienceCheckBox.setText("science");
        scienceCheckBox.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        guiderCheckBox.setSelected(true);
        guiderCheckBox.setText("guider");
        guiderCheckBox.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        visitNameTextField.setColumns(20);
        visitNameTextField.setText("visit-00001");
        visitNameTextField.setToolTipText("Visit name");

        initGuidersButton.setText("initGuiders");
        initGuidersButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                initGuidersButtonActionPerformed(evt);
            }
        });

        roiSpecTextField.setColumns(20);
        roiSpecTextField.setToolTipText("ROI Specification");

        roiLabel.setText("roiSpec");

        wavefrontCheckBox.setSelected(true);
        wavefrontCheckBox.setText("wavefront");
        wavefrontCheckBox.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        enableCalibrationButton.setText("enableCalibration");
        enableCalibrationButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                enableCalibrationButtonActionPerformed(evt);
            }
        });

        disableCalibrationButton.setText("disableCalibration");
        disableCalibrationButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                disableCalibrationButtonActionPerformed(evt);
            }
        });

        javax.swing.GroupLayout commandPanel1Layout = new javax.swing.GroupLayout(commandPanel1);
        commandPanel1.setLayout(commandPanel1Layout);
        commandPanel1Layout.setHorizontalGroup(
            commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(commandPanel1Layout.createSequentialGroup()
                .addGroup(commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(commandPanel1Layout.createSequentialGroup()
                        .addComponent(initImageButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel4)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(deltaTSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(commandPanel1Layout.createSequentialGroup()
                        .addComponent(takeImagesButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel5)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(nImagesSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel6)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(exposureSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(openShutterCheckBox)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(scienceCheckBox)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(wavefrontCheckBox)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(guiderCheckBox)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(visitNameTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(commandPanel1Layout.createSequentialGroup()
                        .addComponent(setFilterButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(filterComboBox, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(commandPanel1Layout.createSequentialGroup()
                        .addComponent(initGuidersButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(roiLabel)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(roiSpecTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(commandPanel1Layout.createSequentialGroup()
                        .addComponent(enableCalibrationButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(disableCalibrationButton)))
                .addGap(0, 0, Short.MAX_VALUE))
        );
        commandPanel1Layout.setVerticalGroup(
            commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(commandPanel1Layout.createSequentialGroup()
                .addGroup(commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(initImageButton)
                    .addComponent(jLabel4)
                    .addComponent(deltaTSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(takeImagesButton)
                    .addComponent(jLabel5)
                    .addComponent(nImagesSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel6)
                    .addComponent(exposureSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(openShutterCheckBox)
                    .addComponent(scienceCheckBox)
                    .addComponent(guiderCheckBox)
                    .addComponent(visitNameTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(wavefrontCheckBox))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(setFilterButton)
                    .addComponent(filterComboBox, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(initGuidersButton)
                    .addComponent(roiSpecTextField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(roiLabel))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(commandPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(enableCalibrationButton)
                    .addComponent(disableCalibrationButton))
                .addContainerGap(javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
        );

        logTextArea.setColumns(80);
        logTextArea.setRows(20);
        jScrollPane2.setViewportView(logTextArea);

        jPanel1.setBorder(javax.swing.BorderFactory.createTitledBorder("OCS Camera Calibration Commands"));

        clearButton.setText("clear");
        clearButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                clearButtonActionPerformed(evt);
            }
        });

        nClearSpinner.setModel(new javax.swing.SpinnerNumberModel(1, 1, 1000, 1));

        jLabel2.setText("n");

        startImageButton.setText("startImage");
        startImageButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                startImageButtonActionPerformed(evt);
            }
        });

        openShutterCheckBox1.setSelected(true);
        openShutterCheckBox1.setText("openShutter");
        openShutterCheckBox1.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        scienceCheckBox1.setSelected(true);
        scienceCheckBox1.setText("science");
        scienceCheckBox1.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        wavefrontCheckBox1.setSelected(true);
        wavefrontCheckBox1.setText("wavefront");
        wavefrontCheckBox1.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        guiderCheckBox1.setSelected(true);
        guiderCheckBox1.setText("guider");
        guiderCheckBox1.setHorizontalTextPosition(javax.swing.SwingConstants.LEADING);

        visitNameTextField1.setColumns(20);
        visitNameTextField1.setText("visit-00001");
        visitNameTextField1.setToolTipText("Visit name");

        jLabel7.setText("timeout");

        timeoutSpinner.setModel(new javax.swing.SpinnerNumberModel(Float.valueOf(15.0f), Float.valueOf(0.0f), Float.valueOf(30.0f), Float.valueOf(1.0f)));

        endImageButton.setText("endImage");
        endImageButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                endImageButtonActionPerformed(evt);
            }
        });

        discardRowsButton.setText("discardRows");
        discardRowsButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                discardRowsButtonActionPerformed(evt);
            }
        });

        jLabel3.setText("rows");

        discardRowsSpinner.setModel(new javax.swing.SpinnerNumberModel(1, 1, 1000, 1));

        javax.swing.GroupLayout jPanel1Layout = new javax.swing.GroupLayout(jPanel1);
        jPanel1.setLayout(jPanel1Layout);
        jPanel1Layout.setHorizontalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addComponent(clearButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel2)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(nClearSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addComponent(startImageButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(openShutterCheckBox1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(scienceCheckBox1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(wavefrontCheckBox1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(guiderCheckBox1)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(visitNameTextField1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel7)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(timeoutSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addComponent(endImageButton)
                    .addGroup(jPanel1Layout.createSequentialGroup()
                        .addComponent(discardRowsButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(jLabel3)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(discardRowsSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addGap(0, 0, Short.MAX_VALUE))
        );
        jPanel1Layout.setVerticalGroup(
            jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(jPanel1Layout.createSequentialGroup()
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(clearButton)
                    .addComponent(nClearSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel2))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(startImageButton)
                    .addComponent(openShutterCheckBox1)
                    .addComponent(scienceCheckBox1)
                    .addComponent(wavefrontCheckBox1)
                    .addComponent(guiderCheckBox1)
                    .addComponent(visitNameTextField1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(timeoutSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel7))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(endImageButton)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(jPanel1Layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(discardRowsButton)
                    .addComponent(jLabel3)
                    .addComponent(discardRowsSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(0, 0, Short.MAX_VALUE))
        );

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(statusPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(commandPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(commandPanel2, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(commandPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(jScrollPane2)
                    .addComponent(jPanel1, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(statusPanel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(commandPanel2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(commandPanel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(commandPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(2, 2, 2)
                .addComponent(jPanel1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addComponent(jScrollPane2, javax.swing.GroupLayout.DEFAULT_SIZE, 294, Short.MAX_VALUE)
                .addContainerGap())
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void enterControlButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_enterControlButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new EnterControlCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_enterControlButtonActionPerformed

    private void exitButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_exitButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new ExitControlCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_exitButtonActionPerformed

    private void startButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_startButtonActionPerformed
        String configuration = startTextField.getText();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new StartCommand(configuration));
                return null;
            }
        };
        sw.execute();

    }//GEN-LAST:event_startButtonActionPerformed

    private void standbyButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_standbyButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new StandbyCommand());
                return null;
            }
        };
        sw.execute();    }//GEN-LAST:event_standbyButtonActionPerformed

    private void enableButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_enableButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new EnableCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_enableButtonActionPerformed

    private void disableButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_disableButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new DisableCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_disableButtonActionPerformed

    private void setAvailableButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_setAvailableButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new CCSSetAvailableCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_setAvailableButtonActionPerformed

    private void revokeAvailableButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_revokeAvailableButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new CCSRevokeAvailableCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_revokeAvailableButtonActionPerformed

    private void simulateFaultActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_simulateFaultActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new CCSSimulateFaultCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_simulateFaultActionPerformed

    private void clearFaultButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_clearFaultButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new CCSClearFaultCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_clearFaultButtonActionPerformed

    private void clearButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_clearButtonActionPerformed
        int nClears = ((Number) nClearSpinner.getModel().getValue()).intValue();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new ClearCommand(nClears));
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_clearButtonActionPerformed

    private void startImageButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_startImageButtonActionPerformed
        boolean openShutter = openShutterCheckBox1.isSelected();
        boolean science = scienceCheckBox1.isSelected();
        boolean wavefront = wavefrontCheckBox1.isSelected();
        boolean guider = guiderCheckBox1.isSelected();
        float timeout = ((Number) timeoutSpinner.getModel().getValue()).floatValue();
        String visitName = visitNameTextField1.getText();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new StartImageCommand(visitName, openShutter, science, wavefront, guider, timeout));
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_startImageButtonActionPerformed

    private void endImageButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_endImageButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new EndImageCommand());
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_endImageButtonActionPerformed

    private void discardRowsButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_discardRowsButtonActionPerformed
        int nRows = ((Number) discardRowsSpinner.getModel().getValue()).intValue();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new DiscardRowsCommand(nRows));
                return null;
            }
        };
        sw.execute();    }//GEN-LAST:event_discardRowsButtonActionPerformed

    private void disableCalibrationButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_disableCalibrationButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new DisableCalibrationCommand());
                return null;
            }
        };
    }//GEN-LAST:event_disableCalibrationButtonActionPerformed

    private void enableCalibrationButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_enableCalibrationButtonActionPerformed
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new EnableCalibrationCommand());
                return null;
            }
        };
    }//GEN-LAST:event_enableCalibrationButtonActionPerformed

    private void initGuidersButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_initGuidersButtonActionPerformed
        String roiSpec = roiSpecTextField.getText();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new InitGuidersCommand(roiSpec));
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_initGuidersButtonActionPerformed

    private void setFilterButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_setFilterButtonActionPerformed
        String filter = filterComboBox.getSelectedItem().toString();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new SetFilterCommand(filter));
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_setFilterButtonActionPerformed

    private void takeImagesButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_takeImagesButtonActionPerformed
        int nImages = ((Number) nImagesSpinner.getModel().getValue()).intValue();
        float exposure = ((Number) exposureSpinner.getModel().getValue()).floatValue();
        boolean openShutter = openShutterCheckBox.isSelected();
        boolean science = scienceCheckBox.isSelected();
        boolean wavefront = wavefrontCheckBox.isSelected();
        boolean guider = guiderCheckBox.isSelected();
        String visitName = visitNameTextField.getText();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new TakeImagesCommand(exposure, nImages, openShutter, science, wavefront, guider, visitName));
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_takeImagesButtonActionPerformed

    private void initImageButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_initImageButtonActionPerformed
        float deltaT = ((Number) deltaTSpinner.getModel().getValue()).floatValue();
        SwingWorker sw = new SwingWorker() {

            @Override
            protected Object doInBackground() throws Exception {
                guiInterface.execute(new InitImageCommand(deltaT));
                return null;
            }
        };
        sw.execute();
    }//GEN-LAST:event_initImageButtonActionPerformed

    private class TextAreaHandler extends StreamHandler {

        @Override
        public void publish(LogRecord record) {
            super.publish(record);
            flush();
            logTextArea.append(getFormatter().format(record));
        }
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton clearButton;
    private javax.swing.JButton clearFaultButton;
    private javax.swing.JPanel commandPanel1;
    private javax.swing.JPanel commandPanel2;
    private javax.swing.JSpinner deltaTSpinner;
    private javax.swing.JButton disableButton;
    private javax.swing.JButton disableCalibrationButton;
    private javax.swing.JButton discardRowsButton;
    private javax.swing.JSpinner discardRowsSpinner;
    private javax.swing.JButton enableButton;
    private javax.swing.JButton enableCalibrationButton;
    private javax.swing.JButton endImageButton;
    private javax.swing.JButton enterControlButton;
    private javax.swing.JButton exitButton;
    private javax.swing.JSpinner exposureSpinner;
    private javax.swing.JComboBox filterComboBox;
    private javax.swing.JCheckBox guiderCheckBox;
    private javax.swing.JCheckBox guiderCheckBox1;
    private javax.swing.JButton initGuidersButton;
    private javax.swing.JButton initImageButton;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JScrollPane jScrollPane2;
    private javax.swing.JTextArea logTextArea;
    private javax.swing.JSpinner nClearSpinner;
    private javax.swing.JSpinner nImagesSpinner;
    private javax.swing.JCheckBox openShutterCheckBox;
    private javax.swing.JCheckBox openShutterCheckBox1;
    private javax.swing.JButton revokeAvailableButton;
    private javax.swing.JLabel roiLabel;
    private javax.swing.JTextField roiSpecTextField;
    private javax.swing.JCheckBox scienceCheckBox;
    private javax.swing.JCheckBox scienceCheckBox1;
    private javax.swing.JButton setAvailableButton;
    private javax.swing.JButton setFilterButton;
    private javax.swing.JButton simulateFault;
    private javax.swing.JButton standbyButton;
    private javax.swing.JButton startButton;
    private javax.swing.JButton startImageButton;
    private javax.swing.JTextField startTextField;
    private javax.swing.JPanel statusPanel;
    private javax.swing.JButton takeImagesButton;
    private javax.swing.JSpinner timeoutSpinner;
    private javax.swing.JTextField visitNameTextField;
    private javax.swing.JTextField visitNameTextField1;
    private javax.swing.JCheckBox wavefrontCheckBox;
    private javax.swing.JCheckBox wavefrontCheckBox1;
    // End of variables declaration//GEN-END:variables
}
