package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.sim.Filter;
import org.lsst.ccs.subsystem.ocsbridge.util.Event;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.AggregateStatus;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.sal.SAL;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.sal.SALException;
import org.lsst.sal.camera.CameraStateChangeEvent;
import org.lsst.sal.camera.event.AvailableFiltersEvent;


/**
 * A standalone version of the OCSGui, which communicates with the camera using
 * SAL
 *
 * @author tonyj
 */
public class StandaloneOCSGUI {

    private final static Logger logger = Logger.getLogger(StandaloneOCSGUI.class.getName());

    public static void main(String[] args) {
        
        OCSBridgeConfig.Device device = OCSBridgeConfig.Device.CAMERA;
        if (args.length > 0) {
            device = OCSBridgeConfig.Device.valueOf(args[0]);
        }
        
        SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr = OCSBridgeConfig.createSALManager(device);
        CCS ccs = new CCS();
        final GUISALLayer guisalLayer = new GUISALLayer(mgr, ccs);
        ToyOCSGUI gui = new ToyOCSGUI(guisalLayer, device);
        gui.setVisible(true);
        guisalLayer.start();
    }

    private static class GUISALLayer implements GUILayer {

        private final SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr;
        private final CCS ccs;

        public GUISALLayer(SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr, CCS ccs) {
            this.mgr = mgr;
            this.ccs = ccs;
        }

        public void start() {

            Thread eventThread = new Thread("SALEventReceiver") {

                @Override
                public void run() {
                    try {
                        runEventLoop();
                    } catch (Exception x) {
                        logger.log(Level.WARNING, "Failed to initialize SAL communication layer for events, "
                                + "check that SAL has been setup correctly.", x);
                    }
                }
            };
            eventThread.start();

//            Thread telemetryThread = new Thread("SALTelemetryReceiver") {
//
//                @Override
//                public void run() {
//                    try {
//                        runTelemetryLoop();
//                    } catch (Exception x) {
//                        logger.log(Level.WARNING, "Failed to initialize SAL communication layer for telemetry, "
//                                + "check that SAL has been setup correctly.", x);
//                    }
//                }
//            };
//            telemetryThread.start();
        }

        
        
        @Override
        public void execute(CameraCommand cmd) {
            try {
                mgr.issueCommand(cmd);
            } catch (SALException ex) {
                logger.log(Level.SEVERE, "Exception executing SAL command: {0}", ex.getMessage());
            }
        }

        private void runEventLoop() {
            try {
                for (;;) {
                    CameraEvent event = mgr.getNextEvent(Duration.ofMinutes(1));
                    if (event == null) {
                        logger.info("Still waiting for an event");
                    } else {
                        logger.log(Level.INFO, "Received {0}", event);
                        if (event instanceof CameraStateChangeEvent) {
                            Enum value = ((CameraStateChangeEvent) event).getSubstate().getEnum();
                            ccs.getAggregateStatus().add(new State(value));
                        } else if (event instanceof AvailableFiltersEvent) {
                            List<String> availableFilters = Arrays.asList(((AvailableFiltersEvent) event).getFilterNames().split(":"));
                            // TODO: This does not seem to be the right event to fire here, should be an OCS event
                            ccs.fireEvent(new Filter.CCSAvailableFiltersEvent(availableFilters));
                        }
                    }
                }
            } catch (SALException ex) {
                logger.log(Level.SEVERE, "Unexpected exception while waiting for events", ex);
            }
        }

        private void runTelemetryLoop() {
            try {
                for (;;) {
                    CameraTelemetry telemetry = mgr.getTelemetry(Duration.ofMinutes(1));
                    if (telemetry != null) {
                        logger.log(Level.INFO, "Received {0}", telemetry);
                    }
                }
            } catch (SALException ex) {
                logger.log(Level.SEVERE, "Unexpected exception while waiting for events", ex);
            }
        }
        
        
        @Override
        public AggregateStatus getAggregateStatus() {
            return ccs.getAggregateStatus();
        }

        @Override
        public void addStateChangeListener(State.StateChangeListener<? extends Enum> listener) {
            ccs.addStateChangeListener(listener);
        }

        @Override
        public void removeStateChangeListener(State.StateChangeListener<? extends Enum> listener) {
            ccs.removeStateChangeListener(listener);
        }

        @Override
        public void execute(CCSCommand command) {
            // NOOP - not supported in standalone mode
        }

        @Override
        public boolean supportsCCSCommands() {
            return false;
        }

        @Override
        public void addEventListener(Event.EventListener listener) {
            ccs.addEventListener(listener);
        }

        @Override
        public void removeEventListener(Event.EventListener listener) {
            ccs.removeEventListener(listener);
        }
    }
}
