package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import java.time.Duration;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;

/**
 * Top level "module" for running the OCSBridge as a CCS Subsytem
 *
 * @author tonyj
 */
public class OCSSubsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subsystem;
    private OCSBridge ocsBridge;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private OCSBridgeConfig config;

    OCSSubsystem() {
    }

    @Override
    public void postStart() {
        CCS ccs = new CCS();
        MCMLayer mcmLayer = new MCMCCSLayer(subsystem, ccs, config);
        ocsBridge = new OCSBridge(config, ccs, mcmLayer);
        ocsBridge.setCCSCommandExecutor(new CCSCommandExecutor() {
            @Override
            void sendAck(CCSCommand.CCSAckOrNack can) {
                // Note: We add 100mS to the estimate to keep ShellCommandConsole from timing
                // out, especially on zero length commands.
                subsystem.sendAck(can.getDuration().plus(Duration.ofMillis(100)));
            }

            @Override
            void sendNack(CCSCommand.CCSAckOrNack can) {
                subsystem.sendNack(can.getReason());
            }
        });

        OCSBridgeSALLayer ocsInterface = new OCSBridgeSALLayer(ocsBridge);
        ocsBridge.setOCSCommandExecutor(ocsInterface);
        ocsInterface.start();
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void setAvailable() throws Exception {
        CCSCommand.CCSSetAvailableCommand setAvailable = new CCSCommand.CCSSetAvailableCommand();
        ocsBridge.execute(setAvailable);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void revokeAvailable() throws Exception {
        CCSCommand.CCSRevokeAvailableCommand revokeAvailable = new CCSCommand.CCSRevokeAvailableCommand();
        ocsBridge.execute(revokeAvailable);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void simulateFault() throws Exception {
        CCSCommand.CCSSimulateFaultCommand simulateFault = new CCSCommand.CCSSimulateFaultCommand();
        ocsBridge.execute(simulateFault);
    }

    @Command(type = Command.CommandType.ACTION, autoAck = false, level = Command.NORMAL)
    public void clearFault() throws Exception {
        CCSCommand.CCSClearFaultCommand clearFault = new CCSCommand.CCSClearFaultCommand();
        ocsBridge.execute(clearFault);
    }
}
