package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter;
import org.lsst.ccs.subsystem.ocsbridge.sim.Rafts;
import java.time.Duration;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.FilterState;
import org.lsst.sal.SAL;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.CameraEvent;
import org.lsst.sal.camera.CameraTelemetry;
import org.lsst.sal.SALException;
import org.lsst.sal.SALReceivedCommand;
import org.lsst.sal.camera.event.AvailableFiltersEvent;
import org.lsst.sal.camera.states.FilterChangerDetailedStateEvent;
import org.lsst.sal.camera.states.CCSCommandStateEvent;
import org.lsst.sal.camera.states.CalibrationDetailedStateEvent;
import org.lsst.sal.camera.states.ImageReadinessDetailedStateEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.CCSCommandStateEvent.IdleBusyState;
import org.lsst.sal.camera.states.RaftsDetailedStateEvent;
import org.lsst.sal.camera.states.ShutterDetailedStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent;

/**
 *
 * @author tonyj
 */
public class OCSBridgeSALLayer extends OCSCommandExecutor {

    private final OCSBridge bridge;
    private final SAL<CameraCommand, CameraEvent, CameraTelemetry> mgr;
    private boolean shutdown = false;
    private static final Logger logger = Logger.getLogger(OCSBridgeSALLayer.class.getName());
    private Thread runThread;

    OCSBridgeSALLayer(OCSBridge bridge) {
        super(bridge);
        this.bridge = bridge;
        mgr = bridge.getConfig().getSALManager();

        bridge.getCCS().addStateChangeListener((currentState, oldState) -> {
            int priority = 1;
            try {
                // DANGER: This code assumes the order of items in the CCS enums is the same as the order in the Simplel-SAL Enums
                // This is very likely to be wrong.
                if (currentState instanceof SummaryStateEvent.LSE209State) {
                    mgr.logEvent(new SummaryStateEvent(priority, (SummaryStateEvent.LSE209State) currentState));
                } else if (currentState instanceof Shutter.ShutterState) {
                    mgr.logEvent(new ShutterDetailedStateEvent(priority, ShutterDetailedStateEvent.ShutterState.values()[currentState.ordinal()]));
                } else if (currentState instanceof IdleBusyState) {
                    mgr.logEvent(new CCSCommandStateEvent(priority, (IdleBusyState) currentState));
                } else if (currentState instanceof FilterState) {
                    mgr.logEvent(new FilterChangerDetailedStateEvent(priority, FilterChangerDetailedStateEvent.FilterChangerState.values()[currentState.ordinal()]));
                } else if (currentState instanceof MCM.TakeImageReadinessState) {
                    mgr.logEvent(new ImageReadinessDetailedStateEvent(priority, ImageReadinessDetailedStateEvent.ImageReadinessState.values()[currentState.ordinal()]));
                } else if (currentState instanceof Rafts.RaftsState) {
                    mgr.logEvent(new RaftsDetailedStateEvent(priority, RaftsDetailedStateEvent.RaftsState.values()[currentState.ordinal()]));
                } else if (currentState instanceof MCM.CalibrationState) {
                    mgr.logEvent(new CalibrationDetailedStateEvent(priority, CalibrationDetailedStateEvent.CalibrationState.values()[currentState.ordinal()]));
                } else if (currentState instanceof OfflineState) {
                    mgr.logEvent(new OfflineDetailedStateEvent(priority, (OfflineState) currentState));
                }
            } catch (SALException x) {
                logger.log(Level.SEVERE, "Unable to log message", x);
            }
        });

        bridge.getCCS().addEventListener((event) -> {
            int priority = 1;
            try {
                if (event instanceof CCSAvailableFiltersEvent) {
                    mgr.logEvent(new AvailableFiltersEvent(priority, String.join(":", ((CCSAvailableFiltersEvent) event).getAvailableFilters())));
                }
            } catch (SALException x) {
                logger.log(Level.SEVERE, "Unable to log message", x);
            }
        });

    }

    public void start() {

        Thread t = new Thread("SALCommandReceiver") {

            @Override
            public void run() {
                try {
                    runCommandLoop();
                } catch (Exception x) {
                    logger.log(Level.WARNING, "Failed to initialize SAL communication layer, "
                            + "check that SAL has been setup correctly. Reverting to standalone mode.", x);
                }
            }
        };
        t.start();
    }

    private void runCommandLoop() {
        try {
            runThread = Thread.currentThread();

            while (!shutdown) {
                currentCommand = mgr.getNextCommand(Duration.ofMinutes(1));
                if (currentCommand == null) {
                    logger.info("Still waiting for a command");
                } else {
                    bridge.execute(currentCommand.getCommand());
                }
            }
            //mgr.salShutdown();
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while waiting for commands", ex);
        } finally {
            runThread = null;
        }
    }
    private SALReceivedCommand<CameraCommand> currentCommand;

    private void shutdown() throws InterruptedException {
        shutdown = true;
        runThread.join();
    }

    @Override
    protected void reportComplete(OCSCommandExecutor.OCSExecutor command) {
        super.reportComplete(command);
        try {
            currentCommand.reportComplete();
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", ex);
        }
    }

    @Override
    protected void reportError(OCSCommandExecutor.OCSExecutor command, Exception ex) {
        super.reportError(command, ex);
        try {
            currentCommand.reportError(ex);
        } catch (SALException x) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", x);
        }
    }

    @Override
    protected void acknowledgeCommand(OCSCommandExecutor.OCSExecutor command, Duration timeout) {
        super.acknowledgeCommand(command, timeout);
        try {
            currentCommand.acknowledgeCommand(timeout);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", ex);
        }
    }

    @Override
    protected void rejectCommand(OCSCommandExecutor.OCSExecutor command, String reason) {
        super.rejectCommand(command, reason);
        try {
            currentCommand.rejectCommand(reason);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error while responding to command", ex);
        }
    }

    @Override
    void sendEvent(CameraEvent event) {
        super.sendEvent(event);
        try {
            mgr.logEvent(event);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error sending event to SAL", ex);
        }
    }

    @Override
    void sendTelemetry(CameraTelemetry telemetry) {
        try {
            mgr.sendTelemetry(telemetry);
        } catch (SALException ex) {
            logger.log(Level.SEVERE, "Unexpected error sending telemetry to SAL", ex);
        }
    }

}
