package org.lsst.ccs.subsystem.ocsbridge;

import org.lsst.ccs.subsystem.ocsbridge.sim.MCMDirectLayer;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM;
import org.lsst.ccs.subsystem.ocsbridge.util.Event;
import org.lsst.ccs.subsystem.ocsbridge.util.CCS;
import org.lsst.ccs.subsystem.ocsbridge.util.State;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSAckOrNack;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSClearFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSCommandResponse;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDisableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSDiscardRowsCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEnableCalibrationCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSEndImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitGuidersCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSInitImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSPreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSRevokeAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetAvailableCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSetFilterCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSSimulateFaultCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSStartImageCommand;
import org.lsst.ccs.subsystem.ocsbridge.CCSCommand.CCSTakeImagesCommand;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.CCSAvailableFiltersEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.Filter.FilterState;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.OCSExecutor;
import org.lsst.ccs.subsystem.ocsbridge.OCSCommandExecutor.PreconditionsNotMet;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSFilterTelemetry;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSImageNameEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSetFilterEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCM.CCSSettingsAppliedEvent;
import org.lsst.ccs.subsystem.ocsbridge.sim.MCMConfig;
import org.lsst.ccs.subsystem.ocsbridge.sim.Rafts.RaftsState;
import org.lsst.ccs.subsystem.ocsbridge.sim.Shutter.ShutterState;
import org.lsst.sal.camera.CameraCommand;
import org.lsst.sal.camera.command.ClearCommand;
import org.lsst.sal.camera.command.DisableCalibrationCommand;
import org.lsst.sal.camera.command.DisableCommand;
import org.lsst.sal.camera.command.DiscardRowsCommand;
import org.lsst.sal.camera.command.EnableCalibrationCommand;
import org.lsst.sal.camera.command.EnableCommand;
import org.lsst.sal.camera.command.EndImageCommand;
import org.lsst.sal.camera.command.EnterControlCommand;
import org.lsst.sal.camera.command.ExitControlCommand;
import org.lsst.sal.camera.command.InitGuidersCommand;
import org.lsst.sal.camera.command.InitImageCommand;
import org.lsst.sal.camera.command.SetFilterCommand;
import org.lsst.sal.camera.command.StandbyCommand;
import org.lsst.sal.camera.command.StartCommand;
import org.lsst.sal.camera.command.StartImageCommand;
import org.lsst.sal.camera.command.TakeImagesCommand;
import org.lsst.sal.camera.event.AppliedSettingsMatchStartEvent;
import org.lsst.sal.camera.event.AvailableFiltersEvent;
import org.lsst.sal.camera.event.EndLoadFilterEvent;
import org.lsst.sal.camera.event.EndOfImageTelemetryEvent;
import org.lsst.sal.camera.event.EndReadoutEvent;
import org.lsst.sal.camera.event.EndRotateCarouselEvent;
import org.lsst.sal.camera.event.EndSetFilterEvent;
import org.lsst.sal.camera.event.EndShutterCloseEvent;
import org.lsst.sal.camera.event.EndShutterOpenEvent;
import org.lsst.sal.camera.event.EndUnloadFilterEvent;
import org.lsst.sal.camera.event.SettingVersionsEvent;
import org.lsst.sal.camera.event.SettingsAppliedEvent;
import org.lsst.sal.camera.event.StartLoadFilterEvent;
import org.lsst.sal.camera.event.StartRaftIntegrationEvent;
import org.lsst.sal.camera.event.StartReadoutEvent;
import org.lsst.sal.camera.event.StartRotateCarouselEvent;
import org.lsst.sal.camera.event.StartSetFilterEvent;
import org.lsst.sal.camera.event.StartShutterCloseEvent;
import org.lsst.sal.camera.event.StartShutterOpenEvent;
import org.lsst.sal.camera.event.StartUnloadFilterEvent;
import org.lsst.sal.camera.states.OfflineDetailedStateEvent.OfflineState;
import org.lsst.sal.camera.states.CCSCommandStateEvent.IdleBusyState;
import org.lsst.sal.camera.states.SummaryStateEvent;
import org.lsst.sal.camera.states.SummaryStateEvent.LSE209State;
import org.lsst.sal.camera.telemetry.CameraFilterTelemetry;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import org.lsst.sal.atcamera.event.ShutterMotionProfileEvent;
import org.lsst.sal.camera.event.ImageReadoutParametersEvent;

/**
 *
 * The Command Layer of the OCS Bridge. Contains executors for all commands
 * supported by the OCSBridge.
 *
 * @author The LSST CCS Team
 */
public class OCSBridge {

    private static final Logger LOG = Logger.getLogger(OCSBridge.class.getName());
    private final CCS ccs;
    private final State lse209State;
    private final State offlineState;
    private OCSCommandExecutor ocsCommandExecutor;
    private CCSCommandExecutor ccsCommandExecutor;
    private final MCMLayer mcm;
    private static final int DEFAULT_EVENT_PRIORITY = 1;
    private BlockingQueue<CCSImageNameEvent> imageNameTransfer = new ArrayBlockingQueue<>(1);
    private CCSImageNameEvent imageNameEvent;
    private CCSTimeStamp ccsTimeStamp;
    private final OCSBridgeConfig config;

    private final State commandState;

    OCSBridge(OCSBridgeConfig config, CCS ccs, MCMLayer mcm) {

        this.config = config;
        this.ccs = ccs;
        this.mcm = mcm;
        lse209State = new State(LSE209State.OFFLINE);
        ccs.getAggregateStatus().add(lse209State);
        offlineState = new State(OfflineState.PUBLISH_ONLY);
        ccs.getAggregateStatus().add(offlineState);
        commandState = new State(IdleBusyState.IDLE);
        ccs.getAggregateStatus().add(commandState);
        ocsCommandExecutor = new OCSCommandExecutor(this);
        ccsCommandExecutor = new CCSCommandExecutor();
        double jitter = 0.2; // change exposure time by a fixed amount
        
        mcm.addStateChangeListener((Enum state, Enum oldState) -> {
            // There has to be a better way to do this but this is what I can implement for now - FA
            // Rafts related events
            if ((oldState == RaftsState.CLEARING || oldState == RaftsState.QUIESCENT) && state == RaftsState.INTEGRATING) {
                ccs.runInBackground(() -> {
                    try {
                        imageNameEvent = imageNameTransfer.poll(1, TimeUnit.SECONDS);
                        if (imageNameEvent == null) {
                            throw new RuntimeException("Image name could not be retreived");
                        }
                        ocsCommandExecutor.sendEvent(new StartRaftIntegrationEvent(DEFAULT_EVENT_PRIORITY,
                                imageNameEvent.getImageSequenceName(), imageNameEvent.getImagesInSequence(), imageNameEvent.getImageName(), imageNameEvent.getSequenceNumber(), 1000.0 * imageNameEvent.getIntegrationStartTime(), imageNameEvent.getExposureTime()));                     
                     ocsCommandExecutor.sendEvent(new ImageReadoutParametersEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageName(), config.ccdNames(), config.ccdType(),
                     config.overRows(), config.overCols(), config.readRows(), config.readCols(), config.readCols2(), config.preCols(), config.preRows(), config.postCols()));
                    }
                    
                    catch (RuntimeException | InterruptedException ex) {
                        LOG.log(Level.SEVERE, "Error sending StartRaftIntegrationEvent", ex);
                    }
                });
            }

            if (oldState == RaftsState.INTEGRATING && state == RaftsState.READING_OUT) {
                ocsCommandExecutor.sendEvent(new StartReadoutEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageSequenceName(), imageNameEvent.getImagesInSequence(), imageNameEvent.getImageName(), imageNameEvent.getSequenceNumber(), 1000.0 * imageNameEvent.getIntegrationStartTime(), imageNameEvent.getExposureTime()));
            }

            if (oldState == RaftsState.READING_OUT) {
                ocsCommandExecutor.sendEvent(new EndReadoutEvent(DEFAULT_EVENT_PRIORITY, imageNameEvent.getImageSequenceName(), imageNameEvent.getImagesInSequence(), imageNameEvent.getImageName(), imageNameEvent.getSequenceNumber(), 1000.0 * imageNameEvent.getIntegrationStartTime(), imageNameEvent.getExposureTime()));
                // For now we will always send the EndOfImageTelemetry event 300ms after the EndRoutoutEvent
                ccs.schedule(Duration.ofMillis(300), new Runnable() {
                    // imageNameEvet may change before 300ms is up.
                    private CCSImageNameEvent event = imageNameEvent;

                    @Override
                    public void run() {
                        ocsCommandExecutor.sendEvent(new EndOfImageTelemetryEvent(DEFAULT_EVENT_PRIORITY, event.getImageSequenceName(), imageNameEvent.getImagesInSequence(), event.getImageName(), event.getSequenceNumber(), 1000.0 * event.getIntegrationStartTime(), event.getExposureTime()));
                    }
                });
            }

            // Shutter
            if (oldState == ShutterState.CLOSED && state == ShutterState.OPENING) {
                ocsCommandExecutor.sendEvent(new StartShutterOpenEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == ShutterState.OPENING && state == ShutterState.OPEN) {
                ocsCommandExecutor.sendEvent(new EndShutterOpenEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == ShutterState.OPEN && state == ShutterState.CLOSING) {
                ocsCommandExecutor.sendEvent(new StartShutterCloseEvent(DEFAULT_EVENT_PRIORITY));
            }

            
            if (oldState == ShutterState.CLOSING && state == ShutterState.CLOSED) {
                ocsCommandExecutor.sendEvent(new EndShutterCloseEvent(DEFAULT_EVENT_PRIORITY));
                ocsCommandExecutor.sendEvent(new ShutterMotionProfileEvent(1, imageNameEvent.getExposureTime()+ jitter));
            }

            // Filter change events
            if (oldState == FilterState.UNLOADED && state == FilterState.ROTATING) {
                ocsCommandExecutor.sendEvent(new StartRotateCarouselEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == FilterState.ROTATING && state == FilterState.UNLOADED) {
                ocsCommandExecutor.sendEvent(new EndRotateCarouselEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == FilterState.UNLOADED && state == FilterState.LOADING) {
                ocsCommandExecutor.sendEvent(new StartLoadFilterEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == FilterState.LOADING && state == FilterState.LOADED) {
                ocsCommandExecutor.sendEvent(new EndLoadFilterEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == FilterState.LOADED && state == FilterState.UNLOADING) {
                ocsCommandExecutor.sendEvent(new StartUnloadFilterEvent(DEFAULT_EVENT_PRIORITY));
            }

            if (oldState == FilterState.UNLOADING && state == FilterState.UNLOADED) {
                ocsCommandExecutor.sendEvent(new EndUnloadFilterEvent(DEFAULT_EVENT_PRIORITY));
            }
        });

        mcm.addEventListener((Event event) -> {
            if (event instanceof CCSAvailableFiltersEvent) {
                List<String> filters = ((CCSAvailableFiltersEvent) event).getAvailableFilters();
                ocsCommandExecutor.sendEvent(new AvailableFiltersEvent(DEFAULT_EVENT_PRIORITY, String.join(":", filters)));
            } else if (event instanceof CCSImageNameEvent) {
                imageNameTransfer.add((CCSImageNameEvent) event);
            } else if (event instanceof CCSSetFilterEvent) {
                CCSSetFilterEvent setFilter = (CCSSetFilterEvent) event;
                if (setFilter.isStart()) {
                    ocsCommandExecutor.sendEvent(new StartSetFilterEvent(DEFAULT_EVENT_PRIORITY, setFilter.getFilterName()));
                } else {
                    ocsCommandExecutor.sendEvent(new EndSetFilterEvent(DEFAULT_EVENT_PRIORITY, setFilter.getFilterName()));
                }
            } else if (event instanceof CCSFilterTelemetry) {
                ocsCommandExecutor.sendTelemetry(new CameraFilterTelemetry(((CCSFilterTelemetry) event).getFilterId()));
            }
            else if (event instanceof CCSSettingsAppliedEvent){ocsCommandExecutor.sendEvent(new SettingsAppliedEvent(DEFAULT_EVENT_PRIORITY, ((CCSSettingsAppliedEvent) event).getSettings(), ((CCSSettingsAppliedEvent) event).getTimeStamp()));
            }
        });
    }

    State<IdleBusyState> getCommandState() {
        return commandState;
    }

    public OCSBridgeConfig getConfig() {
        return config;
    }

    /**
     * Allow a user to provide an alternative implementation of the
     * OCSCommandExecutor. Used to override the default OCSCommandExecutor with
     * one that actually sends acknowledgments back to OCS.
     *
     * @param ocs
     */
    void setOCSCommandExecutor(OCSCommandExecutor ocs) {
        this.ocsCommandExecutor = ocs;
    }

    void setCCSCommandExecutor(CCSCommandExecutor ccs) {
        this.ccsCommandExecutor = ccs;
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        OCSBridge ocs = createOCSBridge();
        ToyOCSGUI gui = new ToyOCSGUI(new GUIDirectLayer(ocs),ocs.getConfig().getDevice());
        gui.setVisible(true);

    }

    /**
     * Create an OCSBridge with default configuration
     * @return 
     */
    static OCSBridge createOCSBridge() {
        OCSBridgeConfig ocsConfig = OCSBridgeConfig.createDefaultConfig();
        MCMConfig mcmConfig = MCMConfig.createDefaultConfig();
        return createOCSBridge(ocsConfig, mcmConfig);
    }
    
    static OCSBridge createOCSBridge(OCSBridgeConfig config, MCMConfig mcmConfig) {
        CCS ccs = new CCS();
        MCMLayer mcm = new MCMDirectLayer(new MCM(ccs, mcmConfig));
        return new OCSBridge(config, ccs, mcm);
    }

    void execute(CameraCommand cmd) {
        if (cmd instanceof SetFilterCommand) {
            execute((SetFilterCommand) cmd);
        } else if (cmd instanceof TakeImagesCommand) {
            execute((TakeImagesCommand) cmd);
        } else if (cmd instanceof InitImageCommand) {
            execute((InitImageCommand) cmd);
        } else if (cmd instanceof EnableCommand) {
            execute((EnableCommand) cmd);
        } else if (cmd instanceof DisableCommand) {
            execute((DisableCommand) cmd);
        } else if (cmd instanceof ExitControlCommand) {
            execute((ExitControlCommand) cmd);
        } else if (cmd instanceof EnterControlCommand) {
            execute((EnterControlCommand) cmd);
        } else if (cmd instanceof StartCommand) {
            execute((StartCommand) cmd);
        } else if (cmd instanceof StandbyCommand) {
            execute((StandbyCommand) cmd);
        } else if (cmd instanceof DiscardRowsCommand) {
            execute((DiscardRowsCommand) cmd);
        } else if (cmd instanceof ClearCommand) {
            execute((ClearCommand) cmd);
        } else if (cmd instanceof EndImageCommand) {
            execute((EndImageCommand) cmd);
        } else if (cmd instanceof StartImageCommand) {
            execute((StartImageCommand) cmd);
        } else if (cmd instanceof EnableCalibrationCommand) {
            execute((EnableCalibrationCommand) cmd);
        } else if (cmd instanceof DisableCalibrationCommand) {
            execute((DisableCalibrationCommand) cmd);
        } else {
            throw new RuntimeException("Unrecognized command " + cmd);
        }
    }

    void execute(InitImageCommand command) {
        OCSExecutor initImage = new InitImageExecutor(command);
        ocsCommandExecutor.executeCommand(initImage);
    }

    void execute(SetFilterCommand command) {
        OCSExecutor setFilter = new SetFilterExecutor(command);
        ocsCommandExecutor.executeCommand(setFilter);
    }

    void execute(TakeImagesCommand command) {
        OCSExecutor takeImages = new TakeImagesExecutor(command);
        ocsCommandExecutor.executeCommand(takeImages);
    }

    void execute(InitGuidersCommand command) {
        OCSExecutor initGuiders = new InitGuidersExecutor(command);
        ocsCommandExecutor.executeCommand(initGuiders);
    }

    void execute(DiscardRowsCommand command) {
        OCSExecutor discardRows = new DiscardRowsExecutor(command);
        ocsCommandExecutor.executeCommand(discardRows);
    }

    void execute(ClearCommand command) {
        OCSExecutor clear = new ClearExecutor(command);
        ocsCommandExecutor.executeCommand(clear);
    }

    void execute(StartImageCommand command) {
        OCSExecutor startImage = new StartImageExecutor(command);
        ocsCommandExecutor.executeCommand(startImage);
    }

    void execute(EndImageCommand command) {
        OCSExecutor endImage = new EndImageExecutor(command);
        ocsCommandExecutor.executeCommand(endImage);
    }

    void execute(EnterControlCommand command) {
        OCSExecutor takeControl = new EnterControlExecutor(command);
        ocsCommandExecutor.executeCommand(takeControl);
    }

    void execute(ExitControlCommand command) {
        OCSExecutor exit = new ExitExecutor(command);
        ocsCommandExecutor.executeCommand(exit);
    }

    void execute(StartCommand command) {
        OCSExecutor start = new StartExecutor(command);
        ocsCommandExecutor.executeCommand(start);
    }

    void execute(StandbyCommand command) {
        OCSExecutor standby = new StandbyExecutor(command);
        ocsCommandExecutor.executeCommand(standby);
    }

    void execute(EnableCommand command) {
        OCSExecutor enable = new EnableExecutor(command);
        ocsCommandExecutor.executeCommand(enable);
    }

    void execute(DisableCommand command) {
        OCSExecutor disable = new DisableExecutor(command);
        ocsCommandExecutor.executeCommand(disable);
    }

    void execute(EnableCalibrationCommand command) {
        OCSExecutor enableCalibration = new EnableCalibrationExecutor(command);
        ocsCommandExecutor.executeCommand(enableCalibration);
    }

    void execute(DisableCalibrationCommand command) {
        OCSExecutor disable = new DisableCalibrationExecutor(command);
        ocsCommandExecutor.executeCommand(disable);
    }

    void execute(CCSCommand command) {
        if (command instanceof CCSSetAvailableCommand) {
            execute((CCSSetAvailableCommand) command);
        } else if (command instanceof CCSRevokeAvailableCommand) {
            execute((CCSRevokeAvailableCommand) command);
        } else if (command instanceof CCSSimulateFaultCommand) {
            execute((CCSSimulateFaultCommand) command);
        } else if (command instanceof CCSClearFaultCommand) {
            execute((CCSClearFaultCommand) command);
        } else {
            throw new RuntimeException("Unknown command type: " + command);
        }
    }

    void execute(CCSSetAvailableCommand command) {
        CCSExecutor setAvailable = new SetAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(setAvailable));
    }

    void execute(CCSRevokeAvailableCommand command) {
        CCSExecutor revokeAvailable = new RevokeAvailableExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(revokeAvailable));
    }

    void execute(CCSSimulateFaultCommand command) {
        CCSExecutor simulateFault = new SimulateFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(simulateFault));
    }

    void execute(CCSClearFaultCommand command) {
        CCSExecutor clearFault = new ClearFaultExecutor(command);
        ccsCommandExecutor.executeCommand(new CCSCommandResponse(clearFault));
    }

    // Utility only used for testing   
    public Future<Void> waitForState(Enum state) {
        return ccs.waitForStatus(state);
    }

    CCS getCCS() {
        return ccs;
    }

    /**
     * Super class for all OCSExecutors which forward commands to MCM
     */
    abstract class ForwardToMCMExecutor extends OCSCommandExecutor.OCSExecutor {

        private final CCSCommand ccsCommand;
        private CCSCommandResponse response;
        private final LSE209State initialState;

        ForwardToMCMExecutor(CameraCommand command, CCSCommand ccsCommand) {
            this(command, ccsCommand, SummaryStateEvent.LSE209State.ENABLED);
        }

        ForwardToMCMExecutor(CameraCommand command, CCSCommand ccsCommand, LSE209State initialState) {
            super(command);
            this.ccsCommand = ccsCommand;
            this.initialState = initialState;
        }

        @Override
        Duration testPreconditions() throws OCSCommandExecutor.PreconditionsNotMet {
            if (!lse209State.isInState(initialState)) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command not accepted in: " + lse209State);
            }
            response = mcm.execute(ccsCommand);
            CCSAckOrNack can = response.waitForAckOrNack();
            if (can.isNack()) {
                throw new OCSCommandExecutor.PreconditionsNotMet("Command rejected: " + can.getReason());
            } else {
                return can.getDuration();
            }
        }

        @Override
        void execute() throws Exception {
            response.waitForCompletion();
        }
    }

    class InitImageExecutor extends ForwardToMCMExecutor {

        public InitImageExecutor(InitImageCommand command) {
            super(command, new CCSInitImageCommand(command.getDeltaT()));
        }
    }

    class TakeImagesExecutor extends ForwardToMCMExecutor {

        public TakeImagesExecutor(TakeImagesCommand command) {
            super(command, new CCSTakeImagesCommand(command.getExpTime(), command.getNumImages(), command.isShutter(), command.isScience(), command.isWfs(), command.isGuide(), command.getImageSequenceName()));
        }

        @Override
        void execute() throws Exception {
            imageNameTransfer.clear();
            super.execute();
        }
    }

    class EnableCalibrationExecutor extends ForwardToMCMExecutor {

        public EnableCalibrationExecutor(EnableCalibrationCommand command) {
            super(command, new CCSEnableCalibrationCommand());
        }
    }

    class DisableCalibrationExecutor extends ForwardToMCMExecutor {

        public DisableCalibrationExecutor(DisableCalibrationCommand command) {
            super(command, new CCSDisableCalibrationCommand());
        }
    }

    class SetFilterExecutor extends ForwardToMCMExecutor {

        public SetFilterExecutor(SetFilterCommand command) {
            super(command, new CCSSetFilterCommand(command.getName()));
        }
    }

    class InitGuidersExecutor extends ForwardToMCMExecutor {

        public InitGuidersExecutor(InitGuidersCommand command) {
            super(command, new CCSInitGuidersCommand(command.getRoiSpec()));
        }
    }

    class ClearExecutor extends ForwardToMCMExecutor {

        public ClearExecutor(ClearCommand command) {
            super(command, new CCSClearCommand(command.getNClears()));
        }
    }

    class StartImageExecutor extends ForwardToMCMExecutor {

        public StartImageExecutor(StartImageCommand command) {
            super(command, new CCSStartImageCommand(command.getImageSequenceName(), command.isShutter(), command.isScience(), command.isWfs(), command.isGuide(), command.getTimeout()));
        }

        @Override
        void execute() throws Exception {
            imageNameTransfer.clear();
            super.execute();
        }
    }

    class EndImageExecutor extends ForwardToMCMExecutor {

        public EndImageExecutor(EndImageCommand command) {
            super(command, new CCSEndImageCommand());
        }
    }

    class DiscardRowsExecutor extends ForwardToMCMExecutor {

        public DiscardRowsExecutor(DiscardRowsCommand command) {
            super(command, new CCSDiscardRowsCommand(command.getNRows()));
        }
    }

    class EnterControlExecutor extends OCSExecutor {

        public EnterControlExecutor(EnterControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.OFFLINE)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.AVAILABLE)) {
                throw new PreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: The argument should be a comma-delimited list of supported settings. Clearly should not be hard-wired here.
            ocsCommandExecutor.sendEvent(new SettingVersionsEvent(DEFAULT_EVENT_PRIORITY, "Normal"));
            lse209State.setState(LSE209State.STANDBY);
        }
    }

    class ExitExecutor extends OCSExecutor {

        public ExitExecutor(ExitControlCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.STANDBY)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(LSE209State.OFFLINE);
            offlineState.setState(OfflineState.PUBLISH_ONLY);
        }
    }

    /**
     * Start command is a special case, since it is a lifecycle command, but
     * must also issue a start command to the MCM (to set configuration).
     */
    class StartExecutor extends ForwardToMCMExecutor {

        public StartExecutor(StartCommand command) {
            super(command, new CCSStartCommand(command.getConfiguration()), LSE209State.STANDBY);
        }

        @Override
        void execute() throws Exception {
            super.execute();
            ocsCommandExecutor.sendEvent(new AppliedSettingsMatchStartEvent(DEFAULT_EVENT_PRIORITY, true));
            

            lse209State.setState(LSE209State.DISABLED);
        }
    }

    class StandbyExecutor extends OCSExecutor {

        public StandbyExecutor(StandbyCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standy command if things are happening?
            //TODO: or wait until things finish and return then?
            //TODO: The argument should be a comma-delimited list of supported settings. Clearly should not be hard-wired here.
            ocsCommandExecutor.sendEvent(new SettingVersionsEvent(DEFAULT_EVENT_PRIORITY, "Normal"));
            lse209State.setState(LSE209State.STANDBY);
        }
    }

    class EnableExecutor extends OCSExecutor {

        public EnableExecutor(EnableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.DISABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            lse209State.setState(LSE209State.ENABLED);
        }
    }

    class DisableExecutor extends OCSExecutor {

        public DisableExecutor(DisableCommand command) {
            super(command);
        }

        @Override
        Duration testPreconditions() throws PreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.ENABLED)) {
                throw new PreconditionsNotMet("Command not accepted in " + lse209State);
            }
            // Fixme: Can we reject the disable command if we are busy?
            // What about if we are not idle?
            // Note logic here is incorrect according to Paul Lotz, we must always accept
            // the disable command.
//            if (startImageTimeout != null && !startImageTimeout.isDone()) {
//                throw new PreconditionsNotMet("Exposure in progress");
//            }
            return Duration.ZERO;
        }

        @Override
        void execute() throws Exception {
            //TODO: should we reject the standy command if things are happening?
            //TODO: or wait until things finish and return then?
            lse209State.setState(LSE209State.DISABLED);
        }
    }

    class SetAvailableExecutor extends CCSExecutor {

        private SetAvailableExecutor(CCSSetAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.PUBLISH_ONLY)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.AVAILABLE);
        }

    }

    class RevokeAvailableExecutor extends CCSExecutor {

        private RevokeAvailableExecutor(CCSRevokeAvailableCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.OFFLINE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            if (!offlineState.isInState(OfflineState.AVAILABLE)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + offlineState);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            offlineState.setState(OfflineState.PUBLISH_ONLY);
        }

    }

    class SimulateFaultExecutor extends CCSExecutor {

        private SimulateFaultExecutor(CCSSimulateFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            //TODO: Should we also attempt to stop the subsystems?
            lse209State.setState(LSE209State.FAULT);
        }

    }

    class ClearFaultExecutor extends CCSExecutor {

        private ClearFaultExecutor(CCSClearFaultCommand command) {
        }

        @Override
        protected Duration testPreconditions() throws CCSPreconditionsNotMet {
            if (!lse209State.isInState(LSE209State.FAULT)) {
                throw new CCSPreconditionsNotMet("Command not accepted in " + lse209State);
            }
            return Duration.ZERO;
        }

        @Override
        protected void execute() throws Exception {
            lse209State.setState(LSE209State.OFFLINE);
            offlineState.setState(OfflineState.PUBLISH_ONLY);
        }

    }
}
